/*
 * SonarQube, open source software quality management tool.
 * Copyright (C) 2008-2014 SonarSource
 * mailto:contact AT sonarsource DOT com
 *
 * SonarQube is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * SonarQube is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.api.config;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Maps;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.sonar.api.utils.DateUtils;

import javax.annotation.Nullable;

import java.io.IOException;
import java.io.StringReader;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * SonarSource license. This class aims to extract metadata but not to validate or - of course -
 * to generate license
 *
 * @since 3.0
 */
public final class License {
  private String product;
  private String organization;
  private String expirationDate;
  private String type;
  private String server;
  private Map<String, String> additionalProperties;

  private License(Map<String, String> properties) {
    this.additionalProperties = Maps.newHashMap(properties);
    product = StringUtils.defaultString(get("Product", properties), get("Plugin", properties));
    organization = StringUtils.defaultString(get("Organisation", properties), get("Name", properties));
    expirationDate = StringUtils.defaultString(get("Expiration", properties), get("Expires", properties));
    type = get("Type", properties);
    server = get("Server", properties);
    // SONAR-4340 Don't expose Digest and Obeo properties
    additionalProperties.remove("Digest");
    additionalProperties.remove("Obeo");
  }

  private String get(String key, Map<String, String> properties) {
    additionalProperties.remove(key);
    return properties.get(key);
  }

  /**
   * Get additional properties available on this license (like threshold conditions)
   * @since 3.6
   */
  public Map<String, String> additionalProperties() {
    return additionalProperties;
  }

  @Nullable
  public String getProduct() {
    return product;
  }

  @Nullable
  public String getOrganization() {
    return organization;
  }

  @Nullable
  public String getExpirationDateAsString() {
    return expirationDate;
  }

  @Nullable
  public Date getExpirationDate() {
    return DateUtils.parseDateQuietly(expirationDate);
  }

  public boolean isExpired() {
    return isExpired(new Date());
  }

  @VisibleForTesting
  boolean isExpired(Date now) {
    Date date = getExpirationDate();
    return date != null && !date.after(org.apache.commons.lang.time.DateUtils.truncate(now, Calendar.DATE));
  }

  @Nullable
  public String getType() {
    return type;
  }

  @Nullable
  public String getServer() {
    return server;
  }

  public static License readBase64(String base64) {
    return readPlainText(new String(Base64.decodeBase64(base64.getBytes())));
  }

  @VisibleForTesting
  static License readPlainText(String data) {
    Map<String, String> props = Maps.newHashMap();
    StringReader reader = new StringReader(data);
    try {
      List<String> lines = IOUtils.readLines(reader);
      for (String line : lines) {
        if (StringUtils.isNotBlank(line) && line.indexOf(':') > 0) {
          String key = StringUtils.substringBefore(line, ":");
          String value = StringUtils.substringAfter(line, ":");
          props.put(StringUtils.trimToEmpty(key), StringUtils.trimToEmpty(value));
        }
      }

    } catch (IOException e) {
      // silently ignore

    } finally {
      IOUtils.closeQuietly(reader);
    }
    return new License(props);
  }
}
