/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2008-2011 SonarSource
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.batch.events;

import com.google.common.collect.Lists;
import org.sonar.api.utils.Logs;

import java.util.List;

/**
 * Dispatches {@link SonarEvent}s. Eases decoupling by allowing objects to interact without having direct dependencies upon one another, and
 * without requiring event sources to deal with maintaining handler lists.
 * 
 * @since 2.7
 */
public class EventBus {

  private EventHandler[] registeredHandlers;

  public EventBus(EventHandler[] handlers) {
    this.registeredHandlers = handlers;
  }

  /**
   * Fires the given event.
   */
  public void fireEvent(SonarEvent event) {
    doFireEvent(event);
  }

  private void doFireEvent(SonarEvent event) {
    List<EventHandler> handlers = getDispatchList(event.getType());
    Logs.INFO.trace("Dispatch event {} for {}", event, handlers);
    for (EventHandler handler : handlers) {
      event.dispatch(handler);
    }
  }

  private List<EventHandler> getDispatchList(Class<? extends EventHandler> handlerType) {
    List<EventHandler> result = Lists.newArrayList();
    for (EventHandler handler : registeredHandlers) {
      if (handlerType.isAssignableFrom(handler.getClass())) {
        result.add(handler);
      }
    }
    return result;
  }

}
