package caseine.project;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.json.JsonObject;

import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

import caseine.CleanFolder;
import caseine.exceptions.TestDirectoryMissingException;
import caseine.exceptions.UnitTestsFileMissingException;
import vplwsclient.FileUtils;
import vplwsclient.RestJsonMoodleClient;
import vplwsclient.RestJsonMoodleClient.VPLService;
import vplwsclient.VplFile;
import vplwsclient.exception.MoodleWebServiceException;
import vplwsclient.exception.VplConnectionException;
import vplwsclient.exception.VplException;

public class CaseineCommonProject implements CaseineProject {
	
	private final static String CASEINE_COMMENT = "For settings, see https://moodle.caseine.org/mod/wiki/view.php?pageid=199\n"+
			"Writable settings are:\n"+
			" - WS.name, WS.shortdescription, WS.intro, WS.introformat, WS.startdate\n"+
			" - WS.duedate, WS.maxfiles, WS.maxfilesize, WS.requirednet, WS.password\n"+
			" - WS.grade, WS.visiblegrade, WS.usevariations, WS.variationtitle\n"+
			" - WS.usableasbase, WS.basedon, WS.run, WS.debug, WS.evaluate\n"+
			" - WS.evaluateonsubmission, WS.automaticgrading, WS.usegradefrom\n"+
			" - WS.maxexetime, WS.restrictededitor, WS.allowsubmissionviaeditor\n"+
			" - WS.forbiddeneditors, WS.example, WS.maxexememory, WS.maxexefilesize\n"+
			" - WS.maxexeprocesses, WS.jailservers, WS.worktype, WS.emailteachers\n"+
			" - WS.timemodified, WS.freeevaluations, WS.reductionbyevaluation\n"+
			" - WS.sebrequired, WS.sebkeys, WS.runscript, WS.debugscript";
	
	protected static final String VPLID_0 = "0";

	public static String CASEINE_VPL_ID = "CASEINE_VPL_ID";
	public static String CASEINE_VPL_TYPE = "CASEINE_VPL_TYPE";
	
	protected String projectPath;

	protected String vplId;
	
	protected File caseineFile;

	protected static String INTERNAL_FILE = ".caseine";
	
	protected static Logger log = Logger.getLogger(CaseineJavaProject.class.getName());
	
	private String url;

	private String token;
	
	protected static final String RF = "rf";
	protected static final String EF = "ef";
	protected static final String CF = "cf";

	/**
	 * By convention sources of a CaseInE project must be here.
	 */
	protected static final String PATH_SRC = File.separator + "src";

	/**
	 * By convention sources of a CaseInE project must be here.
	 */
	protected static final String PATH_CLASSES = File.separator + "target" + File.separator + "classes";

	/**
	 * By convention sources of a CaseInE project must be here.
	 */
	protected static final String PATH_DOC = File.separator + "doc";
	
	/**
	 * By convention binaries of a CaseInE project must be here.
	 */
	protected static String PATH_BIN = File.separator + "target" + File.separator;

	/**
	 * By convention test sources of a CaseInE project must be here.
	 */
	protected static final String PATH_TEST = File.separator + "test";
	/**
	 * By convention resources of a CaseInE project must be here.
	 */
	protected static final String PATH_RESOURCES_SRC = File.separator + "resources" + File.separator + "src";
	/**
	 * By convention test resources of a CaseInE project must be here.
	 */
	protected static final String PATH_RESOURCES_TEST = File.separator + "resources" + File.separator + "test";
	/**
	 * By convention lib resources of a CaseInE project must be here.
	 */
	protected static final String PATH_RESOURCES_LIB = File.separator + "resources" + File.separator + "lib";

	protected static final String PLUGIN_LIB = "caseine.vpl.tools.plugin.jar";

	protected boolean settings = false;



	public CaseineCommonProject(String projectPath, String vplId, String url, String token, boolean settings) {
		this.projectPath = projectPath;
		caseineFile = new File(projectPath + File.separator + INTERNAL_FILE);
		this.vplId = vplId;
		this.url = url;
		this.token = token;
		this.settings = settings;
	}

	/**
	 * 
	 * @return True if it is a Caseine project, Maven projects are not managed
	 */
	public boolean isCaseine() {
		boolean result = false;
		
		if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
			result = true;
		}
		if (caseineFile.exists()) {
			result = true;
		}
		return result;
	}

	public ProjectType getType() {
		ProjectType result =  ProjectType.JAVA; 
		File pom = new File(projectPath + File.separator + "pom.xml");
		if (pom.exists()) {
			result = ProjectType.MAVEN;
		}
		Properties props = getProperties();
		if (props.containsKey(CASEINE_VPL_TYPE)) {
			String type = (String) props.get(CASEINE_VPL_TYPE);
			result = ProjectType.valueOf(type);
		}
		return result;
	}

	/**
	 * 
	 * @return all properties in the .caseine file
	 */
	public Properties getProperties() {
		Properties props = new Properties();
		if (caseineFile.exists()) {
			FileInputStream fis = null;
			try {
				fis = new FileInputStream(caseineFile);
				props.load(fis);
			} catch (Exception e) {
				log.severe(e.getMessage());
			} finally {
				if (fis != null) {
					try {
						fis.close();
					} catch (IOException e) {
						log.severe(e.getMessage());
					}					
				}
			}
		}
		return props;
	}
	
	public String getVplId() {
		String result = null;
		if (caseineFile.exists()) {
			Properties props = getProperties();
			if (props.containsKey(CASEINE_VPL_ID)) {
				result = (String) props.get(CASEINE_VPL_ID);
			}
		} else {
			if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
				// Get it in the pom.xml
				File pom = new File(projectPath + File.separator + "pom.xml");
				MavenXpp3Reader reader = new MavenXpp3Reader();
				try {
					Model model = reader.read(new FileReader(pom));
					Properties properties = model.getProperties();
					result = (String) properties.get(CASEINE_VPL_ID);
				} catch (IOException | XmlPullParserException e) {
					// Nothing found
				}
			}
		}
		return result;
	}

	public void fillCaseine(String vplId) {
		if (caseineFile.exists()) {
			Properties props = new Properties();
			FileInputStream fis = null;
			FileOutputStream fos = null;
			try {
				fis = new FileInputStream(caseineFile);
				props.load(fis);
				if (!vplId.equals(VPLID_0)) {
					props.setProperty(CASEINE_VPL_ID, vplId);
				}
				props.setProperty(CASEINE_VPL_TYPE, getType().toString());
				fos = new FileOutputStream(caseineFile);
				props.store(fos, CASEINE_COMMENT);
			} catch (Exception e) {
				log.severe(e.getMessage());
			} finally {
				if (fis != null) {
					try {
						fis.close();
					} catch (IOException e) {
						log.severe(e.getMessage());
					}
				}
				if (fos != null) {
					try {
						fos.close();
					} catch (IOException e) {
						log.severe(e.getMessage());
					}
				}
			}
		}
	}

	@Override
	public void generate(boolean mvn, int template) throws CaseineProjectAlreadyExistingException, BadIDEException,
			IOException, TestDirectoryMissingException, FileMissingException, UnitTestsFileMissingException {}

	@Override
	public void local(ClassLoader cl) throws IOException, ClassNotFoundException, MavenProjectException {}

	/**
	 * Publishes the templates to the remote caseine server.
	 * The full description of the lab is synchronized.
	 * 
	 * @param cl an optional classloader to provide dependencies
	 * @throws IOException            if something wrong
	 * @throws ClassNotFoundException if something wrong
	 * @throws VplException           if something wrong
	 * @throws VPLIDMissingException
	 * @throws MavenProjectException 
	 * @throws NothingPushedException 
	 */
	@Override
	public void push(ClassLoader cl) throws IOException, ClassNotFoundException, VplException, VPLIDMissingException, MavenProjectException, NothingPushedException {
		
		boolean somethingPushed = false;
		
		// Run local to be sure
		local(cl);
		
		vplId = getVplId();
		
		if (vplId != null) {

			if (settings) {
				// Synchronize settings
				somethingPushed =  somethingPushed | updateSettings();
			}
		
			String[] ids = vplId.split(",");
			
			for (String id: ids) {
	
				RestJsonMoodleClient client = new RestJsonMoodleClient(id, token, url);
				client.setServerSupportsIsBinary(true);
	
				somethingPushed = saveFiles(client, projectPath + getPATH_REQUESTED_FILES(), VPLService.VPL_SAVE_RF, true, true);
	
				// Execution Files
				somethingPushed =  somethingPushed | saveFiles(client, projectPath + getPATH_EXECUTION_FILES(), VPLService.VPL_SAVE_EF, true, false);
	
				// Corrected Files
				somethingPushed =  somethingPushed | saveFiles(client, projectPath + getPATH_CORRECTED_FILES(), VPLService.VPL_SAVE_CF, true, true);		
			}
			
			// Synchronize the documentation
			somethingPushed =  somethingPushed | documentation();
		}
		if (!somethingPushed) {
			throw new NothingPushedException("Nothing pushed on the server : a VPL_ID is needed");
		}
	}

	@Override
	public void clean() throws IOException {
		new File(projectPath + getPATH_BIN()).mkdir();
		if (new File(projectPath + getPATH_OUTPUT()).exists()) {
			Path folder = Paths.get(projectPath + getPATH_OUTPUT());
			CleanFolder.clean(folder);
		}
	}


	@Override
	public boolean isGenerated() {
		return new File(projectPath + getPATH_OUTPUT()).exists();
	}

	@Override
	public void nature() throws CaseineProjectAlreadyExistingException, IOException {
		if (isCaseine()) {
			if (getType() == CaseineJavaProject.ProjectType.MAVEN) {
				throw new CaseineProjectAlreadyExistingException(
						"Nature command is not implemented for Maven projects");
			}
			throw new CaseineProjectAlreadyExistingException("A project already exists in " + projectPath);
		}
		if (!caseineFile.createNewFile()) {
			log.severe(".caseine cannot be created");
		}
		// Adds vplId in the project
		fillCaseine(vplId);
	}


	@Override
	public boolean documentation() throws VplConnectionException, MoodleWebServiceException, IOException {
		
		boolean somethingPushed = false;
		
		vplId = getVplId();

		if (vplId != null) {
			
			String[] ids = vplId.split(",");
		
			for (String id: ids) {
	
				RestJsonMoodleClient client = new RestJsonMoodleClient(id, token, url);
				client.setServerSupportsIsBinary(true);
		
				// Get Format

				JsonObject introObject = client.callService("mod_vpl_get_setting", "settingname", "introformat");
				String format = introObject.getString("value");
				
				String filePath = projectPath + PATH_DOC;
				String fileName = filePath + File.separator + "intro";
				if (format.equals("1")) {
					fileName += ".html";
				}
		
				File doc = new File(fileName); 
				File docDir = new File(filePath); 
				if (doc.exists()) { // Push it to the server
					String introduction = Files.readString(Path.of(fileName));
					client.callService("mod_vpl_set_setting", "settingname", "intro", "settingvalue", introduction);
					somethingPushed = true;
				} else { // Get it locally
					docDir.mkdir();
					JsonObject intro = client.callService("mod_vpl_info");
					String introHtml = intro.getString("intro","A problem occured while recovering the VPL description.");
					try (BufferedWriter writer = new BufferedWriter(new FileWriter(fileName))) {
						writer.write(introHtml);
					}	
				}
			}
		}
		return somethingPushed;
	}

	public boolean updateSettings() throws VplConnectionException, MoodleWebServiceException, IOException {
		
		vplId = getVplId();
		
		Properties properties = getProperties();

		if (vplId != null) {
			
			String[] ids = vplId.split(",");
		
			for (String id: ids) {
	
				RestJsonMoodleClient client = new RestJsonMoodleClient(id, token, url);
				client.setServerSupportsIsBinary(true);
				properties.keySet().stream().filter(key -> key.toString().startsWith("WS.")).forEach(name -> 
					{
						try {
							String settingName = name.toString().substring(3);
							String settingValue = properties.get(name).toString();
							log.info(String.format("Trying to update setting %s with value %s", settingName, settingValue));
							client.callService("mod_vpl_set_setting", "settingname", settingName, "settingvalue", settingValue);
						} catch (VplConnectionException | MoodleWebServiceException e) {
							log.severe(e.getMessage());
						}
					});
			}
		}
		return !properties.keySet().isEmpty();
	}

	/**
	 * By convention requested files generated for a CaseInE project must be here.
	 */
	protected static String getPATH_REQUESTED_FILES() {
		return getPATH_OUTPUT() + File.separator + RF + File.separator;
	}

	/**
	 * By convention corrected files generated for a CaseInE project must be here.
	 */
	protected static String getPATH_CORRECTED_FILES() {
		return getPATH_OUTPUT() + File.separator + CF + File.separator;
	}

	/**
	 * By convention execution files generated for a CaseInE project must be here.
	 */
	protected static String getPATH_EXECUTION_FILES() {
		return getPATH_OUTPUT() + File.separator + EF + File.separator;
	}

	/**
	 * By convention outputs of a CaseInE project must be here.
	 */
	protected static String getPATH_OUTPUT() {
		return getPATH_BIN() + "caseine-output";
	}

	/**
	 * By convention binaries of a CaseInE project must be generated here.
	 */
	protected static String getPATH_BIN() {
		return PATH_BIN;
	}

	/**
	 * 
	 * @param client          the Caseine client
	 * @param projectLocation the local root directory
	 * @param service         the action key for the Caseine WS
	 * @param override        is True if we want local files to override remote
	 *                        ones.
	 * @param checkMax        is True if we must check the number of files to push
	 * @throws IOException
	 * @throws VplException
	 * @return true if files are saved on the server
	 */
	private static boolean saveFiles(RestJsonMoodleClient client, String projectLocation, VPLService service,
			boolean override, boolean checkMax) throws IOException, VplException {
		boolean somethingPushed = false;
		List<File> files = FileUtils.listFiles(projectLocation);
		if (files == null) {
			System.out.println("Directory " + projectLocation + " not found");
		} else if (files.isEmpty()) {
			System.out.println("No files found in " + projectLocation);
		} else {
			Stream<VplFile> vplFiles = files.stream().map(file -> {
				return fileToCaseinePath(file, projectLocation);
			});
			somethingPushed = !files.isEmpty();
			JsonObject result = client.callServiceWithFiles(service.toString(), vplFiles.collect(Collectors.toList()));
			System.out.println(result == null ? "Ok" : result);
		}
		return somethingPushed;
	}
	
	protected void fillCaseineProjectFile() {
		try {
			caseine.FileUtils.fillTemplate(projectPath + File.separatorChar + ".project", "/*VPL_NAME*/", getProjectName());
		} catch (IOException e) {
			log.severe(e.getMessage());
		}
	}

	protected static VplFile fileToCaseinePath(File f, String projectLocation) {
		projectLocation = new File(projectLocation).getAbsolutePath();
		if (!projectLocation.endsWith(File.separator)) {
			projectLocation = projectLocation + File.separator;
		}
		String fProjectRelativeName = f.getAbsolutePath().replace(projectLocation, "").replace(File.separator, "/");
		if (fProjectRelativeName.startsWith(PATH_SRC + "/"))
			fProjectRelativeName = fProjectRelativeName.substring(PATH_SRC.length() + 1);
		try {
			return new VplFile(f, fProjectRelativeName);
		} catch (IOException e) {
			return null;
		}
	}

	public String getProjectName() {
		String projectName = null;
		File projectFile = new File(projectPath);
		if (projectPath.endsWith(".")) {
			projectName = projectFile.getParentFile().getName();
		} else {
			projectName = projectFile.getName();
		}
		return projectName;
	}

}
