/*
 * Copyright 2019: Christophe Saint-Marcel
 * This software is part of the Caseine project.
 * This software was developped with the support of the following organizations:
 * Université Grenoble Alpes
 * Institut Polytechnique de Grenoble
 * 
 * 
 * This file is part of the VPL Design Tool.
 * The VPL Design Tool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * The VPL Design Tool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the VPL Design Tool.  If not, see <https://www.gnu.org/licenses/>.
 */
package caseine;

import java.io.IOException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;

import caseine.exceptions.TestDirectoryMissingException;
import caseine.exceptions.UnitTestsFileMissingException;
import caseine.project.BadIDEException;
import caseine.project.CaseineCppProject;
import caseine.project.CaseineJavaProject;
import caseine.project.CaseineProject;
import caseine.project.CaseineProjectAlreadyExistingException;
import caseine.project.CaseineProjectLoader;
import caseine.project.CaseinePythonProject;
import caseine.project.FileMissingException;
import caseine.project.MavenProjectException;
import caseine.project.NotACaseineProjectException;
import caseine.project.NothingPushedException;
import caseine.project.VPLIDMissingException;
import vplwsclient.exception.MoodleWebServiceException;
import vplwsclient.exception.VplConnectionException;
import vplwsclient.exception.VplException;

/**
 * The push command for the vpl design tool. It can be used to generate locally the caseine structure
 * and remotely the VPL files( Requested Files, Execution Files, Corrected Files).
 */
public class Caseine {

	private static String url = "https://moodle.caseine.org/webservice/rest/server.php";
	private static String token = "";

	private static Options options;

	static {
		options = new Options();

		options.addOption("t", "token", true, "Your token.");
		options.addOption("h", "help", false, "Prints usage and exits.");
		options.addOption("d", "doc", false, "Synchronizes the documentation.");
		options.addOption("v", "vplId", true, "specify the vplid. If unspecified, only generate gaseine files.");
	}

	/**
	 *
	 * @param args command line argument
	 * @throws Exception if something wrong
	 */
	public static void main(String[] args) throws Exception {

		CommandLineParser parser = new DefaultParser();
		CommandLine cmd = parser.parse(options, args);

		// Check for args
		if (cmd.hasOption("h")) {
			printUsage();
			System.exit(0);
		}

		if (!cmd.hasOption("t")) {
			System.out.println("The token is mandatory");
			System.exit(0);
		} else {
			token = cmd.getOptionValue("token");
		}

		String projectPath = ".";
		String vplId = "0";

		String[] remainingArgs = cmd.getArgs();
		if (remainingArgs != null && remainingArgs.length >= 1) {
			projectPath = remainingArgs[0];
		}

		if (cmd.hasOption("vplId")) {
			vplId = cmd.getOptionValue("vplId");
		}

		if (!vplId.equals("0")) {
			if (cmd.hasOption("d")) {
				documentation(projectPath, vplId, url, token, cmd.hasOption("settings"));
				
			} else {
				clean(projectPath);
				local(projectPath, vplId, null);
				push(projectPath, vplId, url, token, null, cmd.hasOption("settings"));
			}
		}
	}

	private static void printUsage() {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp("java -cp /path/to/vplplugin.jar caseine.Push [OPTIONS] [projectFile]", "With OPTIONS in:",
				options, "projectFile is the folder containing the original maven project. Defaults to \".\"", false);
	}
	
	/**
	 * Generates the project template.
	 * 
	 * @param projectPath
	 * @param vplId
	 * @param lang
	 * @param ide
	 * @param mvn 
	 * @throws CaseineProjectAlreadyExistingException
	 * @throws IOException
	 * @throws BadIDEException 
	 * @throws UnitTestsFileMissingException 
	 * @throws FileMissingException 
	 * @throws TestDirectoryMissingException 
	 */
    public static void generate(String projectPath, String vplId, String lang, String ide, boolean mvn, int template, boolean settings) throws CaseineProjectAlreadyExistingException, IOException, BadIDEException, TestDirectoryMissingException, FileMissingException, UnitTestsFileMissingException {
    	
    	CaseineProject proj = null;
    	switch(lang) {
    		case "python":
    			proj = new CaseinePythonProject(projectPath, vplId, url, token, settings);
    			break;
    		case "cpp":
    			proj = new CaseineCppProject(projectPath, vplId, url, token, settings);
    			break;
    		default: // Java
    			proj = new CaseineJavaProject(projectPath, vplId, ide, url, token, settings);
    	}
    	proj.generate(mvn, template);
    }

	/**
	 * Generates the local templates.
	 * 
	 * @param projectPath the project location
	 * @param vplId
	 * @param cl an optional classloader to provide dependencies
	 * @throws IOException if ioException
	 * @throws ClassNotFoundException if the class is not found
	 * @throws NotACaseineProjectException 
	 * @throws MavenProjectException 
	 */
	public static void local(String projectPath, String vplId, ClassLoader cl) throws IOException, ClassNotFoundException, NotACaseineProjectException, MavenProjectException {

		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, vplId, null, null, null, false);
	    
    	proj.local(cl);        	
 	}
	
	/**
	 * Publishes the templates to the remote caseine server.
	 * 
	 * @param projectPath the local path
	 * @param vplId the id of the VPL
	 * @param url optionnal url 
	 * @param token your token to be authenticated
	 * @param cl an optional classloader to provide dependencies
	 * @throws IOException if something wrong
	 * @throws ClassNotFoundException if something wrong
	 * @throws VplException if something wrong
	 * @throws VPLIDMissingException 
	 * @throws MavenProjectException 
	 * @throws NotACaseineProjectException 
	 * @throws NothingPushedException 
	 */
	public static void push(String projectPath, String vplId, String url, String token, ClassLoader cl, boolean settings)
			throws IOException, ClassNotFoundException, VplException, VPLIDMissingException, MavenProjectException, NotACaseineProjectException, NothingPushedException {
		
		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, vplId, null, url, token, settings);

		proj.push(cl);
	}

	/**
	 * Cleans the local templates.
	 * 
	 * @param projectPath the project location
	 * @throws IOException if an error occurs with the deleted files
	 * @throws NotACaseineProjectException 
	 */
	public static void clean(String projectPath) throws IOException, NotACaseineProjectException {
		
		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, null, null, null, null, false);
		
		proj.clean();
	}

	public static void nature(String projectPath, String vplId) throws CaseineProjectAlreadyExistingException, IOException, NotACaseineProjectException {
		
		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, vplId, null, null, null, false);
		
		proj.nature();		
	}

	public static void documentation(String projectPath, String vplId, String url, String token, boolean settings) throws VplConnectionException, MoodleWebServiceException, IOException, NotACaseineProjectException {

		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, vplId, null, url, token, settings);
		
		proj.documentation();		
	}
	public static void udpateSettings(String projectPath, String vplId, String url, String token, boolean settings) throws VplConnectionException, MoodleWebServiceException, IOException, NotACaseineProjectException {

		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, vplId, null, url, token, settings);

		proj.updateSettings();
	}
	
	/*
	 * @param projectPath the project location
	 * @return True if the Caseine project is generated locally
	 */
	public static boolean isGenerated(String projectPath) throws NotACaseineProjectException {
		
		CaseineProject proj = CaseineProjectLoader.getProject(projectPath, null, null, null, null, true);
		
		return proj.isGenerated();
	}

}
