/*
 * Decompiled with CFR 0.152.
 */
package org.carlspring.cloud.storage.s3fs.cache;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import java.nio.file.NoSuchFileException;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.PosixFileAttributes;
import java.util.ArrayList;
import java.util.Optional;
import org.carlspring.cloud.storage.s3fs.S3ObjectId;
import org.carlspring.cloud.storage.s3fs.S3Path;
import org.carlspring.cloud.storage.s3fs.attribute.S3BasicFileAttributes;
import org.carlspring.cloud.storage.s3fs.attribute.S3PosixFileAttributes;
import org.carlspring.cloud.storage.s3fs.cache.S3FileAttributesCachePolicy;
import org.carlspring.cloud.storage.s3fs.util.S3Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3FileAttributesCache {
    private static final Logger logger = LoggerFactory.getLogger(S3FileAttributesCache.class);
    private final S3Utils s3Utils = new S3Utils();
    private volatile Cache<String, Optional<S3BasicFileAttributes>> cache;

    public S3FileAttributesCache(int cacheTTL, int cacheSize) {
        this.cache = this.cacheBuilder(cacheTTL, cacheSize).build();
    }

    public static String generateCacheKey(S3Path path, Class<? extends BasicFileAttributes> attributeClass) {
        S3ObjectId s3ObjectId = path.toS3ObjectId();
        return S3FileAttributesCache.generateCacheKey(s3ObjectId, attributeClass);
    }

    public static String generateCacheKey(S3ObjectId s3ObjectId, Class<? extends BasicFileAttributes> attributeClass) {
        StringBuilder key = new StringBuilder();
        key.append(s3ObjectId.getBucket().replaceAll("/", "%2F")).append("_").append(s3ObjectId.getKey().replaceAll("/", "%2F")).append("_");
        if (attributeClass == BasicFileAttributes.class) {
            key.append(S3BasicFileAttributes.class.getSimpleName());
        } else if (attributeClass == PosixFileAttributes.class) {
            key.append(S3PosixFileAttributes.class.getSimpleName());
        } else {
            key.append(attributeClass.getSimpleName());
        }
        return key.toString();
    }

    public S3BasicFileAttributes get(S3Path path, Class<? extends BasicFileAttributes> attrType) {
        String key = S3FileAttributesCache.generateCacheKey(path, attrType);
        logger.trace("Get cache for key {}", (Object)key);
        Optional<S3BasicFileAttributes> attrs = (Optional<S3BasicFileAttributes>)this.cache.getIfPresent((Object)key);
        if (attrs == null) {
            logger.trace("No cache found for key {}", (Object)key);
            logger.trace("Fetch data for key {}", (Object)key);
            attrs = Optional.ofNullable(this.fetchAttribute(path, key));
            this.put(path, attrs);
        }
        return attrs.orElse(null);
    }

    public boolean contains(S3Path path, Class<? extends BasicFileAttributes> attrType) {
        String key = S3FileAttributesCache.generateCacheKey(path, attrType);
        return this.contains(key);
    }

    public boolean contains(String key) {
        return this.cache.asMap().containsKey(key);
    }

    public void put(S3Path path, S3BasicFileAttributes attrs) {
        this.put(path, Optional.ofNullable(attrs));
    }

    public void put(S3Path path, Optional<S3BasicFileAttributes> attrs) {
        String basicKey = S3FileAttributesCache.generateCacheKey(path, BasicFileAttributes.class);
        logger.trace("Save response for key {}", (Object)basicKey);
        this.cache.put((Object)basicKey, attrs);
        if (attrs.isPresent() && attrs.get() instanceof PosixFileAttributes) {
            String posixKey = S3FileAttributesCache.generateCacheKey(path, PosixFileAttributes.class);
            logger.trace("Save response for key {}", (Object)posixKey);
            this.cache.put((Object)posixKey, attrs);
        }
    }

    public void invalidate(S3Path path, Class<? extends BasicFileAttributes> attrType) {
        String key = S3FileAttributesCache.generateCacheKey(path, attrType);
        logger.trace("Invalidate cache key {}", (Object)key);
        this.cache.invalidate((Object)key);
    }

    public void invalidate(String key) {
        this.cache.invalidate((Object)key);
    }

    public void invalidate(S3Path path) {
        this.invalidate(path.toS3ObjectId());
    }

    public void invalidate(S3ObjectId objectId) {
        S3ObjectId fileObjectId;
        ArrayList<String> keys = new ArrayList<String>();
        keys.add(S3FileAttributesCache.generateCacheKey(objectId, BasicFileAttributes.class));
        keys.add(S3FileAttributesCache.generateCacheKey(objectId, PosixFileAttributes.class));
        if (objectId.getKey().endsWith("/")) {
            fileObjectId = S3ObjectId.builder().bucket(objectId.getBucket()).key(objectId.getKey().substring(0, objectId.getKey().length() - 1)).build();
            keys.add(S3FileAttributesCache.generateCacheKey(fileObjectId, BasicFileAttributes.class));
            keys.add(S3FileAttributesCache.generateCacheKey(fileObjectId, PosixFileAttributes.class));
        } else {
            fileObjectId = S3ObjectId.builder().bucket(objectId.getBucket()).key(objectId.getKey() + "/").build();
            keys.add(S3FileAttributesCache.generateCacheKey(fileObjectId, BasicFileAttributes.class));
            keys.add(S3FileAttributesCache.generateCacheKey(fileObjectId, PosixFileAttributes.class));
        }
        for (String key : keys) {
            try {
                logger.trace("Invalidate cache key {}", (Object)key);
                this.cache.invalidate((Object)key);
            }
            catch (NullPointerException nullPointerException) {}
        }
    }

    public void invalidateAll() {
        logger.trace("Invalidate all cache");
        this.cache.invalidateAll();
    }

    public CacheStats stats() {
        return this.cache.stats();
    }

    protected Caffeine<String, Optional<S3BasicFileAttributes>> cacheBuilder(int cacheTTL, int cacheSize) {
        Caffeine builder = Caffeine.newBuilder().expireAfter((Expiry)new S3FileAttributesCachePolicy(cacheTTL));
        builder.maximumSize((long)cacheSize);
        builder.recordStats();
        builder.evictionListener((key, value, cause) -> logger.trace("Key {} was evicted (reason: {})", key, (Object)cause));
        builder.removalListener((key, value, cause) -> logger.trace("Key {} was removed (reason: {})", key, (Object)cause));
        return builder;
    }

    protected S3BasicFileAttributes fetchAttribute(S3Path path, String key) {
        try {
            if (key.contains(BasicFileAttributes.class.getSimpleName())) {
                return this.s3Utils.getS3FileAttributes(path);
            }
            if (key.contains(PosixFileAttributes.class.getSimpleName())) {
                return this.s3Utils.getS3PosixFileAttributes(path);
            }
            return null;
        }
        catch (NoSuchFileException e) {
            return null;
        }
    }
}

