/*
 * Decompiled with CFR 0.152.
 */
package org.cardanofoundation.cip30;

import java.security.MessageDigest;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.cardanofoundation.cip30.Cip30VerificationResult;
import org.cardanofoundation.cip30.ValidationError;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import shaded.co.nstant.in.cbor.CborDecoder;
import shaded.co.nstant.in.cbor.CborException;
import shaded.co.nstant.in.cbor.model.Array;
import shaded.co.nstant.in.cbor.model.ByteString;
import shaded.co.nstant.in.cbor.model.DataItem;
import shaded.co.nstant.in.cbor.model.MajorType;
import shaded.co.nstant.in.cbor.model.Map;
import shaded.co.nstant.in.cbor.model.UnicodeString;
import shaded.com.bloxbean.cardano.client.address.Address;
import shaded.com.bloxbean.cardano.client.address.AddressProvider;
import shaded.com.bloxbean.cardano.client.cip.cip8.COSEKey;
import shaded.com.bloxbean.cardano.client.common.cbor.CborSerializationUtil;
import shaded.com.bloxbean.cardano.client.util.HexUtil;
import shaded.javax.annotation.Nullable;
import shaded.javax.annotation.ParametersAreNonnullByDefault;
import shaded.net.i2p.crypto.eddsa.EdDSAEngine;
import shaded.net.i2p.crypto.eddsa.EdDSAPublicKey;
import shaded.net.i2p.crypto.eddsa.spec.EdDSANamedCurveTable;
import shaded.net.i2p.crypto.eddsa.spec.EdDSAParameterSpec;
import shaded.net.i2p.crypto.eddsa.spec.EdDSAPublicKeySpec;

@ParametersAreNonnullByDefault
public final class CIP30Verifier {
    private static final Logger logger = LoggerFactory.getLogger(CIP30Verifier.class);
    private static final EdDSAParameterSpec ED_DSA_PARAMETER_SPEC = EdDSANamedCurveTable.getByName("Ed25519");
    private final String coseSign1;
    private final Optional<String> coseKey;

    public CIP30Verifier(String coseSign1) {
        this(coseSign1, Optional.empty());
    }

    public CIP30Verifier(String coseSign1, @Nullable String coseKey) {
        this(coseSign1, Optional.ofNullable(coseKey));
    }

    public CIP30Verifier(String coseSign1, Optional<String> coseKey) {
        Objects.requireNonNull(coseSign1, "signature cannot be null");
        if (coseSign1.isBlank()) {
            throw new IllegalArgumentException("signature cannot blank");
        }
        this.coseSign1 = coseSign1;
        this.coseKey = coseKey;
    }

    private static boolean verifyMessage(byte[] cosePayload, byte[] signatureBytes, byte[] publicKeyBytes) {
        try {
            EdDSAPublicKey publicKey = new EdDSAPublicKey(new EdDSAPublicKeySpec(publicKeyBytes, ED_DSA_PARAMETER_SPEC));
            EdDSAEngine signature = new EdDSAEngine(MessageDigest.getInstance(ED_DSA_PARAMETER_SPEC.getHashAlgorithm()));
            signature.initVerify(publicKey);
            signature.setParameter(EdDSAEngine.ONE_SHOT_MODE);
            signature.update(cosePayload);
            return signature.verify(signatureBytes);
        }
        catch (Exception e) {
            return false;
        }
    }

    public Cip30VerificationResult verify() {
        try {
            byte[] signatureAsBytes = HexUtil.decodeHexString(this.coseSign1);
            DataItem coseCbor = CborDecoder.decode(signatureAsBytes).get(0);
            if (coseCbor.getMajorType() != MajorType.ARRAY) {
                logger.error("Invalid CIP-30 signature. Structure is not an array.");
                return Cip30VerificationResult.createInvalid(ValidationError.CIP8_FORMAT_ERROR);
            }
            List<DataItem> dataItems = ((Array)coseCbor).getDataItems();
            ByteString protectedHeader = (ByteString)dataItems.get(0);
            ByteString messageByteString = (ByteString)dataItems.get(2);
            ByteString ed25519SignatureByteString = (ByteString)dataItems.get(3);
            DataItem protectedHeaderDecoded = CborDecoder.decode(protectedHeader.getBytes()).get(0);
            if (protectedHeaderDecoded.getMajorType() != MajorType.MAP) {
                logger.error("Invalid CIP-30 signature. Protected header structure is not a map.");
                return Cip30VerificationResult.createInvalid(ValidationError.CIP8_FORMAT_ERROR);
            }
            Map protectedHeaderMap = (Map)protectedHeaderDecoded;
            Array signatureArray = new Array();
            signatureArray.add(new UnicodeString("Signature1"));
            signatureArray.add(protectedHeader);
            signatureArray.add(new ByteString(new byte[0]));
            signatureArray.add(messageByteString);
            byte[] ed25519PublicKeyBytes = CIP30Verifier.deserializeED25519PublicKey(this.coseKey, protectedHeaderMap);
            if (ed25519PublicKeyBytes == null) {
                logger.warn("No public key found.");
                return Cip30VerificationResult.createInvalid(ValidationError.NO_PUBLIC_KEY);
            }
            byte[] cosePayload = CborSerializationUtil.serialize(signatureArray);
            boolean isSignatureVerified = CIP30Verifier.verifyMessage(cosePayload, ed25519SignatureByteString.getBytes(), ed25519PublicKeyBytes);
            Optional<byte[]> pubKey1 = Optional.ofNullable(CIP30Verifier.deserializeCardanoAddressFromHeaderMap(protectedHeaderMap));
            Optional<byte[]> pubKey2 = Optional.ofNullable(CIP30Verifier.getED25519PublicKeyFromCoseKey(this.coseKey));
            boolean isAddressVerified = true;
            if (pubKey1.isPresent() && pubKey2.isPresent()) {
                isAddressVerified = AddressProvider.verifyAddress(new Address(pubKey1.orElseThrow()), pubKey2.orElseThrow());
            }
            Cip30VerificationResult.Builder b = Cip30VerificationResult.Builder.newBuilder();
            if (isSignatureVerified && isAddressVerified) {
                b.valid();
            }
            Optional.ofNullable(CIP30Verifier.deserializeCardanoAddressFromHeaderMap(protectedHeaderMap)).ifPresent(b::address);
            Optional.ofNullable(messageByteString.getBytes()).map(b::message);
            b.ed25519PublicKey(ed25519PublicKeyBytes);
            b.ed25519Signature(ed25519SignatureByteString.getBytes());
            b.cosePayload(cosePayload);
            return b.build();
        }
        catch (ClassCastException | CborException e) {
            return Cip30VerificationResult.createInvalid(ValidationError.CIP8_FORMAT_ERROR);
        }
    }

    public String getCOSESign1() {
        return this.coseSign1;
    }

    public Optional<String> getCoseKey() {
        return this.coseKey;
    }

    @Nullable
    private static byte[] deserializeED25519PublicKey(Optional<String> coseKey, Map protectedHeaderMap) {
        return coseKey.map(hexString -> COSEKey.deserialize(HexUtil.decodeHexString(hexString)).otherHeaderAsBytes(-2L)).orElseGet(() -> CIP30Verifier.deserializeCardanoAddressFromHeaderMap(protectedHeaderMap));
    }

    @Nullable
    private static byte[] getED25519PublicKeyFromCoseKey(Optional<String> coseKey) {
        return coseKey.map(hexString -> COSEKey.deserialize(HexUtil.decodeHexString(hexString)).otherHeaderAsBytes(-2L)).orElse(null);
    }

    @Nullable
    private static byte[] deserializeCardanoAddressFromHeaderMap(Map protectedHeaderMap) {
        return Optional.ofNullable((ByteString)protectedHeaderMap.get(new UnicodeString("address"))).filter(byteString -> byteString.getBytes().length > 0).map(ByteString::getBytes).orElse(null);
    }
}

