/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.tools;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Resources;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.protobuf.ByteString;
import com.subgraph.orchid.TorClient;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.security.SecureRandom;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.LogManager;
import javax.annotation.Nullable;
import joptsimple.ArgumentAcceptingOptionSpec;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import joptsimple.ValueConverter;
import joptsimple.util.DateConverter;
import org.bitcoin.protocols.payments.Protos;
import org.bitcoinj.core.AbstractBlockChain;
import org.bitcoinj.core.Address;
import org.bitcoinj.core.AddressFormatException;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.BlockChain;
import org.bitcoinj.core.CheckpointManager;
import org.bitcoinj.core.Coin;
import org.bitcoinj.core.Context;
import org.bitcoinj.core.DumpedPrivateKey;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.FilteredBlock;
import org.bitcoinj.core.FullPrunedBlockChain;
import org.bitcoinj.core.InsufficientMoneyException;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Peer;
import org.bitcoinj.core.PeerAddress;
import org.bitcoinj.core.PeerGroup;
import org.bitcoinj.core.ScriptException;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.core.Utils;
import org.bitcoinj.core.VerificationException;
import org.bitcoinj.core.WrongNetworkException;
import org.bitcoinj.core.listeners.BlocksDownloadedEventListener;
import org.bitcoinj.core.listeners.DownloadProgressTracker;
import org.bitcoinj.core.listeners.PeerDataEventListener;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.MnemonicCode;
import org.bitcoinj.crypto.MnemonicException;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.net.discovery.DnsDiscovery;
import org.bitcoinj.net.discovery.PeerDiscovery;
import org.bitcoinj.params.MainNetParams;
import org.bitcoinj.params.RegTestParams;
import org.bitcoinj.params.TestNet3Params;
import org.bitcoinj.protocols.payments.PaymentProtocol;
import org.bitcoinj.protocols.payments.PaymentProtocolException;
import org.bitcoinj.protocols.payments.PaymentSession;
import org.bitcoinj.script.ScriptBuilder;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.FullPrunedBlockStore;
import org.bitcoinj.store.H2FullPrunedBlockStore;
import org.bitcoinj.store.SPVBlockStore;
import org.bitcoinj.tools.NetworkEnum;
import org.bitcoinj.uri.BitcoinURI;
import org.bitcoinj.uri.BitcoinURIParseException;
import org.bitcoinj.utils.BriefLogFormatter;
import org.bitcoinj.wallet.DeterministicKeyChain;
import org.bitcoinj.wallet.DeterministicSeed;
import org.bitcoinj.wallet.DeterministicUpgradeRequiredException;
import org.bitcoinj.wallet.DeterministicUpgradeRequiresPassword;
import org.bitcoinj.wallet.MarriedKeyChain;
import org.bitcoinj.wallet.Protos;
import org.bitcoinj.wallet.SendRequest;
import org.bitcoinj.wallet.Wallet;
import org.bitcoinj.wallet.WalletExtension;
import org.bitcoinj.wallet.WalletProtobufSerializer;
import org.bitcoinj.wallet.listeners.WalletChangeEventListener;
import org.bitcoinj.wallet.listeners.WalletCoinsReceivedEventListener;
import org.bitcoinj.wallet.listeners.WalletCoinsSentEventListener;
import org.bitcoinj.wallet.listeners.WalletReorganizeEventListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.params.KeyParameter;
import org.spongycastle.util.encoders.Hex;

public class WalletTool {
    private static final Logger log = LoggerFactory.getLogger(WalletTool.class);
    private static OptionSet options;
    private static OptionSpec<Date> dateFlag;
    private static OptionSpec<Long> unixtimeFlag;
    private static OptionSpec<String> seedFlag;
    private static OptionSpec<String> watchFlag;
    private static OptionSpec<String> xpubkeysFlag;
    private static NetworkParameters params;
    private static File walletFile;
    private static BlockStore store;
    private static AbstractBlockChain chain;
    private static PeerGroup peers;
    private static Wallet wallet;
    private static File chainFileName;
    private static ValidationMode mode;
    private static String password;
    private static Protos.PaymentRequest paymentRequest;
    private static OptionSpec<Integer> lookaheadSize;
    private static Condition condition;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws Exception {
        ActionEnum action;
        OptionParser parser = new OptionParser();
        parser.accepts("help");
        parser.accepts("force");
        parser.accepts("debuglog");
        ArgumentAcceptingOptionSpec walletFileName = parser.accepts("wallet").withRequiredArg().defaultsTo((Object)"wallet", (Object[])new String[0]);
        seedFlag = parser.accepts("seed").withRequiredArg();
        watchFlag = parser.accepts("watchkey").withRequiredArg();
        ArgumentAcceptingOptionSpec netFlag = parser.accepts("net").withRequiredArg().ofType(NetworkEnum.class).defaultsTo((Object)NetworkEnum.MAIN, (Object[])new NetworkEnum[0]);
        dateFlag = parser.accepts("date").withRequiredArg().ofType(Date.class).withValuesConvertedBy((ValueConverter)DateConverter.datePattern((String)"yyyy/MM/dd"));
        ArgumentAcceptingOptionSpec waitForFlag = parser.accepts("waitfor").withRequiredArg().ofType(WaitForEnum.class);
        ArgumentAcceptingOptionSpec modeFlag = parser.accepts("mode").withRequiredArg().ofType(ValidationMode.class).defaultsTo((Object)ValidationMode.SPV, (Object[])new ValidationMode[0]);
        ArgumentAcceptingOptionSpec chainFlag = parser.accepts("chain").withRequiredArg();
        parser.accepts("pubkey").withRequiredArg();
        parser.accepts("privkey").withRequiredArg();
        parser.accepts("addr").withRequiredArg();
        parser.accepts("peers").withRequiredArg();
        xpubkeysFlag = parser.accepts("xpubkeys").withRequiredArg();
        ArgumentAcceptingOptionSpec outputFlag = parser.accepts("output").withRequiredArg();
        parser.accepts("value").withRequiredArg();
        ArgumentAcceptingOptionSpec feePerKbOption = parser.accepts("fee-per-kb").withRequiredArg();
        unixtimeFlag = parser.accepts("unixtime").withRequiredArg().ofType(Long.class);
        ArgumentAcceptingOptionSpec conditionFlag = parser.accepts("condition").withRequiredArg();
        parser.accepts("locktime").withRequiredArg();
        parser.accepts("allow-unconfirmed");
        parser.accepts("offline");
        parser.accepts("ignore-mandatory-extensions");
        lookaheadSize = parser.accepts("lookahead-size").withRequiredArg().ofType(Integer.class);
        ArgumentAcceptingOptionSpec passwordFlag = parser.accepts("password").withRequiredArg();
        ArgumentAcceptingOptionSpec paymentRequestLocation = parser.accepts("payment-request").withRequiredArg();
        parser.accepts("no-pki");
        parser.accepts("tor");
        parser.accepts("dump-privkeys");
        ArgumentAcceptingOptionSpec refundFlag = parser.accepts("refund-to").withRequiredArg();
        ArgumentAcceptingOptionSpec txHashFlag = parser.accepts("txhash").withRequiredArg();
        options = parser.parse(args);
        if (args.length == 0 || options.has("help") || options.nonOptionArguments().size() < 1 || options.nonOptionArguments().contains("help")) {
            System.out.println(Resources.toString((URL)WalletTool.class.getResource("wallet-tool-help.txt"), (Charset)Charsets.UTF_8));
            return;
        }
        try {
            String actionStr = (String)options.nonOptionArguments().get(0);
            actionStr = actionStr.toUpperCase().replace("-", "_");
            action = ActionEnum.valueOf(actionStr);
        }
        catch (IllegalArgumentException e) {
            System.err.println("Could not understand action name " + (String)options.nonOptionArguments().get(0));
            return;
        }
        if (options.has("debuglog")) {
            BriefLogFormatter.init();
            log.info("Starting up ...");
        } else {
            java.util.logging.Logger logger = LogManager.getLogManager().getLogger("");
            logger.setLevel(Level.SEVERE);
        }
        switch ((NetworkEnum)((Object)netFlag.value(options))) {
            case MAIN: 
            case PROD: {
                params = MainNetParams.get();
                chainFileName = new File("mainnet.chain");
                break;
            }
            case TEST: {
                params = TestNet3Params.get();
                chainFileName = new File("testnet.chain");
                break;
            }
            case REGTEST: {
                params = RegTestParams.get();
                chainFileName = new File("regtest.chain");
                break;
            }
            default: {
                throw new RuntimeException("Unreachable.");
            }
        }
        Context.propagate((Context)new Context(params));
        mode = (ValidationMode)((Object)modeFlag.value(options));
        if (options.has((OptionSpec)chainFlag)) {
            chainFileName = new File((String)chainFlag.value(options));
        }
        if (options.has("condition")) {
            condition = new Condition((String)conditionFlag.value(options));
        }
        if (options.has((OptionSpec)passwordFlag)) {
            password = (String)passwordFlag.value(options);
        }
        walletFile = new File((String)walletFileName.value(options));
        if (action == ActionEnum.CREATE) {
            WalletTool.createWallet(options, params, walletFile);
            return;
        }
        if (!walletFile.exists()) {
            System.err.println("Specified wallet file " + walletFile + " does not exist. Try wallet-tool --wallet=" + walletFile + " create");
            return;
        }
        if (action == ActionEnum.RAW_DUMP) {
            try (FileInputStream stream = new FileInputStream(walletFile);){
                Protos.Wallet proto = WalletProtobufSerializer.parseToProto((InputStream)stream);
                proto = WalletTool.attemptHexConversion(proto);
                System.out.println(proto.toString());
                return;
            }
        }
        try (InputStream walletInputStream = null;){
            boolean forceReset = action == ActionEnum.RESET || action == ActionEnum.SYNC && options.has("force");
            WalletProtobufSerializer loader = new WalletProtobufSerializer();
            if (options.has("ignore-mandatory-extensions")) {
                loader.setRequireMandatoryExtensions(false);
            }
            if (!(wallet = loader.readWallet(walletInputStream = new BufferedInputStream(new FileInputStream(walletFile)), forceReset, (WalletExtension[])null)).getParams().equals((Object)params)) {
                System.err.println("Wallet does not match requested network parameters: " + wallet.getParams().getId() + " vs " + params.getId());
                return;
            }
        }
        switch (action) {
            case DUMP: {
                WalletTool.dumpWallet();
                break;
            }
            case ADD_KEY: {
                WalletTool.addKey();
                break;
            }
            case ADD_ADDR: {
                WalletTool.addAddr();
                break;
            }
            case DELETE_KEY: {
                WalletTool.deleteKey();
                break;
            }
            case CURRENT_RECEIVE_ADDR: {
                WalletTool.currentReceiveAddr();
                break;
            }
            case RESET: {
                WalletTool.reset();
                break;
            }
            case SYNC: {
                WalletTool.syncChain();
                break;
            }
            case SEND: {
                String lockTime;
                if (options.has((OptionSpec)paymentRequestLocation) && options.has((OptionSpec)outputFlag)) {
                    System.err.println("--payment-request and --output cannot be used together.");
                    return;
                }
                if (options.has((OptionSpec)outputFlag)) {
                    Coin feePerKb = null;
                    if (options.has((OptionSpec)feePerKbOption)) {
                        feePerKb = Coin.parseCoin((String)((String)options.valueOf((OptionSpec)feePerKbOption)));
                    }
                    lockTime = null;
                    if (options.has("locktime")) {
                        lockTime = (String)options.valueOf("locktime");
                    }
                    boolean allowUnconfirmed = options.has("allow-unconfirmed");
                    WalletTool.send(outputFlag.values(options), feePerKb, lockTime, allowUnconfirmed);
                    break;
                }
                if (options.has((OptionSpec)paymentRequestLocation)) {
                    WalletTool.sendPaymentRequest((String)paymentRequestLocation.value(options), !options.has("no-pki"));
                    break;
                }
                System.err.println("You must specify a --payment-request or at least one --output=addr:value.");
                return;
            }
            case SEND_CLTVPAYMENTCHANNEL: {
                if (!options.has((OptionSpec)outputFlag)) {
                    System.err.println("You must specify a --output=addr:value");
                    return;
                }
                Coin feePerKb = null;
                if (options.has((OptionSpec)feePerKbOption)) {
                    feePerKb = Coin.parseCoin((String)((String)options.valueOf((OptionSpec)feePerKbOption)));
                }
                if (!options.has("locktime")) {
                    System.err.println("You must specify a --locktime");
                    return;
                }
                String lockTime = (String)options.valueOf("locktime");
                boolean allowUnconfirmed = options.has("allow-unconfirmed");
                if (!options.has((OptionSpec)refundFlag)) {
                    System.err.println("You must specify an address to refund money to after expiry: --refund-to=addr");
                    return;
                }
                WalletTool.sendCLTVPaymentChannel((String)refundFlag.value(options), (String)outputFlag.value(options), feePerKb, lockTime, allowUnconfirmed);
                break;
            }
            case SETTLE_CLTVPAYMENTCHANNEL: {
                if (!options.has((OptionSpec)outputFlag)) {
                    System.err.println("You must specify a --output=addr:value");
                    return;
                }
                Coin feePerKb = null;
                if (options.has((OptionSpec)feePerKbOption)) {
                    feePerKb = Coin.parseCoin((String)((String)options.valueOf((OptionSpec)feePerKbOption)));
                }
                boolean allowUnconfirmed = options.has("allow-unconfirmed");
                if (!options.has((OptionSpec)txHashFlag)) {
                    System.err.println("You must specify the transaction to spend: --txhash=tx-hash");
                    return;
                }
                WalletTool.settleCLTVPaymentChannel((String)txHashFlag.value(options), (String)outputFlag.value(options), feePerKb, allowUnconfirmed);
                break;
            }
            case REFUND_CLTVPAYMENTCHANNEL: {
                if (!options.has((OptionSpec)outputFlag)) {
                    System.err.println("You must specify a --output=addr:value");
                    return;
                }
                Coin feePerKb = null;
                if (options.has((OptionSpec)feePerKbOption)) {
                    feePerKb = Coin.parseCoin((String)((String)options.valueOf((OptionSpec)feePerKbOption)));
                }
                boolean allowUnconfirmed = options.has("allow-unconfirmed");
                if (!options.has((OptionSpec)txHashFlag)) {
                    System.err.println("You must specify the transaction to spend: --txhash=tx-hash");
                    return;
                }
                WalletTool.refundCLTVPaymentChannel((String)txHashFlag.value(options), (String)outputFlag.value(options), feePerKb, allowUnconfirmed);
                break;
            }
            case ENCRYPT: {
                WalletTool.encrypt();
                break;
            }
            case DECRYPT: {
                WalletTool.decrypt();
                break;
            }
            case MARRY: {
                WalletTool.marry();
                break;
            }
            case ROTATE: {
                WalletTool.rotate();
                break;
            }
            case SET_CREATION_TIME: {
                WalletTool.setCreationTime();
            }
        }
        if (!wallet.isConsistent()) {
            System.err.println("************** WALLET IS INCONSISTENT *****************");
            return;
        }
        WalletTool.saveWallet(walletFile);
        if (options.has((OptionSpec)waitForFlag)) {
            WaitForEnum value;
            try {
                value = (WaitForEnum)((Object)waitForFlag.value(options));
            }
            catch (Exception e) {
                System.err.println("Could not understand the --waitfor flag: Valid options are WALLET_TX, BLOCK, BALANCE and EVER");
                return;
            }
            WalletTool.wait(value);
            if (!wallet.isConsistent()) {
                System.err.println("************** WALLET IS INCONSISTENT *****************");
                return;
            }
            WalletTool.saveWallet(walletFile);
        }
        WalletTool.shutdown();
    }

    private static Protos.Wallet attemptHexConversion(Protos.Wallet proto) {
        try {
            Protos.Wallet.Builder builder = proto.toBuilder();
            for (Protos.Transaction.Builder tx : builder.getTransactionBuilderList()) {
                tx.setHash(WalletTool.bytesToHex(tx.getHash()));
                for (int i = 0; i < tx.getBlockHashCount(); ++i) {
                    tx.setBlockHash(i, WalletTool.bytesToHex(tx.getBlockHash(i)));
                }
                for (Protos.TransactionInput.Builder input : tx.getTransactionInputBuilderList()) {
                    input.setTransactionOutPointHash(WalletTool.bytesToHex(input.getTransactionOutPointHash()));
                }
                for (Protos.TransactionOutput.Builder output : tx.getTransactionOutputBuilderList()) {
                    if (!output.hasSpentByTransactionHash()) continue;
                    output.setSpentByTransactionHash(WalletTool.bytesToHex(output.getSpentByTransactionHash()));
                }
            }
            return builder.build();
        }
        catch (Throwable throwable) {
            log.error("Failed to do hex conversion on wallet proto", throwable);
            return proto;
        }
    }

    private static ByteString bytesToHex(ByteString bytes) {
        return ByteString.copyFrom((byte[])Utils.HEX.encode(bytes.toByteArray()).getBytes());
    }

    private static void marry() {
        if (!options.has(xpubkeysFlag)) {
            throw new IllegalStateException();
        }
        String[] xpubkeys = ((String)options.valueOf(xpubkeysFlag)).split(",");
        ImmutableList.Builder keys = ImmutableList.builder();
        for (String xpubkey : xpubkeys) {
            keys.add((Object)DeterministicKey.deserializeB58(null, (String)xpubkey.trim(), (NetworkParameters)params));
        }
        MarriedKeyChain chain = ((MarriedKeyChain.Builder)MarriedKeyChain.builder().random(new SecureRandom())).followingKeys((List)keys.build()).build();
        wallet.addAndActivateHDChain((DeterministicKeyChain)chain);
    }

    private static void rotate() throws BlockStoreException {
        WalletTool.setup();
        peers.start();
        long rotationTimeSecs = Utils.currentTimeSeconds();
        if (options.has(dateFlag)) {
            rotationTimeSecs = ((Date)options.valueOf(dateFlag)).getTime() / 1000L;
        } else if (options.has(unixtimeFlag)) {
            rotationTimeSecs = (Long)options.valueOf(unixtimeFlag);
        }
        log.info("Setting wallet key rotation time to {}", (Object)rotationTimeSecs);
        wallet.setKeyRotationTime(rotationTimeSecs);
        KeyParameter aesKey = null;
        if (wallet.isEncrypted() && (aesKey = WalletTool.passwordToKey(true)) == null) {
            return;
        }
        Futures.getUnchecked((Future)wallet.doMaintenance(aesKey, true));
    }

    private static void encrypt() {
        if (password == null) {
            System.err.println("You must provide a --password");
            return;
        }
        if (wallet.isEncrypted()) {
            System.err.println("This wallet is already encrypted.");
            return;
        }
        wallet.encrypt((CharSequence)password);
    }

    private static void decrypt() {
        if (password == null) {
            System.err.println("You must provide a --password");
            return;
        }
        if (!wallet.isEncrypted()) {
            System.err.println("This wallet is not encrypted.");
            return;
        }
        try {
            wallet.decrypt((CharSequence)password);
        }
        catch (KeyCrypterException e) {
            System.err.println("Password incorrect.");
        }
    }

    private static void addAddr() {
        String addr = (String)options.valueOf("addr");
        if (addr == null) {
            System.err.println("You must specify an --addr to watch.");
            return;
        }
        try {
            Address address = Address.fromBase58((NetworkParameters)params, (String)addr);
            wallet.addWatchedAddress(address, WalletTool.getCreationTimeSeconds());
        }
        catch (AddressFormatException e) {
            System.err.println("Could not parse given address, or wrong network: " + addr);
        }
    }

    private static void send(List<String> outputs, Coin feePerKb, String lockTimeStr, boolean allowUnconfirmed) throws VerificationException {
        try {
            Transaction t = new Transaction(params);
            for (String spec : outputs) {
                try {
                    OutputSpec outputSpec = new OutputSpec(spec);
                    if (outputSpec.isAddress()) {
                        t.addOutput(outputSpec.value, outputSpec.addr);
                        continue;
                    }
                    t.addOutput(outputSpec.value, outputSpec.key);
                }
                catch (WrongNetworkException e) {
                    System.err.println("Malformed output specification, address is for a different network: " + spec);
                    return;
                }
                catch (AddressFormatException e) {
                    System.err.println("Malformed output specification, could not parse as address: " + spec);
                    return;
                }
                catch (NumberFormatException e) {
                    System.err.println("Malformed output specification, could not parse as value: " + spec);
                    return;
                }
                catch (IllegalArgumentException e) {
                    System.err.println(e.getMessage());
                    return;
                }
            }
            SendRequest req = SendRequest.forTx((Transaction)t);
            if (t.getOutputs().size() == 1 && t.getOutput(0L).getValue().equals((Object)wallet.getBalance())) {
                log.info("Emptying out wallet, recipient may get less than what you expect");
                req.emptyWallet = true;
            }
            if (feePerKb != null) {
                req.feePerKb = feePerKb;
            }
            if (allowUnconfirmed) {
                wallet.allowSpendingUnconfirmedTransactions();
            }
            if (password != null) {
                req.aesKey = WalletTool.passwordToKey(true);
                if (req.aesKey == null) {
                    return;
                }
            }
            wallet.completeTx(req);
            try {
                if (lockTimeStr != null) {
                    t.setLockTime(WalletTool.parseLockTimeStr(lockTimeStr));
                    ((TransactionInput)t.getInputs().get(0)).setSequenceNumber(0L);
                    wallet.signTransaction(req);
                }
            }
            catch (ParseException e) {
                System.err.println("Could not understand --locktime of " + lockTimeStr);
                return;
            }
            catch (ScriptException e) {
                throw new RuntimeException(e);
            }
            t = req.tx;
            System.out.println(t.getHashAsString());
            if (options.has("offline")) {
                wallet.commitTx(t);
                return;
            }
            WalletTool.setup();
            peers.start();
            peers.broadcastTransaction(t).future().get();
            List peerList = peers.getConnectedPeers();
            if (peerList.size() == 1) {
                ((Peer)peerList.get(0)).ping().get();
            }
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (KeyCrypterException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        catch (InsufficientMoneyException e) {
            System.err.println("Insufficient funds: have " + wallet.getBalance().toFriendlyString());
        }
    }

    private static void sendCLTVPaymentChannel(String refund, String output, Coin feePerKb, String lockTimeStr, boolean allowUnconfirmed) throws VerificationException {
        try {
            long lockTime;
            ECKey refundKey;
            Coin value;
            ECKey outputKey;
            try {
                OutputSpec outputSpec = new OutputSpec(output);
                if (outputSpec.isAddress()) {
                    System.err.println("Output specification must be a public key");
                    return;
                }
                outputKey = outputSpec.key;
                value = outputSpec.value;
                byte[] refundPubKey = new BigInteger(refund, 16).toByteArray();
                refundKey = ECKey.fromPublicOnly((byte[])refundPubKey);
            }
            catch (WrongNetworkException e) {
                System.err.println("Malformed output specification, address is for a different network.");
                return;
            }
            catch (AddressFormatException e) {
                System.err.println("Malformed output specification, could not parse as address.");
                return;
            }
            catch (NumberFormatException e) {
                System.err.println("Malformed output specification, could not parse as value.");
                return;
            }
            catch (IllegalArgumentException e) {
                System.err.println(e.getMessage());
                return;
            }
            try {
                lockTime = WalletTool.parseLockTimeStr(lockTimeStr);
            }
            catch (ParseException e) {
                System.err.println("Could not understand --locktime of " + lockTimeStr);
                return;
            }
            catch (ScriptException e) {
                throw new RuntimeException(e);
            }
            SendRequest req = SendRequest.toCLTVPaymentChannel((NetworkParameters)params, (BigInteger)BigInteger.valueOf(lockTime), (ECKey)refundKey, (ECKey)outputKey, (Coin)value);
            if (req.tx.getOutputs().size() == 1 && req.tx.getOutput(0L).getValue().equals((Object)wallet.getBalance())) {
                log.info("Emptying out wallet, recipient may get less than what you expect");
                req.emptyWallet = true;
            }
            if (feePerKb != null) {
                req.feePerKb = feePerKb;
            }
            if (allowUnconfirmed) {
                wallet.allowSpendingUnconfirmedTransactions();
            }
            if (password != null) {
                req.aesKey = WalletTool.passwordToKey(true);
                if (req.aesKey == null) {
                    return;
                }
            }
            wallet.completeTx(req);
            System.out.println(req.tx.getHashAsString());
            if (options.has("offline")) {
                wallet.commitTx(req.tx);
                return;
            }
            WalletTool.setup();
            peers.start();
            peers.broadcastTransaction(req.tx).future().get();
            List peerList = peers.getConnectedPeers();
            if (peerList.size() == 1) {
                ((Peer)peerList.get(0)).ping().get();
            }
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (KeyCrypterException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        catch (InsufficientMoneyException e) {
            System.err.println("Insufficient funds: have " + wallet.getBalance().toFriendlyString());
        }
    }

    private static void settleCLTVPaymentChannel(String txHash, String output, Coin feePerKb, boolean allowUnconfirmed) {
        try {
            Transaction lockTimeVerify;
            SendRequest req;
            Coin value;
            OutputSpec outputSpec;
            try {
                outputSpec = new OutputSpec(output);
                value = outputSpec.value;
            }
            catch (WrongNetworkException e) {
                System.err.println("Malformed output specification, address is for a different network.");
                return;
            }
            catch (AddressFormatException e) {
                System.err.println("Malformed output specification, could not parse as address.");
                return;
            }
            catch (NumberFormatException e) {
                System.err.println("Malformed output specification, could not parse as value.");
                return;
            }
            catch (IllegalArgumentException e) {
                System.err.println(e.getMessage());
                return;
            }
            SendRequest sendRequest = req = outputSpec.isAddress() ? SendRequest.to((Address)outputSpec.addr, (Coin)value) : SendRequest.to((NetworkParameters)params, (ECKey)outputSpec.key, (Coin)value);
            if (feePerKb != null) {
                req.feePerKb = feePerKb;
            }
            if ((lockTimeVerify = wallet.getTransaction(Sha256Hash.wrap((String)txHash))) == null) {
                System.err.println("Couldn't find transaction with given hash");
                return;
            }
            TransactionOutput lockTimeVerifyOutput = null;
            for (TransactionOutput out : lockTimeVerify.getOutputs()) {
                if (!out.getScriptPubKey().isSentToCLTVPaymentChannel()) continue;
                lockTimeVerifyOutput = out;
            }
            if (lockTimeVerifyOutput == null) {
                System.err.println("TX to spend wasn't sent to LockTimeVerify");
                return;
            }
            if (!value.equals((Object)lockTimeVerifyOutput.getValue())) {
                System.err.println("You must spend all the money in the input transaction");
            }
            if (allowUnconfirmed) {
                wallet.allowSpendingUnconfirmedTransactions();
            }
            if (password != null) {
                req.aesKey = WalletTool.passwordToKey(true);
                if (req.aesKey == null) {
                    return;
                }
            }
            ECKey key1 = wallet.findKeyFromPubKey(lockTimeVerifyOutput.getScriptPubKey().getCLTVPaymentChannelSenderPubKey());
            ECKey key2 = wallet.findKeyFromPubKey(lockTimeVerifyOutput.getScriptPubKey().getCLTVPaymentChannelRecipientPubKey());
            if (key1 == null || key2 == null) {
                System.err.println("Don't own private keys for both pubkeys");
                return;
            }
            TransactionInput input = new TransactionInput(params, req.tx, new byte[0], lockTimeVerifyOutput.getOutPointFor());
            req.tx.addInput(input);
            TransactionSignature sig1 = req.tx.calculateSignature(0, key1, lockTimeVerifyOutput.getScriptPubKey(), Transaction.SigHash.SINGLE, false);
            TransactionSignature sig2 = req.tx.calculateSignature(0, key2, lockTimeVerifyOutput.getScriptPubKey(), Transaction.SigHash.SINGLE, false);
            input.setScriptSig(ScriptBuilder.createCLTVPaymentChannelInput((TransactionSignature)sig1, (TransactionSignature)sig2));
            System.out.println(req.tx.getHashAsString());
            if (options.has("offline")) {
                wallet.commitTx(req.tx);
                return;
            }
            WalletTool.setup();
            peers.start();
            peers.broadcastTransaction(req.tx).future().get();
            List peerList = peers.getConnectedPeers();
            if (peerList.size() == 1) {
                ((Peer)peerList.get(0)).ping().get();
            }
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (KeyCrypterException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private static void refundCLTVPaymentChannel(String txHash, String output, Coin feePerKb, boolean allowUnconfirmed) {
        try {
            ECKey key;
            Transaction lockTimeVerify;
            SendRequest req;
            Coin value;
            OutputSpec outputSpec;
            try {
                outputSpec = new OutputSpec(output);
                value = outputSpec.value;
            }
            catch (WrongNetworkException e) {
                System.err.println("Malformed output specification, address is for a different network.");
                return;
            }
            catch (AddressFormatException e) {
                System.err.println("Malformed output specification, could not parse as address.");
                return;
            }
            catch (NumberFormatException e) {
                System.err.println("Malformed output specification, could not parse as value.");
                return;
            }
            catch (IllegalArgumentException e) {
                System.err.println(e.getMessage());
                return;
            }
            SendRequest sendRequest = req = outputSpec.isAddress() ? SendRequest.to((Address)outputSpec.addr, (Coin)value) : SendRequest.to((NetworkParameters)params, (ECKey)outputSpec.key, (Coin)value);
            if (feePerKb != null) {
                req.feePerKb = feePerKb;
            }
            if ((lockTimeVerify = wallet.getTransaction(Sha256Hash.wrap((String)txHash))) == null) {
                System.err.println("Couldn't find transaction with given hash");
                return;
            }
            TransactionOutput lockTimeVerifyOutput = null;
            for (TransactionOutput out : lockTimeVerify.getOutputs()) {
                if (!out.getScriptPubKey().isSentToCLTVPaymentChannel()) continue;
                lockTimeVerifyOutput = out;
            }
            if (lockTimeVerifyOutput == null) {
                System.err.println("TX to spend wasn't sent to LockTimeVerify");
                return;
            }
            req.tx.setLockTime(lockTimeVerifyOutput.getScriptPubKey().getCLTVPaymentChannelExpiry().longValue());
            if (!value.equals((Object)lockTimeVerifyOutput.getValue())) {
                System.err.println("You must spend all the money in the input transaction");
            }
            if (allowUnconfirmed) {
                wallet.allowSpendingUnconfirmedTransactions();
            }
            if (password != null) {
                req.aesKey = WalletTool.passwordToKey(true);
                if (req.aesKey == null) {
                    return;
                }
            }
            if ((key = wallet.findKeyFromPubKey(lockTimeVerifyOutput.getScriptPubKey().getCLTVPaymentChannelSenderPubKey())) == null) {
                System.err.println("Don't own private key for pubkey");
                return;
            }
            TransactionInput input = new TransactionInput(params, req.tx, new byte[0], lockTimeVerifyOutput.getOutPointFor());
            input.setSequenceNumber(0L);
            req.tx.addInput(input);
            TransactionSignature sig = req.tx.calculateSignature(0, key, lockTimeVerifyOutput.getScriptPubKey(), Transaction.SigHash.SINGLE, false);
            input.setScriptSig(ScriptBuilder.createCLTVPaymentChannelRefund((TransactionSignature)sig));
            System.out.println(req.tx.getHashAsString());
            if (options.has("offline")) {
                wallet.commitTx(req.tx);
                return;
            }
            WalletTool.setup();
            peers.start();
            peers.broadcastTransaction(req.tx).future().get();
            List peerList = peers.getConnectedPeers();
            if (peerList.size() == 1) {
                ((Peer)peerList.get(0)).ping().get();
            }
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (KeyCrypterException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private static long parseLockTimeStr(String lockTimeStr) throws ParseException {
        if (lockTimeStr.indexOf("/") != -1) {
            SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd", Locale.US);
            Date date = format.parse(lockTimeStr);
            return date.getTime() / 1000L;
        }
        return Long.parseLong(lockTimeStr);
    }

    private static void sendPaymentRequest(String location, boolean verifyPki) {
        if (location.startsWith("http") || location.startsWith("bitcoin")) {
            try {
                ListenableFuture future;
                if (location.startsWith("http")) {
                    future = PaymentSession.createFromUrl((String)location, (boolean)verifyPki);
                } else {
                    BitcoinURI paymentRequestURI = new BitcoinURI(location);
                    future = PaymentSession.createFromBitcoinUri((BitcoinURI)paymentRequestURI, (boolean)verifyPki);
                }
                PaymentSession session = (PaymentSession)future.get();
                if (session != null) {
                    WalletTool.send(session);
                }
                System.err.println("Server returned null session");
                System.exit(1);
            }
            catch (PaymentProtocolException e) {
                System.err.println("Error creating payment session " + e.getMessage());
                System.exit(1);
            }
            catch (BitcoinURIParseException e) {
                System.err.println("Invalid bitcoin uri: " + e.getMessage());
                System.exit(1);
            }
            catch (InterruptedException e) {
            }
            catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        } else {
            FileInputStream stream = null;
            try {
                File paymentRequestFile = new File(location);
                stream = new FileInputStream(paymentRequestFile);
            }
            catch (Exception e) {
                System.err.println("Failed to open file: " + e.getMessage());
                System.exit(1);
            }
            try {
                paymentRequest = ((Protos.PaymentRequest.Builder)Protos.PaymentRequest.newBuilder().mergeFrom((InputStream)stream)).build();
            }
            catch (IOException e) {
                System.err.println("Failed to parse payment request from file " + e.getMessage());
                System.exit(1);
            }
            PaymentSession session = null;
            try {
                session = new PaymentSession(paymentRequest, verifyPki);
            }
            catch (PaymentProtocolException e) {
                System.err.println("Error creating payment session " + e.getMessage());
                System.exit(1);
            }
            WalletTool.send(session);
        }
    }

    private static void send(PaymentSession session) {
        try {
            System.out.println("Payment Request");
            System.out.println("Coin: " + session.getValue().toFriendlyString());
            System.out.println("Date: " + session.getDate());
            System.out.println("Memo: " + session.getMemo());
            if (session.pkiVerificationData != null) {
                System.out.println("Pki-Verified Name: " + session.pkiVerificationData.displayName);
                System.out.println("PKI data verified by: " + session.pkiVerificationData.rootAuthorityName);
            }
            SendRequest req = session.getSendRequest();
            if (password != null) {
                req.aesKey = WalletTool.passwordToKey(true);
                if (req.aesKey == null) {
                    return;
                }
            }
            wallet.completeTx(req);
            if (options.has("offline")) {
                wallet.commitTx(req.tx);
                return;
            }
            WalletTool.setup();
            ListenableFuture future = session.sendPayment((List)ImmutableList.of((Object)req.tx), null, null);
            if (future == null) {
                peers.start();
                peers.broadcastTransaction(req.tx).future().get();
            } else {
                PaymentProtocol.Ack ack = (PaymentProtocol.Ack)future.get();
                wallet.commitTx(req.tx);
                System.out.println("Memo from server: " + ack.getMemo());
            }
        }
        catch (PaymentProtocolException e) {
            System.err.println("Failed to send payment " + e.getMessage());
            System.exit(1);
        }
        catch (VerificationException e) {
            System.err.println("Failed to send payment " + e.getMessage());
            System.exit(1);
        }
        catch (ExecutionException e) {
            System.err.println("Failed to send payment " + e.getMessage());
            System.exit(1);
        }
        catch (IOException e) {
            System.err.println("Invalid payment " + e.getMessage());
            System.exit(1);
        }
        catch (InterruptedException e) {
        }
        catch (InsufficientMoneyException e) {
            System.err.println("Insufficient funds: have " + wallet.getBalance().toFriendlyString());
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private static void wait(WaitForEnum waitFor) throws BlockStoreException {
        final CountDownLatch latch = new CountDownLatch(1);
        WalletTool.setup();
        switch (waitFor) {
            case EVER: {
                break;
            }
            case WALLET_TX: {
                wallet.addCoinsReceivedEventListener(new WalletCoinsReceivedEventListener(){

                    public void onCoinsReceived(Wallet wallet, Transaction tx, Coin prevBalance, Coin newBalance) {
                        System.out.println(tx.getHashAsString());
                        latch.countDown();
                    }
                });
                wallet.addCoinsSentEventListener(new WalletCoinsSentEventListener(){

                    public void onCoinsSent(Wallet wallet, Transaction tx, Coin prevBalance, Coin newBalance) {
                        System.out.println(tx.getHashAsString());
                        latch.countDown();
                    }
                });
                break;
            }
            case BLOCK: {
                peers.addBlocksDownloadedEventListener(new BlocksDownloadedEventListener(){

                    public void onBlocksDownloaded(Peer peer, Block block, @Nullable FilteredBlock filteredBlock, int blocksLeft) {
                        if (latch.getCount() == 0L) {
                            return;
                        }
                        System.out.println(block.getHashAsString());
                        latch.countDown();
                    }
                });
                break;
            }
            case BALANCE: {
                if (condition.matchBitcoins(wallet.getBalance(Wallet.BalanceType.ESTIMATED))) {
                    latch.countDown();
                    break;
                }
                WalletEventListener listener = new WalletEventListener(latch);
                wallet.addCoinsReceivedEventListener((WalletCoinsReceivedEventListener)listener);
                wallet.addCoinsSentEventListener((WalletCoinsSentEventListener)listener);
                wallet.addChangeEventListener((WalletChangeEventListener)listener);
                wallet.addReorganizeEventListener((WalletReorganizeEventListener)listener);
            }
        }
        if (!peers.isRunning()) {
            peers.startAsync();
        }
        try {
            latch.await();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private static void reset() {
        wallet.clearTransactions(0);
        WalletTool.saveWallet(walletFile);
    }

    private static void setup() throws BlockStoreException {
        boolean reset;
        if (store != null) {
            return;
        }
        boolean bl = reset = !chainFileName.exists();
        if (reset) {
            System.out.println("Chain file is missing so resetting the wallet.");
            WalletTool.reset();
        }
        if (mode == ValidationMode.SPV) {
            store = new SPVBlockStore(params, chainFileName);
            chain = new BlockChain(params, wallet, store);
            if (reset) {
                try {
                    CheckpointManager.checkpoint((NetworkParameters)params, (InputStream)CheckpointManager.openStream((NetworkParameters)params), (BlockStore)store, (long)wallet.getEarliestKeyCreationTime());
                    StoredBlock head = store.getChainHead();
                    System.out.println("Skipped to checkpoint " + head.getHeight() + " at " + Utils.dateTimeFormat((long)(head.getHeader().getTimeSeconds() * 1000L)));
                }
                catch (IOException x) {
                    System.out.println("Could not load checkpoints: " + x.getMessage());
                }
            }
        } else if (mode == ValidationMode.FULL) {
            H2FullPrunedBlockStore s = new H2FullPrunedBlockStore(params, chainFileName.getAbsolutePath(), 5000);
            store = s;
            chain = new FullPrunedBlockChain(params, wallet, (FullPrunedBlockStore)s);
        }
        wallet.autosaveToFile(walletFile, 5L, TimeUnit.SECONDS, null);
        if (options.has("tor")) {
            try {
                peers = PeerGroup.newWithTor((NetworkParameters)params, (AbstractBlockChain)chain, (TorClient)new TorClient());
            }
            catch (TimeoutException e) {
                System.err.println("Tor startup timed out, falling back to clear net ...");
            }
        }
        if (peers == null) {
            peers = new PeerGroup(params, chain);
        }
        peers.setUserAgent("WalletTool", "1.0");
        if (params == RegTestParams.get()) {
            peers.setMinBroadcastConnections(1);
        }
        peers.addWallet(wallet);
        if (options.has("peers")) {
            String[] peerAddrs;
            String peersFlag = (String)options.valueOf("peers");
            for (String peer : peerAddrs = peersFlag.split(",")) {
                try {
                    peers.addAddress(new PeerAddress(InetAddress.getByName(peer), params.getPort()));
                }
                catch (UnknownHostException e) {
                    System.err.println("Could not understand peer domain name/IP address: " + peer + ": " + e.getMessage());
                    System.exit(1);
                }
            }
        } else if (!options.has("tor")) {
            peers.addPeerDiscovery((PeerDiscovery)new DnsDiscovery(params));
        }
    }

    private static void syncChain() {
        try {
            WalletTool.setup();
            int startTransactions = wallet.getTransactions(true).size();
            DownloadProgressTracker listener = new DownloadProgressTracker();
            peers.start();
            peers.startBlockChainDownload((PeerDataEventListener)listener);
            try {
                listener.await();
            }
            catch (InterruptedException e) {
                System.err.println("Chain download interrupted, quitting ...");
                System.exit(1);
            }
            int endTransactions = wallet.getTransactions(true).size();
            if (endTransactions > startTransactions) {
                System.out.println("Synced " + (endTransactions - startTransactions) + " transactions.");
            }
        }
        catch (BlockStoreException e) {
            System.err.println("Error reading block chain file " + chainFileName + ": " + e.getMessage());
            e.printStackTrace();
        }
    }

    private static void shutdown() {
        try {
            if (peers == null) {
                return;
            }
            if (peers.isRunning()) {
                peers.stop();
            }
            WalletTool.saveWallet(walletFile);
            store.close();
            wallet = null;
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private static void createWallet(OptionSet options, NetworkParameters params, File walletFile) throws IOException {
        if (walletFile.exists() && !options.has("force")) {
            System.err.println("Wallet creation requested but " + walletFile + " already exists, use --force");
            return;
        }
        long creationTimeSecs = WalletTool.getCreationTimeSeconds();
        if (creationTimeSecs == 0L) {
            creationTimeSecs = MnemonicCode.BIP39_STANDARDISATION_TIME_SECS;
        }
        if (options.has(seedFlag)) {
            String seedStr = (String)options.valueOf(seedFlag);
            ImmutableList split = ImmutableList.copyOf((Iterable)Splitter.on((String)" ").omitEmptyStrings().split((CharSequence)seedStr));
            String passphrase = "";
            DeterministicSeed seed = new DeterministicSeed((List)split, null, passphrase, creationTimeSecs);
            try {
                seed.check();
            }
            catch (MnemonicException.MnemonicLengthException e) {
                System.err.println("The seed did not have 12 words in, perhaps you need quotes around it?");
                return;
            }
            catch (MnemonicException.MnemonicWordException e) {
                System.err.println("The seed contained an unrecognised word: " + e.badWord);
                return;
            }
            catch (MnemonicException.MnemonicChecksumException e) {
                System.err.println("The seed did not pass checksumming, perhaps one of the words is wrong?");
                return;
            }
            catch (MnemonicException e) {
                throw new RuntimeException(e);
            }
            wallet = Wallet.fromSeed((NetworkParameters)params, (DeterministicSeed)seed);
        } else {
            wallet = options.has(watchFlag) ? Wallet.fromWatchingKeyB58((NetworkParameters)params, (String)((String)options.valueOf(watchFlag)), (long)creationTimeSecs) : new Wallet(params);
        }
        if (password != null) {
            wallet.encrypt((CharSequence)password);
        }
        wallet.saveToFile(walletFile);
    }

    private static void saveWallet(File walletFile) {
        try {
            wallet.saveToFile(walletFile);
        }
        catch (IOException e) {
            System.err.println("Failed to save wallet! Old wallet should be left untouched.");
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static void addKey() {
        if (options.has("privkey") || options.has("pubkey")) {
            WalletTool.importKey();
        } else {
            DeterministicKey key;
            if (options.has(lookaheadSize)) {
                Integer size = (Integer)options.valueOf(lookaheadSize);
                log.info("Setting keychain lookahead size to {}", (Object)size);
                wallet.setKeyChainGroupLookaheadSize(size.intValue());
            }
            try {
                key = wallet.freshReceiveKey();
            }
            catch (DeterministicUpgradeRequiredException e) {
                try {
                    KeyParameter aesKey = WalletTool.passwordToKey(false);
                    wallet.upgradeToDeterministic(aesKey);
                }
                catch (DeterministicUpgradeRequiresPassword e2) {
                    System.err.println("This wallet must be upgraded to be deterministic, but it's encrypted: please supply the password and try again.");
                    return;
                }
                key = wallet.freshReceiveKey();
            }
            System.out.println(key.toAddress(params) + " " + key);
        }
    }

    @Nullable
    private static KeyParameter passwordToKey(boolean printError) {
        if (password == null) {
            if (printError) {
                System.err.println("You must provide a password.");
            }
            return null;
        }
        if (!wallet.checkPassword((CharSequence)password)) {
            if (printError) {
                System.err.println("The password is incorrect.");
            }
            return null;
        }
        return ((KeyCrypter)Preconditions.checkNotNull((Object)wallet.getKeyCrypter())).deriveKey((CharSequence)password);
    }

    private static void importKey() {
        ECKey key;
        long creationTimeSeconds = WalletTool.getCreationTimeSeconds();
        if (options.has("privkey")) {
            String data = (String)options.valueOf("privkey");
            if (data.startsWith("5J") || data.startsWith("5H") || data.startsWith("5K")) {
                DumpedPrivateKey dpk;
                try {
                    dpk = DumpedPrivateKey.fromBase58((NetworkParameters)params, (String)data);
                }
                catch (AddressFormatException e) {
                    System.err.println("Could not parse dumped private key " + data);
                    return;
                }
                key = dpk.getKey();
            } else {
                byte[] decode = Utils.parseAsHexOrBase58((String)data);
                if (decode == null) {
                    System.err.println("Could not understand --privkey as either hex or base58: " + data);
                    return;
                }
                key = ECKey.fromPrivate((BigInteger)new BigInteger(1, decode));
            }
            if (options.has("pubkey")) {
                System.out.println("You don't have to specify --pubkey when a private key is supplied.");
            }
            key.setCreationTimeSeconds(creationTimeSeconds);
        } else if (options.has("pubkey")) {
            byte[] pubkey = Utils.parseAsHexOrBase58((String)((String)options.valueOf("pubkey")));
            key = ECKey.fromPublicOnly((byte[])pubkey);
            key.setCreationTimeSeconds(creationTimeSeconds);
        } else {
            throw new IllegalStateException();
        }
        if (wallet.findKeyFromPubKey(key.getPubKey()) != null) {
            System.err.println("That key already exists in this wallet.");
            return;
        }
        try {
            if (wallet.isEncrypted()) {
                KeyParameter aesKey = WalletTool.passwordToKey(true);
                if (aesKey == null) {
                    return;
                }
                key = key.encrypt((KeyCrypter)Preconditions.checkNotNull((Object)wallet.getKeyCrypter()), aesKey);
            }
            wallet.importKey(key);
            System.out.println(key.toAddress(params) + " " + key);
        }
        catch (KeyCrypterException kce) {
            System.err.println("There was an encryption related error when adding the key. The error was '" + kce.getMessage() + "'.");
        }
    }

    private static long getCreationTimeSeconds() {
        if (options.has(unixtimeFlag)) {
            return (Long)unixtimeFlag.value(options);
        }
        if (options.has(dateFlag)) {
            return ((Date)dateFlag.value(options)).getTime() / 1000L;
        }
        return 0L;
    }

    private static void deleteKey() {
        String pubkey = (String)options.valueOf("pubkey");
        String addr = (String)options.valueOf("addr");
        if (pubkey == null && addr == null) {
            System.err.println("One of --pubkey or --addr must be specified.");
            return;
        }
        ECKey key = null;
        if (pubkey != null) {
            key = wallet.findKeyFromPubKey(Hex.decode((String)pubkey));
        } else {
            try {
                Address address = Address.fromBase58((NetworkParameters)wallet.getParams(), (String)addr);
                key = wallet.findKeyFromPubHash(address.getHash160());
            }
            catch (AddressFormatException e) {
                System.err.println(addr + " does not parse as a Bitcoin address of the right network parameters.");
                return;
            }
        }
        if (key == null) {
            System.err.println("Wallet does not seem to contain that key.");
            return;
        }
        wallet.removeKey(key);
    }

    private static void currentReceiveAddr() {
        DeterministicKey key = wallet.currentReceiveKey();
        System.out.println(key.toAddress(params) + " " + key);
    }

    private static void dumpWallet() throws BlockStoreException {
        if (chainFileName.exists()) {
            WalletTool.setup();
        }
        System.out.println(wallet.toString(options.has("dump-privkeys"), true, true, chain));
    }

    private static void setCreationTime() {
        DeterministicSeed seed = wallet.getActiveKeyChain().getSeed();
        if (seed == null) {
            System.err.println("Active chain does not have a seed.");
            return;
        }
        long creationTime = WalletTool.getCreationTimeSeconds();
        if (creationTime > 0L) {
            System.out.println("Setting creation time to: " + Utils.dateTimeFormat((long)(creationTime * 1000L)));
        } else {
            System.out.println("Clearing creation time.");
        }
        seed.setCreationTimeSeconds(creationTime);
    }

    static synchronized void onChange(CountDownLatch latch) {
        WalletTool.saveWallet(walletFile);
        Coin balance = wallet.getBalance(Wallet.BalanceType.ESTIMATED);
        if (condition.matchBitcoins(balance)) {
            System.out.println(balance.toFriendlyString());
            latch.countDown();
        }
    }

    private static class WalletEventListener
    implements WalletChangeEventListener,
    WalletCoinsReceivedEventListener,
    WalletCoinsSentEventListener,
    WalletReorganizeEventListener {
        private final CountDownLatch latch;

        private WalletEventListener(CountDownLatch latch) {
            this.latch = latch;
        }

        public void onWalletChanged(Wallet wallet) {
            WalletTool.onChange(this.latch);
        }

        public void onCoinsReceived(Wallet wallet, Transaction tx, Coin prevBalance, Coin newBalance) {
            WalletTool.onChange(this.latch);
        }

        public void onCoinsSent(Wallet wallet, Transaction tx, Coin prevBalance, Coin newBalance) {
            WalletTool.onChange(this.latch);
        }

        public void onReorganize(Wallet wallet) {
            WalletTool.onChange(this.latch);
        }
    }

    static class OutputSpec {
        public final Coin value;
        public final Address addr;
        public final ECKey key;

        public OutputSpec(String spec) throws IllegalArgumentException {
            String[] parts = spec.split(":");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Malformed output specification, must have two parts separated by :");
            }
            String destination = parts[0];
            this.value = "ALL".equalsIgnoreCase(parts[1]) ? wallet.getBalance(Wallet.BalanceType.ESTIMATED) : Coin.parseCoin((String)parts[1]);
            if (destination.startsWith("0")) {
                byte[] pubKey = new BigInteger(destination, 16).toByteArray();
                this.key = ECKey.fromPublicOnly((byte[])pubKey);
                this.addr = null;
            } else {
                this.addr = Address.fromBase58((NetworkParameters)params, (String)destination);
                this.key = null;
            }
        }

        public boolean isAddress() {
            return this.addr != null;
        }
    }

    public static enum ValidationMode {
        FULL,
        SPV;

    }

    public static enum WaitForEnum {
        EVER,
        WALLET_TX,
        BLOCK,
        BALANCE;

    }

    public static enum ActionEnum {
        DUMP,
        RAW_DUMP,
        CREATE,
        ADD_KEY,
        ADD_ADDR,
        DELETE_KEY,
        CURRENT_RECEIVE_ADDR,
        SYNC,
        RESET,
        SEND,
        SEND_CLTVPAYMENTCHANNEL,
        SETTLE_CLTVPAYMENTCHANNEL,
        REFUND_CLTVPAYMENTCHANNEL,
        ENCRYPT,
        DECRYPT,
        MARRY,
        ROTATE,
        SET_CREATION_TIME;

    }

    public static class Condition {
        Type type;
        String value;

        public Condition(String from) {
            String s;
            if (from.length() < 2) {
                throw new RuntimeException("Condition string too short: " + from);
            }
            if (from.startsWith("<=")) {
                this.type = Type.LTE;
            } else if (from.startsWith(">=")) {
                this.type = Type.GTE;
            } else if (from.startsWith("<")) {
                this.type = Type.LT;
            } else if (from.startsWith("=")) {
                this.type = Type.EQUAL;
            } else if (from.startsWith(">")) {
                this.type = Type.GT;
            } else {
                throw new RuntimeException("Unknown operator in condition: " + from);
            }
            switch (this.type) {
                case LT: 
                case GT: 
                case EQUAL: {
                    s = from.substring(1);
                    break;
                }
                case LTE: 
                case GTE: {
                    s = from.substring(2);
                    break;
                }
                default: {
                    throw new RuntimeException("Unreachable");
                }
            }
            this.value = s;
        }

        public boolean matchBitcoins(Coin comparison) {
            try {
                Coin units = Coin.parseCoin((String)this.value);
                switch (this.type) {
                    case LT: {
                        return comparison.compareTo(units) < 0;
                    }
                    case GT: {
                        return comparison.compareTo(units) > 0;
                    }
                    case EQUAL: {
                        return comparison.compareTo(units) == 0;
                    }
                    case LTE: {
                        return comparison.compareTo(units) <= 0;
                    }
                    case GTE: {
                        return comparison.compareTo(units) >= 0;
                    }
                }
                throw new RuntimeException("Unreachable");
            }
            catch (NumberFormatException e) {
                System.err.println("Could not parse value from condition string: " + this.value);
                System.exit(1);
                return false;
            }
        }

        public static enum Type {
            EQUAL,
            LT,
            GT,
            LTE,
            GTE;

        }
    }
}

