/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.rabbit.batch.client.config;

import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.PropertySource;
import org.springframework.context.annotation.PropertySources;

import org.autumnframework.service.queue.api.QueueApiConfiguration;
import org.autumnframework.service.rabbit.batch.client.listener.BatchListenerComponent;
import org.autumnframework.service.rabbit.client.config.MQJsonConfig;
import org.autumnframework.service.rabbit.client.config.RabbitClientConfig;
import org.autumnframework.service.rabbit.client.listener.CrudMessageListenerAndDelegationService;

/**
 * 
 */
@Configuration
@EnableRabbit
@Import({QueueApiConfiguration.class, MQJsonConfig.class, RabbitClientConfig.class})
@PropertySources({@PropertySource("classpath:service.properties")})
public class RabbitBatchClientConfig{

    /**
     * @param rabbitConnectionFactory
     * @param serviceName
     * @param batchSize
     * @return
     */
    @Bean
    @ConditionalOnExpression("'${autumn.messaging.type:rabbitmq}' == 'rabbitmq' and ${autumn.messaging.batch-enabled:false}")
    public SimpleRabbitListenerContainerFactory consumerBatchContainerFactory(
            ConnectionFactory rabbitConnectionFactory,
            @Value("${autumn.service.name:no-service-name-set}") String serviceName,
            @Value("${autumn.messaging.batch-size:100}") Integer batchSize) {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(rabbitConnectionFactory);
        factory.setConsumerTagStrategy(q -> serviceName + "." + q);
        factory.setBatchListener(true);
        factory.setBatchSize(batchSize);
        factory.setDeBatchingEnabled(true);
        factory.setConsumerBatchEnabled(true);
        return factory;
    }
    
    
    /**
     * @param bean
     * @return
     */
    @Bean
    @ConditionalOnBean(name = "consumerBatchContainerFactory")
    public BatchListenerComponent batchListenerComponent(CrudMessageListenerAndDelegationService bean) {
        return new BatchListenerComponent(bean);
    }
}
