/*
 * Decompiled with CFR 0.152.
 */
package org.arvados.client.logic.keep;

import com.google.common.collect.Lists;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.arvados.client.api.client.CollectionsApiClient;
import org.arvados.client.api.client.KeepWebApiClient;
import org.arvados.client.api.model.Collection;
import org.arvados.client.exception.ArvadosClientException;
import org.arvados.client.logic.collection.FileToken;
import org.arvados.client.logic.collection.ManifestDecoder;
import org.arvados.client.logic.collection.ManifestStream;
import org.arvados.client.logic.keep.KeepClient;
import org.arvados.client.logic.keep.KeepLocator;
import org.arvados.client.logic.keep.exception.DownloadFolderAlreadyExistsException;
import org.arvados.client.logic.keep.exception.FileAlreadyExistsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileDownloader {
    private final KeepClient keepClient;
    private final ManifestDecoder manifestDecoder;
    private final CollectionsApiClient collectionsApiClient;
    private final KeepWebApiClient keepWebApiClient;
    private final Logger log = LoggerFactory.getLogger(FileDownloader.class);

    public FileDownloader(KeepClient keepClient, ManifestDecoder manifestDecoder, CollectionsApiClient collectionsApiClient, KeepWebApiClient keepWebApiClient) {
        this.keepClient = keepClient;
        this.manifestDecoder = manifestDecoder;
        this.collectionsApiClient = collectionsApiClient;
        this.keepWebApiClient = keepWebApiClient;
    }

    public List<FileToken> listFileInfoFromCollection(String collectionUuid) {
        Collection requestedCollection = (Collection)this.collectionsApiClient.get(collectionUuid);
        String manifestText = requestedCollection.getManifestText();
        return this.manifestDecoder.decode(manifestText).stream().flatMap(p -> p.getFileTokens().stream()).collect(Collectors.toList());
    }

    public File downloadSingleFileUsingKeepWeb(String filePathName, String collectionUuid, String pathToDownloadFolder) {
        FileToken fileToken = this.getFileTokenFromCollection(filePathName, collectionUuid);
        if (fileToken == null) {
            throw new ArvadosClientException(String.format("%s not found in Collection with UUID %s", filePathName, collectionUuid));
        }
        File downloadedFile = this.checkIfFileExistsInTargetLocation(fileToken, pathToDownloadFolder);
        try (FileOutputStream fos = new FileOutputStream(downloadedFile);){
            fos.write(this.keepWebApiClient.download(collectionUuid, filePathName));
        }
        catch (IOException e) {
            throw new ArvadosClientException(String.format("Unable to write down file %s", fileToken.getFileName()), e);
        }
        return downloadedFile;
    }

    public List<File> downloadFilesFromCollectionUsingKeepWeb(String collectionUuid, String pathToDownloadFolder) {
        String collectionTargetDir = this.setTargetDirectory(collectionUuid, pathToDownloadFolder).getAbsolutePath();
        List<FileToken> fileTokens = this.listFileInfoFromCollection(collectionUuid);
        ArrayList futures = Lists.newArrayList();
        for (FileToken fileToken : fileTokens) {
            futures.add(CompletableFuture.supplyAsync(() -> this.downloadOneFileFromCollectionUsingKeepWeb(fileToken, collectionUuid, collectionTargetDir)));
        }
        CompletableFuture[] array = futures.toArray(new CompletableFuture[0]);
        return Stream.of(array).map(CompletableFuture::join).collect(Collectors.toList());
    }

    private FileToken getFileTokenFromCollection(String filePathName, String collectionUuid) {
        return this.listFileInfoFromCollection(collectionUuid).stream().filter(p -> p.getFullPath().equals(filePathName)).findFirst().orElse(null);
    }

    private File checkIfFileExistsInTargetLocation(FileToken fileToken, String pathToDownloadFolder) {
        String fileName = fileToken.getFileName();
        File downloadFile = new File(pathToDownloadFolder + "/" + fileName);
        if (downloadFile.exists()) {
            throw new FileAlreadyExistsException(String.format("File %s exists in location %s", fileName, pathToDownloadFolder));
        }
        return downloadFile;
    }

    private File downloadOneFileFromCollectionUsingKeepWeb(FileToken fileToken, String collectionUuid, String pathToDownloadFolder) {
        String filePathName = fileToken.getPath() + fileToken.getFileName();
        File downloadedFile = new File(pathToDownloadFolder + "/" + filePathName);
        downloadedFile.getParentFile().mkdirs();
        try (FileOutputStream fos = new FileOutputStream(downloadedFile);){
            fos.write(this.keepWebApiClient.download(collectionUuid, filePathName));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return downloadedFile;
    }

    public List<File> downloadFilesFromCollection(String collectionUuid, String pathToDownloadFolder) {
        Collection requestedCollection = (Collection)this.collectionsApiClient.get(collectionUuid);
        String manifestText = requestedCollection.getManifestText();
        File collectionTargetDir = this.setTargetDirectory(collectionUuid, pathToDownloadFolder);
        List<ManifestStream> manifestStreams = this.manifestDecoder.decode(manifestText);
        ArrayList<File> downloadedFilesFromCollection = new ArrayList<File>();
        for (ManifestStream manifestStream : manifestStreams) {
            downloadedFilesFromCollection.addAll(this.downloadFilesFromSingleManifestStream(manifestStream, collectionTargetDir));
        }
        this.log.debug(String.format("Total of: %d files downloaded", downloadedFilesFromCollection.size()));
        return downloadedFilesFromCollection;
    }

    private File setTargetDirectory(String collectionUUID, String pathToDownloadFolder) {
        File collectionTargetDir = new File(pathToDownloadFolder + "/" + collectionUUID);
        if (collectionTargetDir.exists()) {
            throw new DownloadFolderAlreadyExistsException(String.format("Directory for collection UUID %s already exists", collectionUUID));
        }
        collectionTargetDir.mkdirs();
        return collectionTargetDir;
    }

    private List<File> downloadFilesFromSingleManifestStream(ManifestStream manifestStream, File collectionTargetDir) {
        ArrayList<File> downloadedFiles = new ArrayList<File>();
        List<KeepLocator> keepLocators = manifestStream.getKeepLocators();
        DownloadHelper downloadHelper = new DownloadHelper(keepLocators);
        for (FileToken fileToken : manifestStream.getFileTokens()) {
            File downloadedFile = new File(collectionTargetDir.getAbsolutePath() + "/" + fileToken.getFullPath());
            downloadedFile.getParentFile().mkdirs();
            try (FileOutputStream fos = new FileOutputStream(downloadedFile, true);){
                downloadHelper.setBytesToDownload(fileToken.getFileSize());
                do {
                    downloadHelper.requestNewDataChunk();
                    downloadHelper.writeDownFile(fos);
                } while (downloadHelper.getBytesToDownload() != 0L);
            }
            catch (IOException | ArvadosClientException e) {
                throw new ArvadosClientException(String.format("Unable to write down file %s", fileToken.getFileName()), e);
            }
            downloadedFiles.add(downloadedFile);
            this.log.debug(String.format("File %d / %d downloaded from manifest stream", manifestStream.getFileTokens().indexOf(fileToken) + 1, manifestStream.getFileTokens().size()));
        }
        return downloadedFiles;
    }

    private class DownloadHelper {
        int currentDataChunkNumber = -1;
        int bytesDownloadedFromChunk = 0;
        long bytesToDownload;
        byte[] currentDataChunk;
        boolean remainingDataInChunk = false;
        final List<KeepLocator> keepLocators;

        private DownloadHelper(List<KeepLocator> keepLocators) {
            this.keepLocators = keepLocators;
        }

        private long getBytesToDownload() {
            return this.bytesToDownload;
        }

        private void setBytesToDownload(long bytesToDownload) {
            this.bytesToDownload = bytesToDownload;
        }

        private void requestNewDataChunk() {
            if (!this.remainingDataInChunk) {
                ++this.currentDataChunkNumber;
                if (this.currentDataChunkNumber < this.keepLocators.size()) {
                    this.currentDataChunk = FileDownloader.this.keepClient.getDataChunk(this.keepLocators.get(this.currentDataChunkNumber));
                    FileDownloader.this.log.debug(String.format("%d of %d data chunks from manifest stream downloaded", this.currentDataChunkNumber + 1, this.keepLocators.size()));
                } else {
                    throw new ArvadosClientException("Data chunk required for download is missing.");
                }
            }
        }

        private void writeDownFile(FileOutputStream fos) throws IOException {
            if (this.bytesToDownload >= (long)(this.currentDataChunk.length - this.bytesDownloadedFromChunk)) {
                this.writeDownWholeDataChunk(fos);
            } else {
                this.writeDownDataChunkPartially(fos);
            }
        }

        private void writeDownWholeDataChunk(FileOutputStream fos) throws IOException {
            fos.write(this.currentDataChunk, this.bytesDownloadedFromChunk, this.currentDataChunk.length - this.bytesDownloadedFromChunk);
            this.bytesToDownload -= (long)(this.currentDataChunk.length - this.bytesDownloadedFromChunk);
            this.remainingDataInChunk = false;
            this.bytesDownloadedFromChunk = 0;
        }

        private void writeDownDataChunkPartially(FileOutputStream fos) throws IOException {
            fos.write(this.currentDataChunk, this.bytesDownloadedFromChunk, (int)this.bytesToDownload);
            this.bytesDownloadedFromChunk = (int)((long)this.bytesDownloadedFromChunk + this.bytesToDownload);
            this.remainingDataInChunk = true;
            this.bytesToDownload = 0L;
        }
    }
}

