package org.apache.tapestry;

import static org.easymock.EasyMock.reportMatcher;
import org.easymock.IArgumentMatcher;

/**
 * An EasyMock 2.0 argument matcher that captures a method argument value. This allows an object
 * created inside a test method to be interrogated after the method completes, even when the object
 * is not a return value, but is merely passed as a parameter to a mock object.
 *
 * @author Howard M. Lewis Ship
 * @param <T>
 *            the type of object to capture
 */
public final class Capturer<T> implements IArgumentMatcher
{
    private final Class<T> _matchType;

    private T _captured;

    /**
     * Creates a new Capturer for the given type. Because of Generics syntax, it is easier to use
     * the static method.
     */
    public Capturer(Class<T> matchType)
    {
        _matchType = matchType;
    }

    /**
     * Factory method that invokes the normal constructor in a way that keeps Java Generics happy.
     */
    public static <T> Capturer<T> newCapturer(Class<T> matchType)
    {
        return new Capturer<T>(matchType);
    }

    public void appendTo(StringBuffer buffer)
    {
        buffer.append(String.format("capture(%s)", _matchType.getName()));
    }

    public boolean matches(Object parameter)
    {
        boolean result = _matchType.isInstance(parameter);

        if (result)
            _captured = _matchType.cast(parameter);

        return result;
    }

    /** Returns the method argument value previously captured. */
    public T getCaptured()
    {
        return _captured;
    }

    /**
     * Useage (with static imports):
     * <p>
     * Capturer&lt;Type&gt; c = newCapturer(Type.class);
     * <p>
     * mock.someMethod(capture(c));
     * <p> . . .
     * <p>
     * c.getCaptured().getXXX()
     * <p>
     * The interrogation of the captured argument should occur after the test subject has invoked
     * the method on the mock; the best time for this is typically after invoking
     * {@link TestBase#verify()}.
     * <p>
     * Remember that when you use an argument matcher for one argument of a method invocation, you
     * must use argument matchers for <em>all</em> arguments of the method invocation.
     */
    public static <T> T capture(Capturer<T> capturer)
    {
        reportMatcher(capturer);

        return null;
    }

}
