/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.solr.client.solrj.request;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import org.apache.solr.client.solrj.JacksonContentWriter;
import org.apache.solr.client.solrj.JacksonParsingResponse;
import org.apache.solr.client.solrj.ResponseParser;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.impl.InputStreamResponseParser;
import org.apache.solr.common.params.ModifiableSolrParams;
import org.apache.solr.common.params.SolrParams;

// WARNING: This class is generated from a Mustache template; any intended
// changes should be made to the underlying template and not this file directly.

/**
 * Experimental SolrRequest's and SolrResponse's for fileStore, generated from an OAS.
 *
 * <p>See individual request and response classes for more detailed and relevant information.
 *
 * <p>All SolrRequest implementations rely on v2 APIs which may require a SolrClient configured to
 * use the '/api' path prefix, instead of '/solr'.
 *
 * @lucene.experimental
 */
public class FileStoreApi {

  public static class DeleteFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.SolrJerseyResponse> {
    public DeleteFileResponse() {
      super(org.apache.solr.client.api.model.SolrJerseyResponse.class);
    }
  }

  public static class DeleteFile extends SolrRequest<DeleteFileResponse> {
    private final String path;
    private Boolean localDelete;

    /**
     * Create a DeleteFile request object.
     *
     * @param path Path param - Path to a file or directory within the filestore
     */
    public DeleteFile(String path) {
      super(
          SolrRequest.METHOD.valueOf("DELETE"),
          "/cluster/files{path}".replace("{" + "path" + "}", path));

      this.path = path;
    }

    /**
     * @param localDelete Indicates whether the deletion should only be done on the receiving node.
     *     For internal use only
     */
    public void setLocalDelete(Boolean localDelete) {
      this.localDelete = localDelete;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      if (localDelete != null) {

        params.add("localDelete", localDelete.toString());
      }
      return params;
    }

    @Override
    protected DeleteFileResponse createResponse(SolrClient client) {
      return new DeleteFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class UploadFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.UploadToFileStoreResponse> {
    public UploadFileResponse() {
      super(org.apache.solr.client.api.model.UploadToFileStoreResponse.class);
    }
  }

  public static class UploadFile extends SolrRequest<UploadFileResponse> {
    private final Object requestBody;
    private final String filePath;
    private List<String> sig;

    /**
     * Create a UploadFile request object.
     *
     * @param filePath Path param - File store path
     */
    public UploadFile(String filePath, Object requestBody) {
      super(
          SolrRequest.METHOD.valueOf("PUT"),
          "/cluster/files{filePath}".replace("{" + "filePath" + "}", filePath));

      this.filePath = filePath;
      this.requestBody = requestBody;
      addHeader("Content-type", "application/json");
    }

    /**
     * @param sig Signature(s) for the file being uploaded
     */
    public void setSig(List<String> sig) {
      this.sig = sig;
    }

    @Override
    @SuppressWarnings("unchecked")
    public RequestWriter.ContentWriter getContentWriter(String expectedType) {
      if (requestBody instanceof String) {
        return new RequestWriter.StringPayloadContentWriter((String) requestBody, expectedType);
      } else if (requestBody instanceof InputStream) {
        final var inputStream = (InputStream) requestBody;
        return new RequestWriter.ContentWriter() {
          @Override
          public void write(OutputStream os) throws IOException {
            inputStream.transferTo(os);
          }

          @Override
          public String getContentType() {
            return expectedType;
          }
        };
      }
      return new JacksonContentWriter(expectedType, requestBody);
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      if (sig != null) {
        sig.stream().forEach(v -> params.add("sig", v));
      }
      return params;
    }

    @Override
    protected UploadFileResponse createResponse(SolrClient client) {
      return new UploadFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }
}
