/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.segment.store;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.segment.local.segment.store.FilePerIndexDirectory;
import org.apache.pinot.segment.local.segment.store.IndexKey;
import org.apache.pinot.segment.local.segment.store.SingleFileIndexDirectory;
import org.apache.pinot.segment.local.segment.store.StarTreeIndexReader;
import org.apache.pinot.segment.spi.creator.SegmentVersion;
import org.apache.pinot.segment.spi.index.IndexType;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.memory.PinotDataBuffer;
import org.apache.pinot.segment.spi.store.ColumnIndexDirectory;
import org.apache.pinot.segment.spi.store.SegmentDirectory;
import org.apache.pinot.segment.spi.store.SegmentDirectoryPaths;
import org.apache.pinot.spi.utils.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentLocalFSDirectory
extends SegmentDirectory {
    private static final Logger LOGGER = LoggerFactory.getLogger(SegmentLocalFSDirectory.class);
    private static final int PAGE_SIZE_BYTES = 4096;
    private static final long MAX_MMAP_PREFETCH_PAGES = 0x1900000L;
    private static final double PREFETCH_SLOWDOWN_PCT = 0.67;
    private static final AtomicLong PREFETCHED_PAGES = new AtomicLong(0L);
    private final File _indexDir;
    private final File _segmentDirectory;
    private final SegmentLock _segmentLock;
    private final ReadMode _readMode;
    private SegmentMetadataImpl _segmentMetadata;
    private ColumnIndexDirectory _columnIndexDirectory;
    private StarTreeIndexReader _starTreeIndexReader;
    private String _tier;

    public SegmentLocalFSDirectory(File directory) {
        this._indexDir = directory;
        this._segmentDirectory = null;
        this._segmentLock = new SegmentLock();
        this._readMode = null;
    }

    public SegmentLocalFSDirectory(File directory, ReadMode readMode) throws IOException {
        this(directory, new SegmentMetadataImpl(directory), readMode);
    }

    @VisibleForTesting
    public SegmentLocalFSDirectory(File directoryFile, SegmentMetadataImpl metadata, ReadMode readMode) {
        Preconditions.checkNotNull((Object)directoryFile);
        Preconditions.checkNotNull((Object)metadata);
        this._indexDir = directoryFile;
        this._segmentDirectory = this.getSegmentPath(directoryFile, metadata.getVersion());
        Preconditions.checkState((boolean)this._segmentDirectory.exists(), (Object)("Segment directory: " + directoryFile + " must exist"));
        this._segmentLock = new SegmentLock();
        this._segmentMetadata = metadata;
        this._readMode = readMode;
        try {
            this.load();
        }
        catch (IOException | ConfigurationException e) {
            LOGGER.error("Failed to load segment, error: ", e);
            throw new RuntimeException(e);
        }
    }

    public URI getIndexDir() {
        return this._indexDir.toURI();
    }

    public void copyTo(File dest) throws Exception {
        File src = this._indexDir;
        if (!src.exists()) {
            File parentDir = this._indexDir.getParentFile();
            src = new File(parentDir, this._indexDir.getName() + ".segment.bak");
        }
        if (src.exists() && !src.equals(dest)) {
            FileUtils.copyDirectory((File)src, (File)dest);
        }
    }

    @Nullable
    public String getTier() {
        return this._tier;
    }

    public void setTier(@Nullable String tier) {
        this._tier = tier;
    }

    public SegmentMetadataImpl getSegmentMetadata() {
        return this._segmentMetadata;
    }

    public void reloadMetadata() throws Exception {
        this._segmentMetadata = new SegmentMetadataImpl(this._indexDir);
        this._columnIndexDirectory.setSegmentMetadata(this._segmentMetadata);
    }

    private File getSegmentPath(File segmentDirectory, SegmentVersion segmentVersion) {
        if (segmentVersion == SegmentVersion.v1 || segmentVersion == SegmentVersion.v2) {
            return segmentDirectory;
        }
        if (segmentVersion == SegmentVersion.v3) {
            if (segmentDirectory.getAbsolutePath().endsWith("v3")) {
                return segmentDirectory;
            }
            File v3SubDir = new File(segmentDirectory, "v3");
            if (v3SubDir.exists()) {
                return v3SubDir;
            }
            return segmentDirectory;
        }
        throw new IllegalArgumentException("Unknown segment version: " + segmentVersion);
    }

    public Path getPath() {
        return this._segmentDirectory.toPath();
    }

    public long getDiskSizeBytes() {
        if (this._segmentDirectory.exists()) {
            try {
                return FileUtils.sizeOfDirectory((File)this._segmentDirectory.toPath().toFile());
            }
            catch (IllegalArgumentException e) {
                LOGGER.error("Failed to read disk size for directory: {}", (Object)this._segmentDirectory.getAbsolutePath());
                return -1L;
            }
        }
        if (!SegmentDirectoryPaths.isV3Directory((File)this._segmentDirectory)) {
            LOGGER.error("Segment directory: {} not found on disk and is not v3 format", (Object)this._segmentDirectory.getAbsolutePath());
            return -1L;
        }
        File[] files = this._segmentDirectory.getParentFile().listFiles();
        if (files == null) {
            LOGGER.warn("Empty list of files for path: {}, segmentDirectory: {}", (Object)this._segmentDirectory.getParentFile(), (Object)this._segmentDirectory);
            return -1L;
        }
        long size = 0L;
        for (File file : files) {
            if (!file.isFile()) continue;
            size += file.length();
        }
        return size;
    }

    public Set<String> getColumnsWithIndex(IndexType<?, ?, ?> type) {
        if (this._columnIndexDirectory == null) {
            return Collections.emptySet();
        }
        return this._columnIndexDirectory.getColumnsWithIndex(type);
    }

    public Reader createReader() throws IOException {
        if (this._segmentLock.tryReadLock()) {
            this.loadData();
            return new Reader();
        }
        return null;
    }

    public Writer createWriter() throws IOException {
        if (this._segmentLock.tryWriteLock()) {
            this.loadData();
            return new Writer();
        }
        return null;
    }

    public String toString() {
        return this._segmentDirectory.toString();
    }

    protected void load() throws IOException, ConfigurationException {
        this.loadData();
    }

    private synchronized void loadData() throws IOException {
        if (this._columnIndexDirectory != null) {
            return;
        }
        switch (this._segmentMetadata.getVersion()) {
            case v1: 
            case v2: {
                this._columnIndexDirectory = new FilePerIndexDirectory(this._segmentDirectory, this._segmentMetadata, this._readMode);
                break;
            }
            case v3: {
                try {
                    this._columnIndexDirectory = new SingleFileIndexDirectory(this._segmentDirectory, this._segmentMetadata, this._readMode);
                    break;
                }
                catch (ConfigurationException e) {
                    LOGGER.error("Failed to create columnar index directory", (Throwable)e);
                    throw new RuntimeException(e);
                }
            }
        }
        if (CollectionUtils.isNotEmpty((Collection)this._segmentMetadata.getStarTreeV2MetadataList())) {
            this._starTreeIndexReader = new StarTreeIndexReader(this._segmentDirectory, this._segmentMetadata, this._readMode);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws IOException {
        this._segmentLock.close();
        SegmentLocalFSDirectory segmentLocalFSDirectory = this;
        synchronized (segmentLocalFSDirectory) {
            if (this._columnIndexDirectory != null) {
                this._columnIndexDirectory.close();
                this._columnIndexDirectory = null;
            }
            if (this._starTreeIndexReader != null) {
                this._starTreeIndexReader.close();
                this._starTreeIndexReader = null;
            }
        }
    }

    private PinotDataBuffer getIndexForColumn(String column, IndexType<?, ?, ?> type) throws IOException {
        PinotDataBuffer buffer = this._columnIndexDirectory.getBuffer(column, type);
        if (this._readMode == ReadMode.mmap) {
            this.prefetchMmapData(buffer);
        }
        return buffer;
    }

    private void prefetchMmapData(PinotDataBuffer buffer) {
        if (PREFETCHED_PAGES.get() >= 0x1900000L) {
            return;
        }
        long prefetchSlowdownPageLimit = 0x10C0000L;
        if (PREFETCHED_PAGES.get() >= 0x10C0000L) {
            if (0L < buffer.size()) {
                buffer.getByte(0);
                PREFETCHED_PAGES.incrementAndGet();
            }
        } else {
            for (long pos = 0L; pos < buffer.size() && PREFETCHED_PAGES.get() < 0x10C0000L; pos += 4096L) {
                buffer.getByte(pos);
                PREFETCHED_PAGES.incrementAndGet();
            }
        }
    }

    class SegmentLock
    implements AutoCloseable {
        int _readers = 0;
        int _writers = 0;

        SegmentLock() {
        }

        synchronized boolean tryReadLock() {
            if (this._writers > 0) {
                return false;
            }
            ++this._readers;
            return true;
        }

        synchronized boolean tryWriteLock() {
            if (this._readers > 0 || this._writers > 0) {
                return false;
            }
            ++this._writers;
            return true;
        }

        synchronized void unlock() {
            if (this._writers > 0) {
                --this._writers;
            } else if (this._readers > 0) {
                --this._readers;
            }
        }

        @Override
        public void close() {
            this.unlock();
        }
    }

    public class Writer
    extends SegmentDirectory.Writer {
        public Writer() {
            super((SegmentDirectory)SegmentLocalFSDirectory.this);
        }

        public PinotDataBuffer newIndexFor(String columnName, IndexType<?, ?, ?> indexType, long sizeBytes) throws IOException {
            return this.getNewIndexBuffer(new IndexKey(columnName, indexType), sizeBytes);
        }

        public void removeIndex(String columnName, IndexType<?, ?, ?> indexType) {
            SegmentLocalFSDirectory.this._columnIndexDirectory.removeIndex(columnName, indexType);
        }

        private PinotDataBuffer getNewIndexBuffer(IndexKey key, long sizeBytes) throws IOException {
            return SegmentLocalFSDirectory.this._columnIndexDirectory.newBuffer(key._name, key._type, sizeBytes);
        }

        public void save() {
        }

        public String toString() {
            return SegmentLocalFSDirectory.this._segmentDirectory.toString();
        }

        public void close() throws IOException {
            SegmentLocalFSDirectory.this._segmentLock.unlock();
            if (SegmentLocalFSDirectory.this._columnIndexDirectory != null) {
                SegmentLocalFSDirectory.this._columnIndexDirectory.close();
                SegmentLocalFSDirectory.this._columnIndexDirectory = null;
            }
            if (SegmentLocalFSDirectory.this._starTreeIndexReader != null) {
                SegmentLocalFSDirectory.this._starTreeIndexReader.close();
                SegmentLocalFSDirectory.this._starTreeIndexReader = null;
            }
        }

        public PinotDataBuffer getIndexFor(String column, IndexType<?, ?, ?> type) throws IOException {
            return SegmentLocalFSDirectory.this.getIndexForColumn(column, type);
        }

        public boolean hasIndexFor(String column, IndexType<?, ?, ?> type) {
            return SegmentLocalFSDirectory.this._columnIndexDirectory.hasIndexFor(column, type);
        }
    }

    public class Reader
    extends SegmentDirectory.Reader {
        public Reader() {
            super((SegmentDirectory)SegmentLocalFSDirectory.this);
        }

        public PinotDataBuffer getIndexFor(String column, IndexType<?, ?, ?> type) throws IOException {
            return SegmentLocalFSDirectory.this.getIndexForColumn(column, type);
        }

        public boolean hasIndexFor(String column, IndexType<?, ?, ?> type) {
            return SegmentLocalFSDirectory.this._columnIndexDirectory.hasIndexFor(column, type);
        }

        public boolean hasStarTreeIndex() {
            return SegmentLocalFSDirectory.this._starTreeIndexReader != null;
        }

        public SegmentDirectory.Reader getStarTreeIndexReader(final int starTreeId) {
            return new SegmentDirectory.Reader(){

                public PinotDataBuffer getIndexFor(String column, IndexType<?, ?, ?> type) throws IOException {
                    return SegmentLocalFSDirectory.this._starTreeIndexReader.getBuffer(starTreeId, column, type);
                }

                public boolean hasIndexFor(String column, IndexType<?, ?, ?> type) {
                    return SegmentLocalFSDirectory.this._starTreeIndexReader.hasIndexFor(starTreeId, column, type);
                }

                public String toString() {
                    return SegmentLocalFSDirectory.this._starTreeIndexReader.toString() + " for " + starTreeId;
                }

                public void close() throws IOException {
                }
            };
        }

        public void close() {
            SegmentLocalFSDirectory.this._segmentLock.unlock();
        }

        public String toString() {
            return SegmentLocalFSDirectory.this._segmentDirectory.toString();
        }
    }
}

