/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.common.utils.tls;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import nl.altindag.ssl.SSLFactory;
import nl.altindag.ssl.keymanager.HotSwappableX509ExtendedKeyManager;
import nl.altindag.ssl.trustmanager.HotSwappableX509ExtendedTrustManager;
import nl.altindag.ssl.util.SSLFactoryUtils;
import org.apache.pinot.common.config.TlsConfig;
import org.apache.pinot.common.utils.tls.TlsUtils;
import org.apache.pinot.spi.utils.retry.RetryPolicies;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RenewableTlsUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(RenewableTlsUtils.class);
    private static final String FILE_SCHEME = "file";
    private static final int CERT_RELOAD_JOB_INTERVAL_IN_MINUTES = 1440;
    private static final int CERT_RELOAD_JOB_INITAL_DELAY_IN_MINUTES = 20;

    private RenewableTlsUtils() {
    }

    public static SSLFactory createSSLFactoryAndEnableAutoRenewalWhenUsingFileStores(TlsConfig tlsConfig) {
        return RenewableTlsUtils.createSSLFactoryAndEnableAutoRenewalWhenUsingFileStores(tlsConfig, (Supplier<Boolean>)((Supplier)() -> false));
    }

    public static SSLFactory createSSLFactoryAndEnableAutoRenewalWhenUsingFileStores(TlsConfig tlsConfig, Supplier<Boolean> insecureModeSupplier) {
        SSLFactory sslFactory = RenewableTlsUtils.createSSLFactory(tlsConfig, (Boolean)insecureModeSupplier.get());
        if (TlsUtils.isKeyOrTrustStorePathNullOrHasFileScheme(tlsConfig.getKeyStorePath()) && TlsUtils.isKeyOrTrustStorePathNullOrHasFileScheme(tlsConfig.getTrustStorePath())) {
            RenewableTlsUtils.enableAutoRenewalFromFileStoreForSSLFactory(sslFactory, tlsConfig, insecureModeSupplier);
        }
        return sslFactory;
    }

    private static SSLFactory createSSLFactory(TlsConfig tlsConfig, boolean insecureMode) {
        return RenewableTlsUtils.createSSLFactory(tlsConfig.getKeyStoreType(), tlsConfig.getKeyStorePath(), tlsConfig.getKeyStorePassword(), tlsConfig.getTrustStoreType(), tlsConfig.getTrustStorePath(), tlsConfig.getTrustStorePassword(), null, null, true, tlsConfig.isInsecure() || insecureMode);
    }

    static SSLFactory createSSLFactory(String keyStoreType, String keyStorePath, String keyStorePassword, String trustStoreType, String trustStorePath, String trustStorePassword, String sslContextProtocol, SecureRandom secureRandom, boolean keyAndTrustMaterialSwappable, boolean isInsecure) {
        try {
            SSLFactory.Builder sslFactoryBuilder = SSLFactory.builder();
            InputStream keyStoreStream = null;
            InputStream trustStoreStream = null;
            if (keyStorePath != null) {
                Preconditions.checkNotNull((Object)keyStorePassword, (Object)"key store password must not be null");
                keyStoreStream = TlsUtils.makeKeyOrTrustStoreUrl(keyStorePath).openStream();
                if (keyAndTrustMaterialSwappable) {
                    sslFactoryBuilder.withSwappableIdentityMaterial();
                }
                sslFactoryBuilder.withIdentityMaterial(keyStoreStream, keyStorePassword.toCharArray(), keyStoreType);
            }
            if (isInsecure) {
                if (keyAndTrustMaterialSwappable) {
                    sslFactoryBuilder.withSwappableTrustMaterial();
                }
                sslFactoryBuilder.withUnsafeTrustMaterial();
            } else if (trustStorePath != null) {
                Preconditions.checkNotNull((Object)trustStorePassword, (Object)"trust store password must not be null");
                trustStoreStream = TlsUtils.makeKeyOrTrustStoreUrl(trustStorePath).openStream();
                if (keyAndTrustMaterialSwappable) {
                    sslFactoryBuilder.withSwappableTrustMaterial();
                }
                sslFactoryBuilder.withTrustMaterial(trustStoreStream, trustStorePassword.toCharArray(), trustStoreType);
            }
            if (sslContextProtocol != null) {
                sslFactoryBuilder.withSslContextAlgorithm(sslContextProtocol);
            }
            if (secureRandom != null) {
                sslFactoryBuilder.withSecureRandom(secureRandom);
            }
            SSLFactory sslFactory = sslFactoryBuilder.build();
            if (keyStoreStream != null) {
                keyStoreStream.close();
            }
            if (trustStoreStream != null) {
                trustStoreStream.close();
            }
            LOGGER.info("Successfully created SSLFactory {} with key store {} and trust store {}. Key and trust material swappable: {}", new Object[]{sslFactory, keyStorePath, trustStorePath, keyAndTrustMaterialSwappable});
            return sslFactory;
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    @VisibleForTesting
    static void enableAutoRenewalFromFileStoreForSSLFactory(SSLFactory sslFactory, TlsConfig tlsConfig, Supplier<Boolean> insecureModeSupplier) {
        RenewableTlsUtils.enableAutoRenewalFromFileStoreForSSLFactory(sslFactory, tlsConfig.getKeyStoreType(), tlsConfig.getKeyStorePath(), tlsConfig.getKeyStorePassword(), tlsConfig.getTrustStoreType(), tlsConfig.getTrustStorePath(), tlsConfig.getTrustStorePassword(), null, null, (Supplier<Boolean>)((Supplier)() -> tlsConfig.isInsecure() || (Boolean)insecureModeSupplier.get() != false));
    }

    static void enableAutoRenewalFromFileStoreForSSLFactory(SSLFactory sslFactory, String keyStoreType, String keyStorePath, String keyStorePassword, String trustStoreType, String trustStorePath, String trustStorePassword, String sslContextProtocol, SecureRandom secureRandom, Supplier<Boolean> insecureModeSupplier) {
        try {
            URL trustStoreURL;
            URL keyStoreURL = keyStorePath == null ? null : TlsUtils.makeKeyOrTrustStoreUrl(keyStorePath);
            URL uRL = trustStoreURL = trustStorePath == null ? null : TlsUtils.makeKeyOrTrustStoreUrl(trustStorePath);
            if (keyStoreURL != null) {
                Preconditions.checkArgument((boolean)keyStoreURL.toURI().getScheme().startsWith(FILE_SCHEME), (Object)"key store path must be a local file path or null when SSL auto renew is enabled");
                Preconditions.checkArgument((sslFactory.getKeyManager().isPresent() && sslFactory.getKeyManager().get() instanceof HotSwappableX509ExtendedKeyManager ? 1 : 0) != 0, (Object)"key manager of the existing SSLFactory must be swappable");
            }
            if (trustStoreURL != null) {
                Preconditions.checkArgument((boolean)trustStoreURL.toURI().getScheme().startsWith(FILE_SCHEME), (Object)"trust store path must be a local file path or null when SSL auto renew is enabled");
                Preconditions.checkArgument((sslFactory.getTrustManager().isPresent() && sslFactory.getTrustManager().get() instanceof HotSwappableX509ExtendedTrustManager ? 1 : 0) != 0, (Object)"trust manager of the existing SSLFactory must be swappable");
            }
            Executors.newSingleThreadExecutor().execute(() -> {
                try {
                    RenewableTlsUtils.reloadSslFactoryWhenFileStoreChanges(sslFactory, keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, sslContextProtocol, secureRandom, insecureModeSupplier);
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            Executors.newSingleThreadScheduledExecutor().scheduleAtFixedRate(() -> {
                LOGGER.info("Creating a scheduled thread to reloadSsl once a day");
                try {
                    RenewableTlsUtils.reloadSslFactory(sslFactory, keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, sslContextProtocol, secureRandom, insecureModeSupplier);
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }, 20L, 1440L, TimeUnit.MINUTES);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    static void reloadSslFactoryWhenFileStoreChanges(SSLFactory baseSslFactory, String keyStoreType, String keyStorePath, String keyStorePassword, String trustStoreType, String trustStorePath, String trustStorePassword, String sslContextProtocol, SecureRandom secureRandom, Supplier<Boolean> insecureModeSupplier) throws IOException, URISyntaxException, InterruptedException {
        WatchKey key;
        LOGGER.info("Enable auto renewal of SSLFactory {} when key store {} or trust store {} changes", new Object[]{baseSslFactory, keyStorePath, trustStorePath});
        WatchService watchService = FileSystems.getDefault().newWatchService();
        HashMap<WatchKey, Set<Path>> watchKeyPathMap = new HashMap<WatchKey, Set<Path>>();
        RenewableTlsUtils.registerFile(watchService, watchKeyPathMap, keyStorePath);
        RenewableTlsUtils.registerFile(watchService, watchKeyPathMap, trustStorePath);
        while ((key = watchService.take()) != null) {
            for (WatchEvent<?> event : key.pollEvents()) {
                Path changedFile = (Path)event.context();
                if (!((Set)watchKeyPathMap.get(key)).contains(changedFile)) continue;
                LOGGER.info("Detected change in file: {}, try to renew SSLFactory {} (built from key store {} and truststore {})", new Object[]{changedFile, baseSslFactory, keyStorePath, trustStorePath});
                RenewableTlsUtils.reloadSslFactory(baseSslFactory, keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, sslContextProtocol, secureRandom, insecureModeSupplier);
            }
            key.reset();
        }
    }

    private static synchronized void reloadSslFactory(SSLFactory baseSslFactory, String keyStoreType, String keyStorePath, String keyStorePassword, String trustStoreType, String trustStorePath, String trustStorePassword, String sslContextProtocol, SecureRandom secureRandom, Supplier<Boolean> insecureModeSupplier) {
        LOGGER.info("reloadSslFactory :: Enable auto renewal of SSLFactory {}, key store {}, trust store {}", new Object[]{baseSslFactory, keyStorePath, trustStorePath});
        int maxSslFactoryReloadingAttempts = 3;
        int sslFactoryReloadingRetryDelayMs = 1000;
        try {
            RetryPolicies.fixedDelayRetryPolicy((int)maxSslFactoryReloadingAttempts, (long)sslFactoryReloadingRetryDelayMs).attempt(() -> {
                try {
                    SSLFactory updatedSslFactory = RenewableTlsUtils.createSSLFactory(keyStoreType, keyStorePath, keyStorePassword, trustStoreType, trustStorePath, trustStorePassword, sslContextProtocol, secureRandom, false, (Boolean)insecureModeSupplier.get());
                    SSLFactoryUtils.reload((SSLFactory)baseSslFactory, (SSLFactory)updatedSslFactory);
                    LOGGER.info("reloadSslFactory :: Successfully renewed SSLFactory {} (built from key store {} and truststore {}) on file", new Object[]{baseSslFactory, keyStorePath, trustStorePath});
                    return true;
                }
                catch (Exception e) {
                    LOGGER.info("reloadSslFactory :: Encountered issues when renewing SSLFactory {} (built from key store {} and truststore {}) on ", new Object[]{baseSslFactory, keyStorePath, trustStorePath, e});
                    return false;
                }
            });
        }
        catch (Exception e) {
            LOGGER.error("reloadSslFactory :: Failed to renew SSLFactory {} (built from key store {} and truststore {}) after {} retries", new Object[]{baseSslFactory, keyStorePath, trustStorePath, maxSslFactoryReloadingAttempts, e});
        }
    }

    @VisibleForTesting
    static void registerFile(WatchService watchService, Map<WatchKey, Set<Path>> keyPathMap, String filePath) throws IOException, URISyntaxException {
        if (filePath == null) {
            return;
        }
        Path path = Path.of(TlsUtils.makeKeyOrTrustStoreUrl(filePath).getPath(), new String[0]);
        WatchKey key = path.getParent().register(watchService, StandardWatchEventKinds.ENTRY_MODIFY);
        keyPathMap.computeIfAbsent(key, k -> new HashSet());
        keyPathMap.get(key).add(path.getFileName());
    }
}

