/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.common.utils;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.RateLimiter;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.CompressorException;
import org.apache.commons.compress.compressors.CompressorInputStream;
import org.apache.commons.compress.compressors.CompressorOutputStream;
import org.apache.commons.compress.compressors.CompressorStreamFactory;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

public class TarCompressionUtils {
    public static final long NO_DISK_WRITE_RATE_LIMIT = -1L;
    public static final long SYNC_DISK_WRITE_WITH_UPSTREAM_RATE = 0L;
    private static final int DEFAULT_BUFFER_SIZE = 0x400000;
    public static final String TAR_COMPRESSED_FILE_EXTENSION = ".tar.compressed";
    public static final String TAR_GZ_FILE_EXTENSION = ".tar.gz";
    public static final String TAR_LZ4_FILE_EXTENSION = ".tar.lz4";
    public static final String TAR_ZST_FILE_EXTENSION = ".tar.zst";
    public static final Map<String, String> COMPRESSOR_NAME_BY_FILE_EXTENSIONS = Map.of(".tar.gz", "gz", ".tar.lz4", "lz4-framed", ".tar.zst", "zstd");
    private static final CompressorStreamFactory COMPRESSOR_STREAM_FACTORY = CompressorStreamFactory.getSingleton();
    private static final char ENTRY_NAME_SEPARATOR = '/';
    private static String _defaultCompressorName = "gz";

    private TarCompressionUtils() {
    }

    public static void setDefaultCompressor(String compressorName) {
        if (COMPRESSOR_NAME_BY_FILE_EXTENSIONS.containsKey(compressorName)) {
            _defaultCompressorName = compressorName;
        }
    }

    public static void createCompressedTarFile(File inputFile, File outputFile) throws IOException {
        TarCompressionUtils.createCompressedTarFile(new File[]{inputFile}, outputFile);
    }

    public static void createCompressedTarFile(File[] inputFiles, File outputFile) throws IOException {
        if (outputFile.getName().endsWith(TAR_COMPRESSED_FILE_EXTENSION)) {
            TarCompressionUtils.createCompressedTarFile(inputFiles, outputFile, _defaultCompressorName);
        } else {
            String compressorName = null;
            for (String supportedCompressorExtension : COMPRESSOR_NAME_BY_FILE_EXTENSIONS.keySet()) {
                if (!outputFile.getName().endsWith(supportedCompressorExtension)) continue;
                compressorName = COMPRESSOR_NAME_BY_FILE_EXTENSIONS.get(supportedCompressorExtension);
                TarCompressionUtils.createCompressedTarFile(inputFiles, outputFile, compressorName);
                return;
            }
            Preconditions.checkState((null != compressorName ? 1 : 0) != 0, (String)"Output file: %s does not have a supported compressed tar file extension", (Object)outputFile);
        }
    }

    public static void createCompressedTarFile(File inputFile, File outputFile, String compressorName) throws IOException {
        TarCompressionUtils.createCompressedTarFile(new File[]{inputFile}, outputFile, compressorName);
    }

    public static void createCompressedTarFile(File[] inputFiles, File outputFile, String compressorName) throws IOException {
        try (OutputStream fileOut = Files.newOutputStream(outputFile.toPath(), new OpenOption[0]);
             BufferedOutputStream bufferedOut = new BufferedOutputStream(fileOut);
             CompressorOutputStream compressorOut = COMPRESSOR_STREAM_FACTORY.createCompressorOutputStream(compressorName, (OutputStream)bufferedOut);
             TarArchiveOutputStream tarOut = new TarArchiveOutputStream((OutputStream)compressorOut);){
            tarOut.setBigNumberMode(1);
            tarOut.setLongFileMode(2);
            for (File inputFile : inputFiles) {
                TarCompressionUtils.addFileToCompressedTar((ArchiveOutputStream)tarOut, inputFile, "");
            }
        }
        catch (CompressorException e) {
            throw new IOException(e);
        }
    }

    private static void addFileToCompressedTar(ArchiveOutputStream tarOut, File file, String baseEntryName) throws IOException {
        String entryName = baseEntryName + file.getName();
        TarArchiveEntry entry = new TarArchiveEntry(file, entryName);
        tarOut.putArchiveEntry((ArchiveEntry)entry);
        if (file.isFile()) {
            try (InputStream in = Files.newInputStream(file.toPath(), new OpenOption[0]);){
                IOUtils.copy((InputStream)in, (OutputStream)tarOut);
            }
            tarOut.closeArchiveEntry();
        } else {
            tarOut.closeArchiveEntry();
            File[] children = file.listFiles();
            assert (children != null);
            String baseEntryNameForChildren = entryName + "/";
            for (File child : children) {
                TarCompressionUtils.addFileToCompressedTar(tarOut, child, baseEntryNameForChildren);
            }
        }
    }

    public static List<File> untar(File inputFile, File outputDir) throws IOException {
        try (InputStream fileIn = Files.newInputStream(inputFile.toPath(), new OpenOption[0]);){
            List<File> list = TarCompressionUtils.untar(fileIn, outputDir);
            return list;
        }
    }

    public static List<File> untar(InputStream inputStream, File outputDir) throws IOException {
        return TarCompressionUtils.untarWithRateLimiter(inputStream, outputDir, -1L);
    }

    public static List<File> untarWithRateLimiter(InputStream inputStream, File outputDir, long maxStreamRateInByte) throws IOException {
        Object outputDirCanonicalPath = outputDir.getCanonicalPath();
        if (!((String)outputDirCanonicalPath).endsWith(File.separator)) {
            outputDirCanonicalPath = (String)outputDirCanonicalPath + File.separator;
        }
        ArrayList<File> untarredFiles = new ArrayList<File>();
        try (BufferedInputStream bufferedIn = new BufferedInputStream(inputStream);
             CompressorInputStream compressorIn = COMPRESSOR_STREAM_FACTORY.createCompressorInputStream((InputStream)bufferedIn);
             TarArchiveInputStream tarIn = new TarArchiveInputStream((InputStream)compressorIn);){
            ArchiveEntry entry;
            while ((entry = tarIn.getNextEntry()) != null) {
                String entryName = entry.getName();
                String[] parts = StringUtils.split((String)entryName, (char)'/');
                File outputFile = outputDir;
                for (String part : parts) {
                    outputFile = new File(outputFile, part);
                }
                if (entry.isDirectory()) {
                    if (!outputFile.getCanonicalPath().startsWith((String)outputDirCanonicalPath)) {
                        throw new IOException(String.format("Trying to create directory: %s outside of the output directory: %s", outputFile, outputDir));
                    }
                    if (!outputFile.isDirectory() && !outputFile.mkdirs()) {
                        throw new IOException(String.format("Failed to create directory: %s", outputFile));
                    }
                } else {
                    File parentFile = outputFile.getParentFile();
                    Object parentFileCanonicalPath = parentFile.getCanonicalPath();
                    if (!((String)parentFileCanonicalPath).endsWith(File.separator)) {
                        parentFileCanonicalPath = (String)parentFileCanonicalPath + File.separator;
                    }
                    if (!((String)parentFileCanonicalPath).startsWith((String)outputDirCanonicalPath)) {
                        throw new IOException(String.format("Trying to create directory: %s outside of the output directory: %s", parentFile, outputDir));
                    }
                    if (!parentFile.isDirectory() && !parentFile.mkdirs()) {
                        throw new IOException(String.format("Failed to create directory: %s", parentFile));
                    }
                    try (FileOutputStream out = new FileOutputStream(outputFile.toPath().toString());){
                        if (maxStreamRateInByte != -1L) {
                            TarCompressionUtils.copyWithRateLimiter((InputStream)tarIn, out, maxStreamRateInByte);
                        } else {
                            IOUtils.copy((InputStream)tarIn, (OutputStream)out);
                        }
                    }
                }
                untarredFiles.add(outputFile);
            }
        }
        catch (CompressorException e) {
            throw new IOException(e);
        }
        return untarredFiles;
    }

    public static void untarOneFile(File inputFile, String fileName, File outputFile) throws IOException {
        try (InputStream fileIn = Files.newInputStream(inputFile.toPath(), new OpenOption[0]);
             BufferedInputStream bufferedIn = new BufferedInputStream(fileIn);
             CompressorInputStream compressorIn = COMPRESSOR_STREAM_FACTORY.createCompressorInputStream((InputStream)bufferedIn);
             TarArchiveInputStream tarIn = new TarArchiveInputStream((InputStream)compressorIn);){
            ArchiveEntry entry;
            while ((entry = tarIn.getNextEntry()) != null) {
                if (entry.isDirectory()) continue;
                String entryName = entry.getName();
                String[] parts = StringUtils.split((String)entryName, (char)'/');
                if (parts.length <= 0 || !parts[parts.length - 1].equals(fileName)) continue;
                try (OutputStream out = Files.newOutputStream(outputFile.toPath(), new OpenOption[0]);){
                    IOUtils.copy((InputStream)tarIn, (OutputStream)out);
                }
                return;
            }
            throw new IOException(String.format("Failed to find file: %s in: %s", fileName, inputFile));
        }
        catch (CompressorException e) {
            throw new IOException(e);
        }
    }

    public static long copyWithRateLimiter(InputStream inputStream, FileOutputStream outputStream, long maxStreamRateInByte) throws IOException {
        long count;
        Preconditions.checkState((inputStream != null ? 1 : 0) != 0, (Object)"inputStream is null");
        Preconditions.checkState((outputStream != null ? 1 : 0) != 0, (Object)"outputStream is null");
        FileDescriptor fd = outputStream.getFD();
        byte[] buffer = new byte[0x400000];
        if (maxStreamRateInByte == 0L) {
            int n;
            count = 0L;
            while (-1 != (n = inputStream.read(buffer))) {
                outputStream.write(buffer, 0, n);
                fd.sync();
                count += (long)n;
            }
        } else {
            int n;
            RateLimiter rateLimiter = RateLimiter.create((double)maxStreamRateInByte);
            count = 0L;
            while (-1 != (n = inputStream.read(buffer))) {
                rateLimiter.acquire(n);
                outputStream.write(buffer, 0, n);
                fd.sync();
                count += (long)n;
            }
        }
        return count;
    }
}

