/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.PagedList;
import org.apache.paimon.Snapshot;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogLockContext;
import org.apache.paimon.catalog.CatalogLockFactory;
import org.apache.paimon.catalog.CatalogUtils;
import org.apache.paimon.catalog.Database;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.catalog.PropertyChange;
import org.apache.paimon.catalog.TableMetadata;
import org.apache.paimon.factories.FactoryUtil;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.FileStatus;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.Options;
import org.apache.paimon.partition.Partition;
import org.apache.paimon.partition.PartitionStatistics;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.Instant;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.TableSnapshot;
import org.apache.paimon.table.object.ObjectTable;
import org.apache.paimon.table.sink.BatchTableCommit;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.types.DataField;
import org.apache.paimon.types.RowType;
import org.apache.paimon.utils.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractCatalog
implements Catalog {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractCatalog.class);
    protected final FileIO fileIO;
    protected final Map<String, String> tableDefaultOptions;
    protected final Options catalogOptions;

    protected AbstractCatalog(FileIO fileIO) {
        this.fileIO = fileIO;
        this.tableDefaultOptions = new HashMap<String, String>();
        this.catalogOptions = new Options();
    }

    protected AbstractCatalog(FileIO fileIO, Options options) {
        this.fileIO = fileIO;
        this.tableDefaultOptions = CatalogUtils.tableDefaultOptions(options.toMap());
        this.catalogOptions = options;
    }

    @Override
    public Map<String, String> options() {
        return this.catalogOptions.toMap();
    }

    public abstract String warehouse();

    public FileIO fileIO() {
        return this.fileIO;
    }

    protected FileIO fileIO(Path path) {
        return this.fileIO;
    }

    public Optional<CatalogLockFactory> lockFactory() {
        if (!this.lockEnabled()) {
            return Optional.empty();
        }
        String lock = this.catalogOptions.get(CatalogOptions.LOCK_TYPE);
        if (lock == null) {
            return this.defaultLockFactory();
        }
        return Optional.of(FactoryUtil.discoverFactory(AbstractCatalog.class.getClassLoader(), CatalogLockFactory.class, lock));
    }

    public Optional<CatalogLockFactory> defaultLockFactory() {
        return Optional.empty();
    }

    public Optional<CatalogLockContext> lockContext() {
        return Optional.of(CatalogLockContext.fromOptions(this.catalogOptions));
    }

    protected boolean lockEnabled() {
        return this.catalogOptions.getOptional(CatalogOptions.LOCK_ENABLED).orElse(this.fileIO.isObjectStore());
    }

    protected boolean allowCustomTablePath() {
        return false;
    }

    @Override
    public PagedList<String> listDatabasesPaged(Integer maxResults, String pageToken) {
        return new PagedList<String>(this.listDatabases(), null);
    }

    @Override
    public void createDatabase(String name, boolean ignoreIfExists, Map<String, String> properties) throws Catalog.DatabaseAlreadyExistException {
        CatalogUtils.checkNotSystemDatabase(name);
        try {
            this.getDatabase(name);
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.DatabaseAlreadyExistException(name);
        }
        catch (Catalog.DatabaseNotExistException databaseNotExistException) {
            this.createDatabaseImpl(name, properties);
            return;
        }
    }

    @Override
    public Database getDatabase(String name) throws Catalog.DatabaseNotExistException {
        if (CatalogUtils.isSystemDatabase(name)) {
            return Database.of(name);
        }
        return this.getDatabaseImpl(name);
    }

    protected abstract Database getDatabaseImpl(String var1) throws Catalog.DatabaseNotExistException;

    @Override
    public void markDonePartitions(Identifier identifier, List<Map<String, String>> partitions) throws Catalog.TableNotExistException {
    }

    @Override
    public List<Partition> listPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        return CatalogUtils.listPartitionsFromFileSystem(this.getTable(identifier));
    }

    @Override
    public PagedList<Partition> listPartitionsPaged(Identifier identifier, Integer maxResults, String pageToken) throws Catalog.TableNotExistException {
        return new PagedList<Partition>(this.listPartitions(identifier), null);
    }

    protected abstract void createDatabaseImpl(String var1, Map<String, String> var2);

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        CatalogUtils.checkNotSystemDatabase(name);
        try {
            this.getDatabase(name);
        }
        catch (Catalog.DatabaseNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
        if (!cascade && !this.listTables(name).isEmpty()) {
            throw new Catalog.DatabaseNotEmptyException(name);
        }
        this.dropDatabaseImpl(name);
    }

    protected abstract void dropDatabaseImpl(String var1);

    @Override
    public void alterDatabase(String name, List<PropertyChange> changes, boolean ignoreIfNotExists) throws Catalog.DatabaseNotExistException {
        CatalogUtils.checkNotSystemDatabase(name);
        try {
            if (changes == null || changes.isEmpty()) {
                return;
            }
            this.alterDatabaseImpl(name, changes);
        }
        catch (Catalog.DatabaseNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
    }

    protected abstract void alterDatabaseImpl(String var1, List<PropertyChange> var2) throws Catalog.DatabaseNotExistException;

    @Override
    public List<String> listTables(String databaseName) throws Catalog.DatabaseNotExistException {
        if (CatalogUtils.isSystemDatabase(databaseName)) {
            return SystemTableLoader.loadGlobalTableNames();
        }
        this.getDatabase(databaseName);
        return this.listTablesImpl(databaseName).stream().sorted().collect(Collectors.toList());
    }

    @Override
    public PagedList<String> listTablesPaged(String databaseName, Integer maxResults, String pageToken) throws Catalog.DatabaseNotExistException {
        return new PagedList<String>(this.listTables(databaseName), null);
    }

    @Override
    public PagedList<Table> listTableDetailsPaged(String databaseName, Integer maxResults, String pageToken) throws Catalog.DatabaseNotExistException {
        if (CatalogUtils.isSystemDatabase(databaseName)) {
            List systemTables = SystemTableLoader.loadGlobalTableNames().stream().map(tableName -> {
                try {
                    return this.getTable(Identifier.create(databaseName, tableName));
                }
                catch (Catalog.TableNotExistException ignored) {
                    LOG.warn("system table {}.{} does not exist", (Object)databaseName, tableName);
                    return null;
                }
            }).filter(Objects::nonNull).collect(Collectors.toList());
            return new PagedList<Table>(systemTables, null);
        }
        this.getDatabase(databaseName);
        return this.listTableDetailsPagedImpl(databaseName, maxResults, pageToken);
    }

    protected abstract List<String> listTablesImpl(String var1);

    protected PagedList<Table> listTableDetailsPagedImpl(String databaseName, Integer maxResults, String pageToken) throws Catalog.DatabaseNotExistException {
        PagedList<String> pagedTableNames = this.listTablesPaged(databaseName, maxResults, pageToken);
        return new PagedList<Table>(pagedTableNames.getElements().stream().map(tableName -> {
            try {
                return this.getTable(Identifier.create(databaseName, tableName));
            }
            catch (Catalog.TableNotExistException ignored) {
                LOG.warn("table {}.{} does not exist", (Object)databaseName, tableName);
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList()), pagedTableNames.getNextPageToken());
    }

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        CatalogUtils.checkNotBranch(identifier, "dropTable");
        CatalogUtils.checkNotSystemTable(identifier, "dropTable");
        List<Path> externalPaths = new ArrayList<Path>();
        try {
            Table table = this.getTable(identifier);
            if (table instanceof FileStoreTable) {
                FileStoreTable fileStoreTable = (FileStoreTable)table;
                externalPaths = fileStoreTable.store().pathFactory().getExternalPaths();
            }
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.dropTableImpl(identifier, externalPaths);
    }

    protected abstract void dropTableImpl(Identifier var1, List<Path> var2);

    @Override
    public void createTable(Identifier identifier, Schema schema, boolean ignoreIfExists) throws Catalog.TableAlreadyExistException, Catalog.DatabaseNotExistException {
        CatalogUtils.checkNotBranch(identifier, "createTable");
        CatalogUtils.checkNotSystemTable(identifier, "createTable");
        CatalogUtils.validateAutoCreateClose(schema.options());
        this.validateCustomTablePath(schema.options());
        this.getDatabase(identifier.getDatabaseName());
        try {
            this.getTable(identifier);
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.TableAlreadyExistException(identifier);
        }
        catch (Catalog.TableNotExistException tableNotExistException) {
            this.copyTableDefaultOptions(schema.options());
            switch (Options.fromMap(schema.options()).get(CoreOptions.TYPE)) {
                case TABLE: 
                case MATERIALIZED_TABLE: {
                    this.createTableImpl(identifier, schema);
                    break;
                }
                case OBJECT_TABLE: {
                    this.createObjectTable(identifier, schema);
                    break;
                }
                case FORMAT_TABLE: {
                    this.createFormatTable(identifier, schema);
                }
            }
            return;
        }
    }

    private void createObjectTable(Identifier identifier, Schema schema) {
        RowType rowType = schema.rowType();
        Preconditions.checkArgument(rowType.getFields().isEmpty() || new HashSet<DataField>(ObjectTable.SCHEMA.getFields()).containsAll(rowType.getFields()), "Schema of Object Table can be empty or %s, but is %s.", ObjectTable.SCHEMA, rowType);
        Preconditions.checkArgument(schema.options().containsKey(CoreOptions.OBJECT_LOCATION.key()), "Object table should have object-location option.");
        this.createTableImpl(identifier, schema.copy(ObjectTable.SCHEMA));
    }

    protected abstract void createTableImpl(Identifier var1, Schema var2);

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        CatalogUtils.checkNotBranch(fromTable, "renameTable");
        CatalogUtils.checkNotBranch(toTable, "renameTable");
        CatalogUtils.checkNotSystemTable(fromTable, "renameTable");
        CatalogUtils.checkNotSystemTable(toTable, "renameTable");
        try {
            this.getTable(fromTable);
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(fromTable);
        }
        try {
            this.getTable(toTable);
            throw new Catalog.TableAlreadyExistException(toTable);
        }
        catch (Catalog.TableNotExistException tableNotExistException) {
            this.renameTableImpl(fromTable, toTable);
            return;
        }
    }

    protected abstract void renameTableImpl(Identifier var1, Identifier var2);

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        CatalogUtils.checkNotSystemTable(identifier, "alterTable");
        try {
            this.getTable(identifier);
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.alterTableImpl(identifier, changes);
    }

    protected abstract void alterTableImpl(Identifier var1, List<SchemaChange> var2) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException;

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        return CatalogUtils.loadTable(this, identifier, p -> this.fileIO(), this::fileIO, this::loadTableMetadata, this.lockFactory().orElse(null), this.lockContext().orElse(null));
    }

    @Override
    public void createBranch(Identifier identifier, String branch, @Nullable String fromTag) throws Catalog.TableNotExistException, Catalog.BranchAlreadyExistException, Catalog.TagNotExistException {
        throw new UnsupportedOperationException();
    }

    @Override
    public void dropBranch(Identifier identifier, String branch) throws Catalog.BranchNotExistException {
        throw new UnsupportedOperationException();
    }

    @Override
    public void fastForward(Identifier identifier, String branch) throws Catalog.BranchNotExistException {
        throw new UnsupportedOperationException();
    }

    @Override
    public List<String> listBranches(Identifier identifier) throws Catalog.TableNotExistException {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean commitSnapshot(Identifier identifier, Snapshot snapshot, List<PartitionStatistics> statistics) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Optional<TableSnapshot> loadSnapshot(Identifier identifier) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void rollbackTo(Identifier identifier, Instant instant) throws Catalog.TableNotExistException {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean supportsListObjectsPaged() {
        return false;
    }

    @Override
    public boolean supportsVersionManagement() {
        return false;
    }

    @Override
    public void createPartitions(Identifier identifier, List<Map<String, String>> partitions) throws Catalog.TableNotExistException {
    }

    @Override
    public void dropPartitions(Identifier identifier, List<Map<String, String>> partitions) throws Catalog.TableNotExistException {
        Table table = this.getTable(identifier);
        try (BatchTableCommit commit = table.newBatchWriteBuilder().newCommit();){
            commit.truncatePartitions(partitions);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void alterPartitions(Identifier identifier, List<PartitionStatistics> partitions) throws Catalog.TableNotExistException {
    }

    public void createFormatTable(Identifier identifier, Schema schema) {
        throw new UnsupportedOperationException(this.getClass().getName() + " currently does not support format table");
    }

    public Path newDatabasePath(String database) {
        return AbstractCatalog.newDatabasePath(this.warehouse(), database);
    }

    protected TableMetadata loadTableMetadata(Identifier identifier) throws Catalog.TableNotExistException {
        return new TableMetadata(this.loadTableSchema(identifier), false, null);
    }

    protected abstract TableSchema loadTableSchema(Identifier var1) throws Catalog.TableNotExistException;

    public Path getTableLocation(Identifier identifier) {
        return new Path(this.newDatabasePath(identifier.getDatabaseName()), identifier.getTableName());
    }

    protected void assertMainBranch(Identifier identifier) {
        if (identifier.getBranchName() != null && !"main".equals(identifier.getBranchName())) {
            throw new UnsupportedOperationException(this.getClass().getName() + " currently does not support table branches");
        }
    }

    public static Path newTableLocation(String warehouse, Identifier identifier) {
        CatalogUtils.checkNotBranch(identifier, "newTableLocation");
        CatalogUtils.checkNotSystemTable(identifier, "newTableLocation");
        return new Path(AbstractCatalog.newDatabasePath(warehouse, identifier.getDatabaseName()), identifier.getTableName());
    }

    public static Path newDatabasePath(String warehouse, String database) {
        return new Path(warehouse, database + ".db");
    }

    private void copyTableDefaultOptions(Map<String, String> options) {
        this.tableDefaultOptions.forEach(options::putIfAbsent);
    }

    private void validateCustomTablePath(Map<String, String> options) {
        if (!this.allowCustomTablePath() && options.containsKey(CoreOptions.PATH.key())) {
            throw new UnsupportedOperationException(String.format("The current catalog %s does not support specifying the table path when creating a table.", this.getClass().getSimpleName()));
        }
    }

    protected List<String> listDatabasesInFileSystem(Path warehouse) throws IOException {
        ArrayList<String> databases = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(warehouse)) {
            Path path = status.getPath();
            if (!status.isDir() || !path.getName().endsWith(".db")) continue;
            String fileName = path.getName();
            databases.add(fileName.substring(0, fileName.length() - ".db".length()));
        }
        return databases;
    }

    protected List<String> listTablesInFileSystem(Path databasePath) throws IOException {
        ArrayList<String> tables = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(databasePath)) {
            if (!status.isDir() || !this.tableExistsInFileSystem(status.getPath(), "main")) continue;
            tables.add(status.getPath().getName());
        }
        return tables;
    }

    protected boolean tableExistsInFileSystem(Path tablePath, String branchName) {
        SchemaManager schemaManager = new SchemaManager(this.fileIO, tablePath, branchName);
        boolean schemaZeroExists = schemaManager.schemaExists(0L);
        if (schemaZeroExists) {
            return true;
        }
        return !schemaManager.listAllIds().isEmpty();
    }

    public Optional<TableSchema> tableSchemaInFileSystem(Path tablePath, String branchName) {
        Optional<TableSchema> schema = new SchemaManager(this.fileIO, tablePath, branchName).latest();
        if (!"main".equals(branchName)) {
            schema = schema.map(s -> {
                Options branchOptions = new Options(s.options());
                branchOptions.set(CoreOptions.BRANCH, branchName);
                return s.copy(branchOptions.toMap());
            });
        }
        schema.ifPresent(s -> s.options().put(CoreOptions.PATH.key(), tablePath.toString()));
        return schema;
    }
}

