/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.tools.ci.licensecheck;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JarFileChecker {
    private static final Logger LOG = LoggerFactory.getLogger(JarFileChecker.class);

    public static int checkPath(Path path) throws Exception {
        List<Path> files = JarFileChecker.getBuildJars(path);
        LOG.info("Checking directory {} with a total of {} jar files.", (Object)path, (Object)files.size());
        int severeIssues = 0;
        for (Path file : files) {
            severeIssues += JarFileChecker.checkJar(file);
        }
        return severeIssues;
    }

    private static List<Path> getBuildJars(Path path) throws IOException {
        return Files.walk(path, new FileVisitOption[0]).filter(file -> file.toString().endsWith(".jar")).collect(Collectors.toList());
    }

    static int checkJar(Path file) throws Exception {
        URI uri = file.toUri();
        int numSevereIssues = 0;
        try (FileSystem fileSystem = FileSystems.newFileSystem(new URI("jar:file", uri.getHost(), uri.getPath(), uri.getFragment()), Collections.emptyMap());){
            if (JarFileChecker.isTestJarAndEmpty(file, fileSystem.getPath("/", new String[0]))) {
                int n = 0;
                return n;
            }
            if (!JarFileChecker.noticeFileExistsAndIsValid(fileSystem.getPath("META-INF", "NOTICE"), file)) {
                ++numSevereIssues;
            }
            if (!JarFileChecker.licenseFileExistsAndIsValid(fileSystem.getPath("META-INF", "LICENSE"), file)) {
                ++numSevereIssues;
            }
            numSevereIssues += JarFileChecker.getNumLicenseFilesOutsideMetaInfDirectory(file, fileSystem.getPath("/", new String[0]));
        }
        return numSevereIssues += JarFileChecker.getFilesWithIncompatibleLicenses(file, fileSystem.getPath("/", new String[0]));
    }

    private static boolean isTestJarAndEmpty(Path jar, Path jarRoot) throws IOException {
        if (jar.getFileName().toString().endsWith("-tests.jar")) {
            try (Stream<Path> files = Files.walk(jarRoot, new FileVisitOption[0]);){
                long numClassFiles = files.filter(path -> !path.equals(jarRoot)).filter(path -> path.getFileName().toString().endsWith(".class")).count();
                if (numClassFiles == 0L) {
                    boolean bl = true;
                    return bl;
                }
            }
        }
        return false;
    }

    private static boolean noticeFileExistsAndIsValid(Path noticeFile, Path jar) throws IOException {
        if (!Files.exists(noticeFile, new LinkOption[0])) {
            LOG.error("Missing META-INF/NOTICE in {}", (Object)jar);
            return false;
        }
        String noticeFileContents = JarFileChecker.readFile(noticeFile);
        if (!noticeFileContents.contains("The Apache Software Foundation")) {
            LOG.error("The notice file in {} does not contain the expected entries.", (Object)jar);
            return false;
        }
        return true;
    }

    private static boolean licenseFileExistsAndIsValid(Path licenseFile, Path jar) throws IOException {
        if (!Files.exists(licenseFile, new LinkOption[0])) {
            LOG.error("Missing META-INF/LICENSE in {}", (Object)jar);
            return false;
        }
        String licenseFileContents = JarFileChecker.readFile(licenseFile);
        if (!licenseFileContents.contains("Apache License") || !licenseFileContents.contains("Version 2.0, January 2004")) {
            LOG.error("The license file in {} does not contain the expected entries.", (Object)jar);
            return false;
        }
        return true;
    }

    private static int getFilesWithIncompatibleLicenses(Path jar, Path jarRoot) throws IOException {
        return JarFileChecker.findNonBinaryFilesContainingText(jar, jarRoot, JarFileChecker.asPatterns("Binary Code License", "Intel Simplified Software License", "JSR 275", "Microsoft Limited Public License", "Amazon Software License", "as necessary for your use of Satori services", "REDIS SOURCE AVAILABLE LICENSE", "Booz Allen Public License", "Confluent Community License Agreement Version 1.0", "the License does not grant to you, the right to Sell the Software.", "Sun Community Source License Version 3.0", "GNU General Public License", "GNU Affero General Public License", "GNU Lesser General Public License", "Q Public License", "Sleepycat License", "Server Side Public License", "Code Project Open License", " All advertising materials mentioning features or use of this software must display the following acknowledgement", "The license granted hereunder will terminate, automatically and without notice, for anyone that makes any claim", "The license granted hereunder will terminate, automatically and without notice, if you (or any of your subsidiaries, corporate affiliates or agents) initiate directly or indirectly, or take a direct financial interest in, any Patent Assertion: (i) against Facebook", "Netscape Public License", "SOLIPSISTIC ECLIPSE PUBLIC LICENSE", "Do whatever you like with the original work, just don't be a dick.", "The Software shall be used for Good, not Evil.", "Don\u2019t be evil"));
    }

    private static Collection<Pattern> asPatterns(String ... texts) {
        return Stream.of(texts).map(JarFileChecker::asPatternWithPotentialLineBreaks).collect(Collectors.toList());
    }

    private static Pattern asPatternWithPotentialLineBreaks(String text) {
        return Pattern.compile(text.toLowerCase(Locale.ROOT).replaceAll(" ", " ?\\\\R?[\\\\s/#]*"));
    }

    private static int findNonBinaryFilesContainingText(Path jar, Path jarRoot, Collection<Pattern> forbidden) throws IOException {
        try (Stream<Path> files = Files.walk(jarRoot, new FileVisitOption[0]);){
            int n = files.filter(path -> !path.equals(jarRoot)).filter(path -> !Files.isDirectory(path, new LinkOption[0])).filter(JarFileChecker::isNoClassFile).filter(path -> !JarFileChecker.getFileName(path).equals("dependencies")).filter(path -> !JarFileChecker.getFileName(path).startsWith("license")).filter(path -> !JarFileChecker.getFileName(path).startsWith("notice")).filter(path -> !path.toString().contains("/META-INF/versions/11/javax/xml/bind")).filter(path -> !JarFileChecker.isJavaxManifest(jar, path)).filter(path -> !JarFileChecker.pathStartsWith(path, "/org/glassfish/jersey/internal")).map(path -> {
                try {
                    String fileContents;
                    try {
                        fileContents = JarFileChecker.readFile(path).toLowerCase(Locale.ROOT);
                    }
                    catch (MalformedInputException mie) {
                        return 0;
                    }
                    int violations = 0;
                    for (Pattern text : forbidden) {
                        if (!text.matcher(fileContents).find()) continue;
                        violations = 1;
                        LOG.error("File '{}' in jar '{}' contains match with forbidden regex '{}'.", new Object[]{path, jar, text});
                    }
                    return violations;
                }
                catch (IOException e) {
                    throw new RuntimeException(String.format("Could not read contents of file '%s' in jar '%s'.", path, jar), e);
                }
            }).reduce(Integer::sum).orElse(0);
            return n;
        }
    }

    private static int getNumLicenseFilesOutsideMetaInfDirectory(Path jar, Path jarRoot) throws IOException {
        try (Stream<Path> files = Files.walk(jarRoot, new FileVisitOption[0]);){
            List filesWithIssues = files.filter(path -> !path.equals(jarRoot)).filter(path -> JarFileChecker.getFileName(path).contains("license") || JarFileChecker.getFileName(path).contains("notice")).filter(path -> !Files.isDirectory(path, new LinkOption[0])).filter(JarFileChecker::isNoClassFile).filter(path -> !JarFileChecker.getFileName(path).endsWith(".ftl")).map(Path::toString).filter(path -> !path.contains("META-INF")).filter(path -> !path.endsWith("web/3rdpartylicenses.txt")).collect(Collectors.toList());
            for (String fileWithIssue : filesWithIssues) {
                LOG.error("Jar file {} contains a LICENSE file in an unexpected location: {}", (Object)jar, (Object)fileWithIssue);
            }
            int n = filesWithIssues.size();
            return n;
        }
    }

    private static String getFileName(Path path) {
        return path.getFileName().toString().toLowerCase();
    }

    private static boolean pathStartsWith(Path file, String path) {
        return file.startsWith(file.getFileSystem().getPath(path, new String[0]));
    }

    private static boolean equals(Path file, String path) {
        return file.equals(file.getFileSystem().getPath(path, new String[0]));
    }

    private static boolean isNoClassFile(Path file) {
        return !JarFileChecker.getFileName(file).endsWith(".class");
    }

    private static boolean isJavaxManifest(Path jar, Path potentialManifestFile) {
        try {
            return JarFileChecker.equals(potentialManifestFile, "/META-INF/versions/11/META-INF/MANIFEST.MF") && JarFileChecker.readFile(potentialManifestFile).contains("Specification-Title: jaxb-api");
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Error while reading file %s from jar %s.", potentialManifestFile, jar), e);
        }
    }

    private static String readFile(Path file) throws IOException {
        return new String(Files.readAllBytes(file), StandardCharsets.UTF_8);
    }
}

