/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.lucene.analysis.no;


import java.util.Map;

import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.no.NorwegianMinimalStemFilter;
import org.apache.lucene.analysis.util.TokenFilterFactory;

import static org.apache.lucene.analysis.no.NorwegianLightStemmer.BOKMAAL;
import static org.apache.lucene.analysis.no.NorwegianLightStemmer.NYNORSK;

/** 
 * Factory for {@link NorwegianMinimalStemFilter}.
 * <pre class="prettyprint">
 * &lt;fieldType name="text_svlgtstem" class="solr.TextField" positionIncrementGap="100"&gt;
 *   &lt;analyzer&gt;
 *     &lt;tokenizer class="solr.StandardTokenizerFactory"/&gt;
 *     &lt;filter class="solr.LowerCaseFilterFactory"/&gt;
 *     &lt;filter class="solr.NorwegianMinimalStemFilterFactory" variant="nb"/&gt;
 *   &lt;/analyzer&gt;
 * &lt;/fieldType&gt;</pre>
 */
public class NorwegianMinimalStemFilterFactory extends TokenFilterFactory {
  
  private final int flags;
  
  /** Creates a new NorwegianMinimalStemFilterFactory */
  public NorwegianMinimalStemFilterFactory(Map<String,String> args) {
    super(args);
    String variant = get(args, "variant");
    if (variant == null || "nb".equals(variant)) {
      flags = BOKMAAL;
    } else if ("nn".equals(variant)) {
      flags = NYNORSK;
    } else if ("no".equals(variant)) {
      flags = BOKMAAL | NYNORSK;
    } else {
      throw new IllegalArgumentException("invalid variant: " + variant);
    }
    if (!args.isEmpty()) {
      throw new IllegalArgumentException("Unknown parameters: " + args);
    }
  }
  
  @Override
  public TokenStream create(TokenStream input) {
    return new NorwegianMinimalStemFilter(input, flags);
  }
}
