/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients;

import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.MetadataCache;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.internals.ClusterResourceListeners;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.MetadataResponse;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.test.MockClusterResourceListener;
import org.apache.kafka.test.TestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;

public class MetadataTest {
    private long refreshBackoffMs = 100L;
    private long metadataExpireMs = 1000L;
    private Metadata metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true);
    private AtomicReference<Exception> backgroundError = new AtomicReference();

    @After
    public void tearDown() {
        Assert.assertNull((String)("Exception in background thread : " + this.backgroundError.get()), (Object)this.backgroundError.get());
    }

    private static MetadataResponse emptyMetadataResponse() {
        return new MetadataResponse(Collections.emptyList(), null, -1, Collections.emptyList());
    }

    @Test
    public void testMetadata() throws Exception {
        long time = 0L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        this.metadata.requestUpdate();
        Assert.assertFalse((String)"Still no updated needed due to backoff", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed now that backoff time expired", (this.metadata.timeToNextUpdate(time += this.refreshBackoffMs) == 0L ? 1 : 0) != 0);
        String topic = "my-topic";
        Thread t1 = this.asyncFetch(topic, 500L);
        Thread t2 = this.asyncFetch(topic, 500L);
        Assert.assertTrue((String)"Awaiting update", (boolean)t1.isAlive());
        Assert.assertTrue((String)"Awaiting update", (boolean)t2.isAlive());
        while (t1.isAlive() || t2.isAlive()) {
            if (this.metadata.timeToNextUpdate(time) == 0L) {
                MetadataResponse response = TestUtils.metadataUpdateWith(1, Collections.singletonMap(topic, 1));
                this.metadata.update(response, time);
                time += this.refreshBackoffMs;
            }
            Thread.sleep(1L);
        }
        t1.join();
        t2.join();
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed due to stale metadata.", (this.metadata.timeToNextUpdate(time += this.metadataExpireMs) == 0L ? 1 : 0) != 0);
    }

    @Test
    public void testMetadataAwaitAfterClose() throws InterruptedException {
        long time = 0L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        this.metadata.requestUpdate();
        Assert.assertFalse((String)"Still no updated needed due to backoff", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed now that backoff time expired", (this.metadata.timeToNextUpdate(time += this.refreshBackoffMs) == 0L ? 1 : 0) != 0);
        String topic = "my-topic";
        this.metadata.close();
        Thread t1 = this.asyncFetch(topic, 500L);
        t1.join();
        Assert.assertTrue((this.backgroundError.get().getClass() == KafkaException.class ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.backgroundError.get().toString().contains("Requested metadata update after close"));
        this.clearBackgroundError();
    }

    @Test(expected=IllegalStateException.class)
    public void testMetadataUpdateAfterClose() {
        this.metadata.close();
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 1000L);
    }

    private static void checkTimeToNextUpdate(long refreshBackoffMs, long metadataExpireMs) {
        long now = 10000L;
        if (metadataExpireMs > now || refreshBackoffMs > now) {
            throw new IllegalArgumentException("metadataExpireMs and refreshBackoffMs must be smaller than 'now'");
        }
        long largerOfBackoffAndExpire = Math.max(refreshBackoffMs, metadataExpireMs);
        Metadata metadata = new Metadata(refreshBackoffMs, metadataExpireMs, true);
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now));
        metadata.update(MetadataTest.emptyMetadataResponse(), now);
        Assert.assertEquals((long)largerOfBackoffAndExpire, (long)metadata.timeToNextUpdate(now));
        metadata.requestUpdate();
        Assert.assertEquals((long)refreshBackoffMs, (long)metadata.timeToNextUpdate(now));
        metadata.update(MetadataTest.emptyMetadataResponse(), now);
        Assert.assertEquals((long)largerOfBackoffAndExpire, (long)metadata.timeToNextUpdate(now));
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now += largerOfBackoffAndExpire));
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now + 1L));
    }

    @Test
    public void testTimeToNextUpdate() {
        MetadataTest.checkTimeToNextUpdate(100L, 1000L);
        MetadataTest.checkTimeToNextUpdate(1000L, 100L);
        MetadataTest.checkTimeToNextUpdate(0L, 0L);
        MetadataTest.checkTimeToNextUpdate(0L, 100L);
        MetadataTest.checkTimeToNextUpdate(100L, 0L);
    }

    @Test
    public void testTimeToNextUpdate_RetryBackoff() {
        long now = 10000L;
        this.metadata.failedUpdate(now, null);
        Assert.assertEquals((long)this.refreshBackoffMs, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.requestUpdate();
        Assert.assertEquals((long)this.refreshBackoffMs, (long)this.metadata.timeToNextUpdate(now));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now += this.refreshBackoffMs));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now + 1L));
    }

    @Test
    public void testTimeToNextUpdate_OverwriteBackoff() {
        long now = 10000L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), now);
        this.metadata.add("new-topic");
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(MetadataTest.emptyMetadataResponse(), now);
        this.metadata.setTopics((Collection)this.metadata.topics());
        Assert.assertEquals((long)this.metadataExpireMs, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.setTopics(Collections.singletonList("another-new-topic"));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(MetadataTest.emptyMetadataResponse(), now);
        this.metadata.needMetadataForAllTopics(true);
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(MetadataTest.emptyMetadataResponse(), now);
        this.metadata.needMetadataForAllTopics(true);
        Assert.assertEquals((long)this.metadataExpireMs, (long)this.metadata.timeToNextUpdate(now));
    }

    @Test
    public void testMetadataUpdateWaitTime() throws Exception {
        long time = 0L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        try {
            this.metadata.awaitUpdate(this.metadata.requestUpdate(), 0L);
            Assert.fail((String)"Wait on metadata update was expected to timeout, but it didn't");
        }
        catch (TimeoutException timeoutException) {
            // empty catch block
        }
        long twoSecondWait = 2000L;
        try {
            this.metadata.awaitUpdate(this.metadata.requestUpdate(), 2000L);
            Assert.fail((String)"Wait on metadata update was expected to timeout, but it didn't");
        }
        catch (TimeoutException timeoutException) {
            // empty catch block
        }
    }

    @Test
    public void testFailedUpdate() {
        long time = 100L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1000L));
        this.metadata.failedUpdate(1100L, null);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1100L));
        Assert.assertEquals((long)100L, (long)this.metadata.lastSuccessfulUpdate());
        this.metadata.needMetadataForAllTopics(true);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1000L));
    }

    @Test
    public void testUpdateWithNeedMetadataForAllTopics() {
        long time = 0L;
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        this.metadata.needMetadataForAllTopics(true);
        List<String> expectedTopics = Collections.singletonList("topic");
        this.metadata.setTopics(expectedTopics);
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic", 1);
        partitionCounts.put("topic1", 1);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith(1, partitionCounts);
        this.metadata.update(metadataResponse, 100L);
        Assert.assertArrayEquals((String)"Metadata got updated with wrong set of topics.", (Object[])expectedTopics.toArray(), (Object[])this.metadata.topics().toArray());
        this.metadata.needMetadataForAllTopics(false);
    }

    @Test
    public void testClusterListenerGetsNotifiedOfUpdate() {
        long time = 0L;
        MockClusterResourceListener mockClusterListener = new MockClusterResourceListener();
        ClusterResourceListeners listeners = new ClusterResourceListeners();
        listeners.maybeAdd((Object)mockClusterListener);
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, false, listeners);
        String hostName = "www.example.com";
        this.metadata.bootstrap(Collections.singletonList(new InetSocketAddress(hostName, 9002)), time);
        Assert.assertFalse((String)"ClusterResourceListener should not called when metadata is updated with bootstrap Cluster", (boolean)MockClusterResourceListener.IS_ON_UPDATE_CALLED.get());
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic", 1);
        partitionCounts.put("topic1", 1);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, partitionCounts);
        this.metadata.update(metadataResponse, 100L);
        Assert.assertEquals((String)"MockClusterResourceListener did not get cluster metadata correctly", (Object)"dummy", (Object)mockClusterListener.clusterResource().clusterId());
        Assert.assertTrue((String)"MockClusterResourceListener should be called when metadata is updated with non-bootstrap Cluster", (boolean)MockClusterResourceListener.IS_ON_UPDATE_CALLED.get());
    }

    @Test
    public void testListenerGetsNotifiedOfUpdate() {
        long time = 0L;
        final HashSet topics = new HashSet();
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        this.metadata.addListener(new Metadata.Listener(){

            public void onMetadataUpdate(Cluster cluster, Set<String> unavailableTopics) {
                topics.clear();
                topics.addAll(cluster.topics());
            }
        });
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic", 1);
        partitionCounts.put("topic1", 1);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, partitionCounts);
        this.metadata.update(metadataResponse, 100L);
        Assert.assertEquals((String)"Listener did not update topics list correctly", new HashSet<String>(Arrays.asList("topic", "topic1")), topics);
    }

    @Test
    public void testListenerCanUnregister() {
        long time = 0L;
        final HashSet topics = new HashSet();
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        Metadata.Listener listener = new Metadata.Listener(){

            public void onMetadataUpdate(Cluster cluster, Set<String> unavailableTopics) {
                topics.clear();
                topics.addAll(cluster.topics());
            }
        };
        this.metadata.addListener(listener);
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic", 1);
        partitionCounts.put("topic1", 1);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, partitionCounts);
        this.metadata.update(metadataResponse, 100L);
        this.metadata.removeListener(listener);
        partitionCounts.clear();
        partitionCounts.put("topic2", 1);
        partitionCounts.put("topic3", 1);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, partitionCounts);
        this.metadata.update(metadataResponse, 100L);
        Assert.assertEquals((String)"Listener did not update topics list correctly", new HashSet<String>(Arrays.asList("topic", "topic1")), topics);
    }

    @Test
    public void testTopicExpiry() throws Exception {
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, true, new ClusterResourceListeners());
        long time = 0L;
        this.metadata.add("topic1");
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time += 300000L);
        Assert.assertFalse((String)"Unused topic not expired", (boolean)this.metadata.containsTopic("topic1"));
        this.metadata.add("topic2");
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        for (int i = 0; i < 3; ++i) {
            this.metadata.update(MetadataTest.emptyMetadataResponse(), time += 150000L);
            Assert.assertTrue((String)"Topic expired even though in use", (boolean)this.metadata.containsTopic("topic2"));
            this.metadata.add("topic2");
        }
        HashSet<String> topics = new HashSet<String>();
        topics.add("topic4");
        this.metadata.setTopics(topics);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time += 300000L);
        Assert.assertFalse((String)"Unused topic not expired", (boolean)this.metadata.containsTopic("topic4"));
    }

    @Test
    public void testNonExpiringMetadata() throws Exception {
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, false, new ClusterResourceListeners());
        long time = 0L;
        this.metadata.add("topic1");
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time += 300000L);
        Assert.assertTrue((String)"Unused topic expired when expiry disabled", (boolean)this.metadata.containsTopic("topic1"));
        this.metadata.add("topic2");
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time);
        for (int i = 0; i < 3; ++i) {
            this.metadata.update(MetadataTest.emptyMetadataResponse(), time += 150000L);
            Assert.assertTrue((String)"Topic expired even though in use", (boolean)this.metadata.containsTopic("topic2"));
            this.metadata.add("topic2");
        }
        HashSet<String> topics = new HashSet<String>();
        topics.add("topic4");
        this.metadata.setTopics(topics);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), time += this.metadataExpireMs * 2L);
        Assert.assertTrue((String)"Unused topic expired when expiry disabled", (boolean)this.metadata.containsTopic("topic4"));
    }

    @Test
    public void testRequestUpdate() {
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, false, new ClusterResourceListeners());
        Assert.assertFalse((boolean)this.metadata.updateRequested());
        int[] epochs = new int[]{42, 42, 41, 41, 42, 43, 43, 42, 41, 44};
        boolean[] updateResult = new boolean[]{true, false, false, false, false, true, false, false, false, true};
        TopicPartition tp = new TopicPartition("topic", 0);
        for (int i = 0; i < epochs.length; ++i) {
            this.metadata.updateLastSeenEpochIfNewer(tp, epochs[i]);
            if (updateResult[i]) {
                Assert.assertTrue((String)("Expected metadata update to be requested [" + i + "]"), (boolean)this.metadata.updateRequested());
            } else {
                Assert.assertFalse((String)("Did not expect metadata update to be requested [" + i + "]"), (boolean)this.metadata.updateRequested());
            }
            this.metadata.update(MetadataTest.emptyMetadataResponse(), 0L);
            Assert.assertFalse((boolean)this.metadata.updateRequested());
        }
    }

    @Test
    public void testRejectOldMetadata() {
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic-1", 1);
        TopicPartition tp = new TopicPartition("topic-1", 0);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 0L);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(100), replicas, isr, offlineReplicas));
        this.metadata.update(metadataResponse, 10L);
        Assert.assertNotNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(99), replicas, Collections.emptyList(), offlineReplicas));
        this.metadata.update(metadataResponse, 20L);
        Assert.assertEquals((long)this.metadata.fetch().partition(tp).inSyncReplicas().length, (long)1L);
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(100), replicas, Collections.emptyList(), offlineReplicas));
        this.metadata.update(metadataResponse, 20L);
        Assert.assertEquals((long)this.metadata.fetch().partition(tp).inSyncReplicas().length, (long)0L);
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), Collections.emptyMap(), MetadataResponse.PartitionMetadata::new);
        this.metadata.update(metadataResponse, 20L);
        Assert.assertNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(99), replicas, isr, offlineReplicas));
        this.metadata.update(metadataResponse, 10L);
        Assert.assertNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
    }

    @Test
    public void testMaybeRequestUpdate() {
        TopicPartition tp = new TopicPartition("topic-1", 0);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 0L);
        Assert.assertTrue((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 1));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)1L);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 1L);
        Assert.assertFalse((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 1));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)1L);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 2L);
        Assert.assertFalse((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 0));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)1L);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 3L);
        Assert.assertTrue((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 2));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)2L);
    }

    @Test
    public void testOutOfBandEpochUpdate() {
        HashMap<String, Integer> partitionCounts = new HashMap<String, Integer>();
        partitionCounts.put("topic-1", 5);
        TopicPartition tp = new TopicPartition("topic-1", 0);
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 0L);
        Assert.assertTrue((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 99));
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(100), replicas, isr, offlineReplicas));
        this.metadata.update(metadataResponse, 10L);
        Assert.assertNotNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)100L);
        Assert.assertTrue((boolean)this.metadata.updateLastSeenEpochIfNewer(tp, 101));
        Assert.assertNotNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)this.metadata.fetch().partitionCountForTopic("topic-1").longValue(), (long)5L);
        Assert.assertFalse((boolean)this.metadata.partitionInfoIfCurrent(tp).isPresent());
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)101L);
        this.metadata.update(metadataResponse, 20L);
        Assert.assertNotNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)this.metadata.fetch().partitionCountForTopic("topic-1").longValue(), (long)5L);
        Assert.assertFalse((boolean)this.metadata.partitionInfoIfCurrent(tp).isPresent());
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)101L);
        metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), partitionCounts, (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.of(101), replicas, isr, offlineReplicas));
        this.metadata.update(metadataResponse, 30L);
        Assert.assertNotNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertEquals((long)this.metadata.fetch().partitionCountForTopic("topic-1").longValue(), (long)5L);
        Assert.assertTrue((boolean)this.metadata.partitionInfoIfCurrent(tp).isPresent());
        Assert.assertEquals((long)((Integer)this.metadata.lastSeenLeaderEpoch(tp).get()).longValue(), (long)101L);
        this.metadata.setTopics(Collections.singletonList("topic-2"));
        Assert.assertNull((Object)this.metadata.fetch().partition(tp));
        Assert.assertNull((Object)this.metadata.fetch().partitionCountForTopic("topic-1"));
        Assert.assertFalse((boolean)this.metadata.partitionInfoIfCurrent(tp).isPresent());
    }

    @Test
    public void testNoEpoch() {
        this.metadata.update(MetadataTest.emptyMetadataResponse(), 0L);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 1, Collections.emptyMap(), Collections.singletonMap("topic-1", 1), (error, partition, leader, leaderEpoch, replicas, isr, offlineReplicas) -> new MetadataResponse.PartitionMetadata(error, partition, leader, Optional.empty(), replicas, isr, offlineReplicas));
        this.metadata.update(metadataResponse, 10L);
        TopicPartition tp = new TopicPartition("topic-1", 0);
        Assert.assertFalse((boolean)this.metadata.lastSeenLeaderEpoch(tp).isPresent());
        Assert.assertTrue((boolean)this.metadata.partitionInfoIfCurrent(tp).isPresent());
        Assert.assertEquals((long)((PartitionInfo)this.metadata.partitionInfoIfCurrent(tp).get()).partition(), (long)0L);
        Assert.assertEquals((long)((PartitionInfo)this.metadata.partitionInfoIfCurrent(tp).get()).leader().id(), (long)0L);
    }

    @Test
    public void testClusterCopy() {
        HashMap<String, Integer> counts = new HashMap<String, Integer>();
        HashMap<String, Errors> errors = new HashMap<String, Errors>();
        counts.put("topic1", 2);
        counts.put("topic2", 3);
        counts.put("__consumer_offsets", 3);
        errors.put("topic3", Errors.INVALID_TOPIC_EXCEPTION);
        errors.put("topic4", Errors.TOPIC_AUTHORIZATION_FAILED);
        MetadataResponse metadataResponse = TestUtils.metadataUpdateWith("dummy", 4, errors, counts);
        this.metadata.update(metadataResponse, 0L);
        Cluster cluster = this.metadata.fetch();
        Assert.assertEquals((Object)cluster.clusterResource().clusterId(), (Object)"dummy");
        Assert.assertEquals((long)cluster.nodes().size(), (long)4L);
        Assert.assertEquals((Object)cluster.invalidTopics(), Collections.singleton("topic3"));
        Assert.assertEquals((Object)cluster.unauthorizedTopics(), Collections.singleton("topic4"));
        Assert.assertEquals((long)cluster.topics().size(), (long)3L);
        Assert.assertEquals((Object)cluster.internalTopics(), Collections.singleton("__consumer_offsets"));
        Assert.assertEquals((long)cluster.partitionsForTopic("topic1").size(), (long)2L);
        Assert.assertEquals((long)cluster.partitionsForTopic("topic2").size(), (long)3L);
        InetSocketAddress address = InetSocketAddress.createUnresolved("localhost", 0);
        Cluster fromMetadata = MetadataCache.bootstrap(Collections.singletonList(address)).cluster();
        Cluster fromCluster = Cluster.bootstrap(Collections.singletonList(address));
        Assert.assertEquals((Object)fromMetadata, (Object)fromCluster);
        Cluster fromMetadataEmpty = MetadataCache.empty().cluster();
        Cluster fromClusterEmpty = Cluster.empty();
        Assert.assertEquals((Object)fromMetadataEmpty, (Object)fromClusterEmpty);
    }

    @Test
    public void testRequestVersion() {
        MockTime time = new MockTime();
        this.metadata.requestUpdate();
        Metadata.MetadataRequestAndVersion versionAndBuilder = this.metadata.newMetadataRequestAndVersion();
        this.metadata.update(versionAndBuilder.requestVersion, TestUtils.metadataUpdateWith(1, Collections.singletonMap("topic", 1)), time.milliseconds());
        Assert.assertFalse((boolean)this.metadata.updateRequested());
        this.metadata.requestUpdateForNewTopics();
        versionAndBuilder = this.metadata.newMetadataRequestAndVersion();
        this.metadata.requestUpdateForNewTopics();
        this.metadata.update(versionAndBuilder.requestVersion, TestUtils.metadataUpdateWith(1, Collections.singletonMap("topic", 1)), time.milliseconds());
        Assert.assertTrue((boolean)this.metadata.updateRequested());
        versionAndBuilder = this.metadata.newMetadataRequestAndVersion();
        this.metadata.update(versionAndBuilder.requestVersion, TestUtils.metadataUpdateWith(1, Collections.singletonMap("topic", 1)), time.milliseconds());
        Assert.assertFalse((boolean)this.metadata.updateRequested());
    }

    private void clearBackgroundError() {
        this.backgroundError.set(null);
    }

    private Thread asyncFetch(final String topic, final long maxWaitMs) {
        Thread thread = new Thread(){

            @Override
            public void run() {
                try {
                    while (MetadataTest.this.metadata.fetch().partitionsForTopic(topic).isEmpty()) {
                        MetadataTest.this.metadata.awaitUpdate(MetadataTest.this.metadata.requestUpdate(), maxWaitMs);
                    }
                }
                catch (Exception e) {
                    MetadataTest.this.backgroundError.set(e);
                }
            }
        };
        thread.start();
        return thread;
    }
}

