/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.atmos.binders;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import jakarta.inject.Inject;
import jakarta.inject.Singleton;

import com.google.common.io.BaseEncoding;

import org.jclouds.atmos.domain.AtmosObject;
import org.jclouds.atmos.reference.AtmosHeaders;
import org.jclouds.http.HttpRequest;
import org.jclouds.rest.Binder;

@Singleton
public class BindMetadataToHeaders implements Binder {
   private final BindUserMetadataToHeaders metaBinder;

   @Inject
   protected BindMetadataToHeaders(BindUserMetadataToHeaders metaBinder) {
      this.metaBinder = metaBinder;
   }

   @Override
   public <R extends HttpRequest> R bindToRequest(R request, Object input) {
      checkArgument(checkNotNull(input, "input") instanceof AtmosObject, "this binder is only valid for AtmosObject!");
      checkNotNull(request, "request");

      AtmosObject object = AtmosObject.class.cast(input);
      checkNotNull(object.getPayload(), "object payload");
      checkArgument(object.getPayload().getContentMetadata().getContentLength() != null,
            "contentLength must be set, streaming not supported");
      byte[] contentMD5 = object.getContentMetadata().getContentMD5();
      if (contentMD5 != null) {
         request = (R) request.toBuilder()
               .addHeader(AtmosHeaders.CHECKSUM, "MD5/0/" +
                     BaseEncoding.base16().encode(contentMD5))
               .build();
      }
      return metaBinder.bindToRequest(request, object.getUserMetadata());
   }
}
