/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file.tar;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.file.tar.CleanupContext;
import org.apache.jackrabbit.oak.segment.file.tar.EntryRecovery;
import org.apache.jackrabbit.oak.segment.file.tar.GCGeneration;
import org.apache.jackrabbit.oak.segment.file.tar.SegmentGraph;
import org.apache.jackrabbit.oak.segment.file.tar.TarRecovery;
import org.apache.jackrabbit.oak.segment.file.tar.TarWriter;
import org.apache.jackrabbit.oak.segment.file.tar.binaries.BinaryReferencesIndex;
import org.apache.jackrabbit.oak.segment.file.tar.binaries.BinaryReferencesIndexLoader;
import org.apache.jackrabbit.oak.segment.file.tar.binaries.InvalidBinaryReferencesIndexException;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveEntry;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveManager;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TarReader
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TarReader.class);
    private final SegmentArchiveManager archiveManager;
    private final SegmentArchiveReader archive;
    private final Set<UUID> segmentUUIDs;

    static TarReader open(String file, SegmentArchiveManager archiveManager) throws IOException {
        TarReader reader = TarReader.openFirstFileWithValidIndex(Collections.singletonList(file), archiveManager);
        if (reader != null) {
            return reader;
        }
        throw new IOException("Failed to open tar file " + file);
    }

    static TarReader open(Map<Character, String> files, TarRecovery recovery, SegmentArchiveManager archiveManager) throws IOException {
        TreeMap<Character, String> sorted = new TreeMap<Character, String>();
        sorted.putAll(files);
        ArrayList<String> list = new ArrayList<String>(sorted.values());
        Collections.reverse(list);
        TarReader reader = TarReader.openFirstFileWithValidIndex(list, archiveManager);
        if (reader != null) {
            return reader;
        }
        log.warn("Could not find a valid tar index in {}, recovering...", list);
        LinkedHashMap<UUID, byte[]> entries = new LinkedHashMap<UUID, byte[]>();
        for (String file : sorted.values()) {
            TarReader.collectFileEntries(file, entries, true, archiveManager);
        }
        String file = (String)sorted.values().iterator().next();
        TarReader.generateTarFile(entries, file, recovery, archiveManager);
        reader = TarReader.openFirstFileWithValidIndex(Collections.singletonList(file), archiveManager);
        if (reader != null) {
            return reader;
        }
        throw new IOException("Failed to open recovered tar file " + file);
    }

    static TarReader openRO(Map<Character, String> files, TarRecovery recovery, SegmentArchiveManager archiveManager) throws IOException {
        String file = files.get(Collections.max(files.keySet()));
        OpenStrategy recoverAndOpen = (segmentArchiveManager, archiveName) -> {
            log.info("Could not find a valid tar index in {}, recovering read-only", (Object)archiveName);
            LinkedHashMap<UUID, byte[]> entries = new LinkedHashMap<UUID, byte[]>();
            TarReader.collectFileEntries(archiveName, entries, false, segmentArchiveManager);
            String bakFile = TarReader.findAvailGen(archiveName, ".ro.bak", segmentArchiveManager);
            TarReader.generateTarFile(entries, bakFile, recovery, segmentArchiveManager);
            return segmentArchiveManager.open(bakFile);
        };
        for (OpenStrategy openStrategy : new OpenStrategy[]{SegmentArchiveManager::open, SegmentArchiveManager::forceOpen, recoverAndOpen}) {
            TarReader reader = TarReader.openFirstFileWithValidIndex(Collections.singletonList(file), archiveManager, openStrategy);
            if (reader == null) continue;
            return reader;
        }
        throw new IOException("Failed to open tar file " + file);
    }

    private static void collectFileEntries(String file, LinkedHashMap<UUID, byte[]> entries, boolean backup, SegmentArchiveManager archiveManager) throws IOException {
        log.info("Recovering segments from tar file {}", (Object)file);
        try {
            archiveManager.recoverEntries(file, entries);
        }
        catch (IOException e) {
            log.warn("Could not read tar file {}, skipping...", (Object)file, (Object)e);
        }
        if (backup) {
            TarReader.backupSafely(archiveManager, file, entries.keySet());
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void generateTarFile(LinkedHashMap<UUID, byte[]> entries, String file, TarRecovery recovery, SegmentArchiveManager archiveManager) throws IOException {
        log.info("Regenerating tar file {}", (Object)file);
        try (final TarWriter writer = new TarWriter(archiveManager, file);){
            final HashMap segmentMap = new HashMap(entries.size());
            for (Map.Entry<UUID, byte[]> entry : entries.entrySet()) {
                try {
                    recovery.recoverEntry(entry.getKey(), entry.getValue(), new EntryRecovery(){

                        @Override
                        public void recoverEntry(long msb, long lsb, byte[] data, int offset, int size, GCGeneration generation) throws IOException {
                            writer.writeEntry(msb, lsb, data, offset, size, generation);
                        }

                        @Override
                        public void recoverGraphEdge(UUID from, UUID to) {
                            writer.addGraphEdge(from, to);
                        }

                        @Override
                        public void recoverBinaryReference(GCGeneration generation, UUID segmentId, String reference) {
                            writer.addBinaryReference(generation, segmentId, reference);
                        }

                        @Override
                        public Segment getSegment(SegmentId id) {
                            return (Segment)segmentMap.get(id);
                        }

                        @Override
                        public void addSegment(Segment segment) {
                            segmentMap.put(segment.getSegmentId(), segment);
                        }

                        @Override
                        public Map<SegmentId, Segment> getRecoveredSegments() {
                            return segmentMap;
                        }
                    });
                }
                catch (IOException e) {
                    throw new IOException(String.format("Unable to recover entry %s for file %s", entry.getKey(), file), e);
                    return;
                }
            }
        }
    }

    private static void backupSafely(SegmentArchiveManager archiveManager, String file, Set<UUID> recoveredEntries) throws IOException {
        String backup = TarReader.findAvailGen(file, ".bak", archiveManager);
        log.info("Backing up {} to {}", (Object)file, (Object)backup);
        archiveManager.backup(file, backup, recoveredEntries);
    }

    private static String findAvailGen(String name, String ext, SegmentArchiveManager archiveManager) {
        String backup = name + ext;
        int i = 2;
        while (archiveManager.exists(backup)) {
            backup = name + "." + i + ext;
            ++i;
        }
        return backup;
    }

    private static TarReader openFirstFileWithValidIndex(List<String> archives, SegmentArchiveManager archiveManager) {
        return TarReader.openFirstFileWithValidIndex(archives, archiveManager, SegmentArchiveManager::open);
    }

    private static TarReader openFirstFileWithValidIndex(List<String> archives, SegmentArchiveManager archiveManager, OpenStrategy openStrategy) {
        for (String name : archives) {
            try {
                SegmentArchiveReader reader = openStrategy.open(archiveManager, name);
                if (reader == null) continue;
                for (String other : archives) {
                    if (Objects.equals(other, name)) continue;
                    log.info("Removing unused tar file {}", (Object)other);
                    archiveManager.delete(other);
                }
                return new TarReader(archiveManager, reader);
            }
            catch (IOException e) {
                log.warn("Could not read tar file {}, skipping...", (Object)name, (Object)e);
            }
        }
        return null;
    }

    private TarReader(SegmentArchiveManager archiveManager, SegmentArchiveReader archive) {
        this.archiveManager = archiveManager;
        this.archive = archive;
        this.segmentUUIDs = archive.listSegments().stream().map(e -> new UUID(e.getMsb(), e.getLsb())).collect(Collectors.toSet());
    }

    long size() {
        return this.archive.length();
    }

    Set<UUID> getUUIDs() {
        return this.segmentUUIDs;
    }

    boolean containsEntry(long msb, long lsb) {
        return this.archive.containsSegment(msb, lsb);
    }

    Buffer readEntry(long msb, long lsb) throws IOException {
        return this.archive.readSegment(msb, lsb);
    }

    @NotNull
    SegmentArchiveEntry[] getEntries() {
        List<SegmentArchiveEntry> entryList = this.archive.listSegments();
        return entryList.toArray(new SegmentArchiveEntry[entryList.size()]);
    }

    void collectBlobReferences(@NotNull Consumer<String> collector, Predicate<GCGeneration> skipGeneration) {
        BinaryReferencesIndex references = this.getBinaryReferences();
        if (references == null) {
            return;
        }
        references.forEach((generation, full, compacted, segment, reference) -> {
            if (skipGeneration.test(GCGeneration.newGCGeneration(generation, full, compacted))) {
                return;
            }
            collector.accept(reference);
        });
    }

    void mark(Set<UUID> references, Set<UUID> reclaimable, CleanupContext context) throws IOException {
        if (this.archiveManager.isReadOnly(this.getFileName())) {
            return;
        }
        SegmentGraph graph = this.getGraph();
        SegmentArchiveEntry[] entries = this.getEntries();
        for (int i = entries.length - 1; i >= 0; --i) {
            GCGeneration generation;
            SegmentArchiveEntry entry = entries[i];
            UUID id = new UUID(entry.getMsb(), entry.getLsb());
            if (context.shouldReclaim(id, generation = GCGeneration.newGCGeneration(entry), references.remove(id))) {
                reclaimable.add(id);
                continue;
            }
            for (UUID refId : graph.getEdges(id)) {
                if (!context.shouldFollow(id, refId)) continue;
                references.add(refId);
            }
        }
    }

    TarReader sweep(@NotNull Set<UUID> reclaim, @NotNull Set<UUID> reclaimed) throws IOException {
        if (this.archiveManager.isReadOnly(this.getFileName())) {
            return this;
        }
        String name = this.archive.getName();
        log.debug("Cleaning up {}", (Object)name);
        HashSet<UUID> cleaned = new HashSet<UUID>();
        int afterSize = 0;
        int beforeSize = 0;
        int afterCount = 0;
        SegmentArchiveEntry[] entries = this.getEntries();
        for (int i = 0; i < entries.length; ++i) {
            SegmentArchiveEntry entry = entries[i];
            beforeSize += this.archive.getEntrySize(entry.getLength());
            UUID id2 = new UUID(entry.getMsb(), entry.getLsb());
            if (reclaim.contains(id2)) {
                cleaned.add(id2);
                entries[i] = null;
                continue;
            }
            afterSize += this.archive.getEntrySize(entry.getLength());
            ++afterCount;
        }
        if (afterCount == 0) {
            log.debug("None of the entries of {} are referenceable.", (Object)name);
            return null;
        }
        if (afterSize >= beforeSize * 3 / 4) {
            log.debug("Not enough space savings. ({}/{}). Skipping clean up of {}", new Object[]{this.archive.length() - (long)afterSize, this.archive.length(), name});
            return this;
        }
        int pos = name.length() - "a.tar".length();
        char generation = name.charAt(pos);
        if (generation == 'z') {
            log.debug("No garbage collection after reaching generation z: {}", (Object)name);
            return this;
        }
        String newFile = name.substring(0, pos) + (char)(generation + '\u0001') + ".tar";
        log.debug("Writing new generation {}", (Object)newFile);
        TarWriter writer = new TarWriter(this.archiveManager, newFile);
        for (SegmentArchiveEntry entry : entries) {
            if (entry == null) continue;
            long msb = entry.getMsb();
            long lsb = entry.getLsb();
            int size = entry.getLength();
            GCGeneration gen2 = GCGeneration.newGCGeneration(entry);
            byte[] data = new byte[size];
            this.archive.readSegment(msb, lsb).get(data);
            writer.writeEntry(msb, lsb, data, 0, size, gen2);
        }
        SegmentGraph graph = this.getGraph();
        for (Map.Entry<UUID, Set<UUID>> e : graph.getEdges().entrySet()) {
            UUID from = e.getKey();
            if (cleaned.contains(from)) continue;
            for (UUID to : e.getValue()) {
                if (cleaned.contains(to)) continue;
                writer.addGraphEdge(from, to);
            }
        }
        BinaryReferencesIndex references = this.getBinaryReferences();
        if (references != null) {
            references.forEach((gen, full, compacted, id, reference) -> {
                if (cleaned.contains(id)) {
                    return;
                }
                writer.addBinaryReference(GCGeneration.newGCGeneration(gen, full, compacted), id, reference);
            });
        }
        writer.close();
        TarReader reader = TarReader.openFirstFileWithValidIndex(Collections.singletonList(newFile), this.archiveManager);
        if (reader != null) {
            reclaimed.addAll(cleaned);
            return reader;
        }
        log.warn("Failed to open cleaned up tar file {}", (Object)this.getFileName());
        return this;
    }

    @Override
    public void close() throws IOException {
        this.archive.close();
    }

    @NotNull
    SegmentGraph getGraph() throws IOException {
        return this.archive.getGraph();
    }

    @Nullable
    BinaryReferencesIndex getBinaryReferences() {
        try {
            Buffer binaryReferences = this.archive.getBinaryReferences();
            if (binaryReferences == null) {
                log.info("The archive directory {} still does not have file with binary references written.", (Object)this.archive.getName());
                return null;
            }
            return BinaryReferencesIndexLoader.parseBinaryReferencesIndex(binaryReferences);
        }
        catch (IOException | InvalidBinaryReferencesIndexException e) {
            log.warn("Exception while loading binary reference", (Throwable)e);
            return null;
        }
    }

    String getFileName() {
        return this.archive.getName();
    }

    public String toString() {
        return this.getFileName();
    }

    @FunctionalInterface
    private static interface OpenStrategy {
        public SegmentArchiveReader open(SegmentArchiveManager var1, String var2) throws IOException;
    }
}

