/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file.tar;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.zip.CRC32;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.data.SegmentData;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveEntry;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.util.ReaderAtEnd;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SegmentGraph {
    private static final Logger log = LoggerFactory.getLogger(SegmentGraph.class);
    private static final int FOOTER_SIZE = 16;
    private static final int MAGIC = 170936074;
    @NotNull
    private final Map<UUID, Set<UUID>> edgeMap;

    public SegmentGraph() {
        this.edgeMap = new HashMap<UUID, Set<UUID>>();
    }

    private SegmentGraph(@NotNull Map<UUID, Set<UUID>> edgeMap) {
        this.edgeMap = edgeMap;
    }

    public void addEdge(@NotNull UUID from, @NotNull UUID to) {
        this.edgeMap.computeIfAbsent(from, k -> new HashSet()).add(to);
    }

    @NotNull
    public Map<UUID, Set<UUID>> getEdges() {
        return Collections.unmodifiableMap(this.edgeMap);
    }

    @NotNull
    public Set<UUID> getEdges(@NotNull UUID from) {
        Set set = this.edgeMap.getOrDefault(from, Collections.emptySet());
        return Collections.unmodifiableSet(set);
    }

    @Nullable
    public static SegmentGraph load(ReaderAtEnd readerAtEnd) throws IOException {
        Buffer meta = readerAtEnd.readAtEnd(16, 16);
        int crc32 = meta.getInt();
        int count = meta.getInt();
        int bytes = meta.getInt();
        int magic = meta.getInt();
        if (magic != 170936074) {
            log.warn("Invalid graph magic number");
            return null;
        }
        if (count < 0) {
            log.warn("Invalid number of entries");
            return null;
        }
        if (bytes < 4 + count * 34) {
            log.warn("Invalid entry size");
            return null;
        }
        Buffer buffer = readerAtEnd.readAtEnd(bytes, bytes);
        byte[] b = new byte[bytes - 16];
        buffer.mark();
        buffer.get(b);
        buffer.reset();
        CRC32 checksum = new CRC32();
        checksum.update(b);
        if (crc32 != (int)checksum.getValue()) {
            log.warn("Invalid graph checksum in tar file");
            return null;
        }
        return SegmentGraph.parse(buffer);
    }

    @NotNull
    public static SegmentGraph compute(SegmentArchiveReader archiveReader) throws IOException {
        SegmentGraph graph = new SegmentGraph();
        for (SegmentArchiveEntry entry : archiveReader.listSegments()) {
            if (!SegmentId.isDataSegmentId(entry.getLsb())) continue;
            UUID from = new UUID(entry.getMsb(), entry.getLsb());
            Buffer buffer = archiveReader.readSegment(entry.getMsb(), entry.getLsb());
            SegmentData data = SegmentData.newSegmentData(buffer);
            for (int i = 0; i < data.getSegmentReferencesCount(); ++i) {
                UUID to = new UUID(data.getSegmentReferenceMsb(i), data.getSegmentReferenceLsb(i));
                graph.addEdge(from, to);
            }
        }
        return graph;
    }

    @NotNull
    public static SegmentGraph parse(@NotNull Buffer buffer) {
        int nEntries = buffer.getInt(buffer.limit() - 12);
        HashMap<UUID, Set<UUID>> edgeMap = new HashMap<UUID, Set<UUID>>(nEntries);
        for (int i = 0; i < nEntries; ++i) {
            long msb = buffer.getLong();
            long lsb = buffer.getLong();
            int nVertices = buffer.getInt();
            HashSet<UUID> vertices = new HashSet<UUID>(nVertices);
            for (int j = 0; j < nVertices; ++j) {
                long vMsb = buffer.getLong();
                long vLsb = buffer.getLong();
                vertices.add(new UUID(vMsb, vLsb));
            }
            edgeMap.put(new UUID(msb, lsb), vertices);
        }
        return new SegmentGraph(edgeMap);
    }

    public byte[] write() {
        int graphSize = this.size();
        Buffer buffer = Buffer.allocate((int)graphSize);
        for (Map.Entry<UUID, Set<UUID>> entry : this.edgeMap.entrySet()) {
            UUID from = entry.getKey();
            buffer.putLong(from.getMostSignificantBits());
            buffer.putLong(from.getLeastSignificantBits());
            Set<UUID> adj = entry.getValue();
            buffer.putInt(adj.size());
            for (UUID to : adj) {
                buffer.putLong(to.getMostSignificantBits());
                buffer.putLong(to.getLeastSignificantBits());
            }
        }
        CRC32 checksum = new CRC32();
        checksum.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)checksum.getValue());
        buffer.putInt(this.edgeMap.size());
        buffer.putInt(graphSize);
        buffer.putInt(170936074);
        return buffer.array();
    }

    public int size() {
        int graphSize = 16;
        for (Map.Entry<UUID, Set<UUID>> entry : this.edgeMap.entrySet()) {
            graphSize += 16;
            graphSize += 4;
            graphSize += 16 * entry.getValue().size();
        }
        return graphSize;
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof SegmentGraph) {
            return this.edgeMap.equals(((SegmentGraph)other).edgeMap);
        }
        return false;
    }

    public int hashCode() {
        return this.edgeMap.hashCode();
    }
}

