/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.tool;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.segment.SegmentNodeStoreBuilders;
import org.apache.jackrabbit.oak.segment.file.FileStoreBuilder;
import org.apache.jackrabbit.oak.segment.file.JournalEntry;
import org.apache.jackrabbit.oak.segment.file.JournalReader;
import org.apache.jackrabbit.oak.segment.file.ReadOnlyFileStore;
import org.apache.jackrabbit.oak.segment.file.tar.LocalJournalFile;
import org.apache.jackrabbit.oak.segment.file.tar.TarPersistence;
import org.apache.jackrabbit.oak.segment.file.tooling.ConsistencyChecker;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitorAdapter;

public class Check {
    private final File path;
    private final boolean mmap;
    private final File journal;
    private final long debugInterval;
    private final boolean checkBinaries;
    private final boolean checkHead;
    private final Integer revisionsCount;
    private final Set<String> requestedCheckpoints;
    private final Set<String> filterPaths;
    private final boolean ioStatistics;
    private RepositoryStatistics repoStatistics;
    private final PrintWriter out;
    private final PrintWriter err;
    private int currentNodeCount;
    private int currentPropertyCount;
    private int headNodeCount;
    private int headPropertyCount;
    private long lastDebugEvent;

    public static Builder builder() {
        return new Builder();
    }

    private Check(Builder builder) {
        this.path = builder.path;
        this.mmap = builder.mmap;
        this.debugInterval = builder.debugInterval;
        this.checkHead = builder.checkHead;
        this.checkBinaries = builder.checkBinaries;
        this.requestedCheckpoints = builder.checkpoints;
        this.filterPaths = builder.filterPaths;
        this.ioStatistics = builder.ioStatistics;
        this.repoStatistics = builder.repoStatistics;
        this.out = builder.outWriter;
        this.err = builder.errWriter;
        this.journal = Check.journalPath(builder.path, builder.journal);
        this.revisionsCount = Check.revisionsToCheckCount(builder.revisionsCount);
    }

    private static File journalPath(File segmentStore, File journal) {
        if (journal == null) {
            return new File(segmentStore, "journal.log");
        }
        return journal;
    }

    private static Integer revisionsToCheckCount(Integer revisionsCount) {
        return revisionsCount != null ? revisionsCount : Integer.MAX_VALUE;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public int run() {
        StatisticsIOMonitor ioMonitor = new StatisticsIOMonitor();
        FileStoreBuilder builder = FileStoreBuilder.fileStoreBuilder(this.path).withMemoryMapping(this.mmap).withCustomPersistence(new TarPersistence(this.path, this.journal));
        if (this.ioStatistics) {
            builder.withIOMonitor(ioMonitor);
        }
        try (ReadOnlyFileStore store = builder.buildReadOnly();){
            int n;
            try (JournalReader journal = new JournalReader(new LocalJournalFile(this.journal));){
                int result = this.run(store, journal);
                if (this.ioStatistics) {
                    this.print("[I/O] Segment read: Number of operations: {0}", ioMonitor.ops.get());
                    this.print("[I/O] Segment read: Total size: {0} ({1} bytes)", IOUtils.humanReadableByteCount((long)ioMonitor.bytes.get()), ioMonitor.bytes.get());
                    this.print("[I/O] Segment read: Total time: {0} ns", ioMonitor.time.get());
                }
                if (this.repoStatistics != null) {
                    this.repoStatistics.headNodeCount = this.headNodeCount;
                    this.repoStatistics.headPropertyCount = this.headPropertyCount;
                }
                n = result;
            }
            return n;
        }
        catch (Exception e) {
            e.printStackTrace(this.err);
            return 1;
        }
    }

    private int run(ReadOnlyFileStore store, JournalReader journal) {
        LinkedHashSet checkpoints = this.requestedCheckpoints;
        if (this.requestedCheckpoints.contains("all")) {
            checkpoints = Sets.newLinkedHashSet(SegmentNodeStoreBuilders.builder(store).build().checkpoints());
        }
        ConsistencyChecker.ConsistencyCheckResult result = this.newConsistencyChecker().checkConsistency(store, (Iterator<JournalEntry>)((Object)journal), this.checkHead, checkpoints, this.filterPaths, this.checkBinaries, this.revisionsCount);
        this.print("\nSearched through {0} revisions and {1} checkpoints", result.getCheckedRevisionsCount(), checkpoints.size());
        if (Check.hasAnyRevision(result)) {
            if (this.checkHead) {
                this.print("\nHead", new Object[0]);
                for (Map.Entry entry : result.getHeadRevisions().entrySet()) {
                    this.printRevision(0, (String)entry.getKey(), (ConsistencyChecker.Revision)entry.getValue());
                }
            }
            if (checkpoints.size() > 0) {
                this.print("\nCheckpoints", new Object[0]);
                for (String string : result.getCheckpointRevisions().keySet()) {
                    this.print("- {0}", string);
                    for (Map.Entry<String, ConsistencyChecker.Revision> e : result.getCheckpointRevisions().get(string).entrySet()) {
                        this.printRevision(2, e.getKey(), e.getValue());
                    }
                }
            }
            this.print("\nOverall", new Object[0]);
            this.printOverallRevision(result.getOverallRevision());
            return 0;
        }
        this.print("No good revision found", new Object[0]);
        return 1;
    }

    private ConsistencyChecker newConsistencyChecker() {
        return new ConsistencyChecker(){

            @Override
            protected void onCheckRevision(String revision) {
                Check.this.print("\nChecking revision {0}", new Object[]{revision});
            }

            @Override
            protected void onCheckHead() {
                Check.this.headNodeCount = 0;
                Check.this.headPropertyCount = 0;
                Check.this.print("\nChecking head\n", new Object[0]);
            }

            @Override
            protected void onCheckChekpoints() {
                Check.this.print("\nChecking checkpoints", new Object[0]);
            }

            @Override
            protected void onCheckCheckpoint(String checkpoint) {
                Check.this.print("\nChecking checkpoint {0}", new Object[]{checkpoint});
            }

            @Override
            protected void onCheckpointNotFoundInRevision(String checkpoint) {
                Check.this.printError("Checkpoint {0} not found in this revision!", new Object[]{checkpoint});
            }

            @Override
            protected void onCheckRevisionError(String revision, Exception e) {
                Check.this.printError("Skipping invalid record id {0}: {1}", new Object[]{revision, e});
            }

            @Override
            protected void onConsistentPath(String path) {
                Check.this.print("Path {0} is consistent", new Object[]{path});
            }

            @Override
            protected void onPathNotFound(String path) {
                Check.this.printError("Path {0} not found", new Object[]{path});
            }

            @Override
            protected void onCheckTree(String path, boolean head) {
                Check.this.currentNodeCount = 0;
                Check.this.currentPropertyCount = 0;
                Check.this.print("Checking {0}", new Object[]{path});
            }

            @Override
            protected void onCheckTreeEnd(boolean head) {
                if (head) {
                    Check.this.headNodeCount += Check.this.currentNodeCount;
                    Check.this.headPropertyCount += Check.this.currentPropertyCount;
                }
                Check.this.print("Checked {0} nodes and {1} properties", new Object[]{Check.this.currentNodeCount, Check.this.currentPropertyCount});
            }

            @Override
            protected void onCheckNode(String path) {
                Check.this.debug("Traversing {0}", new Object[]{path});
                Check.this.currentNodeCount++;
            }

            @Override
            protected void onCheckProperty() {
                Check.this.currentPropertyCount++;
            }

            @Override
            protected void onCheckPropertyEnd(String path, PropertyState property) {
                Check.this.debug("Checked {0}/{1}", new Object[]{path, property});
            }

            @Override
            protected void onCheckNodeError(String path, Exception e) {
                Check.this.printError("Error while traversing {0}: {1}", new Object[]{path, e});
            }

            @Override
            protected void onCheckTreeError(String path, Exception e) {
                Check.this.printError("Error while traversing {0}: {1}", new Object[]{path, e.getMessage()});
            }
        };
    }

    private void print(String format, Object ... arguments) {
        this.out.println(MessageFormat.format(format, arguments));
    }

    private void printError(String format, Object ... args) {
        this.err.println(MessageFormat.format(format, args));
    }

    private void debug(String format, Object ... arg) {
        if (this.debug()) {
            this.print(format, arg);
        }
    }

    private boolean debug() {
        if (this.debugInterval == Long.MAX_VALUE) {
            return false;
        }
        if (this.debugInterval == 0L) {
            return true;
        }
        long t = System.currentTimeMillis();
        if ((t - this.lastDebugEvent) / 1000L > this.debugInterval) {
            this.lastDebugEvent = t;
            return true;
        }
        return false;
    }

    private static boolean hasAnyRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return Check.hasAnyHeadRevision(result) || Check.hasAnyCheckpointRevision(result);
    }

    private static boolean hasAnyHeadRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return result.getHeadRevisions().values().stream().anyMatch(Objects::nonNull);
    }

    private static boolean hasAnyCheckpointRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return result.getCheckpointRevisions().values().stream().flatMap(m -> m.values().stream()).anyMatch(Objects::nonNull);
    }

    private void printRevision(int indent, String path, ConsistencyChecker.Revision revision) {
        Optional<ConsistencyChecker.Revision> r = Optional.ofNullable(revision);
        this.print("{0}Latest good revision for path {1} is {2} from {3}", Strings.repeat((String)" ", (int)indent), path, r.map(ConsistencyChecker.Revision::getRevision).orElse("none"), r.map(ConsistencyChecker.Revision::getTimestamp).map(Check::timestampToString).orElse("unknown time"));
    }

    private void printOverallRevision(ConsistencyChecker.Revision revision) {
        Optional<ConsistencyChecker.Revision> r = Optional.ofNullable(revision);
        this.print("Latest good revision for paths and checkpoints checked is {0} from {1}", r.map(ConsistencyChecker.Revision::getRevision).orElse("none"), r.map(ConsistencyChecker.Revision::getTimestamp).map(Check::timestampToString).orElse("unknown time"));
    }

    private static String timestampToString(long timestamp) {
        return DateFormat.getDateTimeInstance().format(new Date(timestamp));
    }

    public static class RepositoryStatistics {
        int headNodeCount;
        int headPropertyCount;

        public int getHeadNodeCount() {
            return this.headNodeCount;
        }

        public int getHeadPropertyCount() {
            return this.headPropertyCount;
        }
    }

    private static class StatisticsIOMonitor
    extends IOMonitorAdapter {
        AtomicLong ops = new AtomicLong(0L);
        AtomicLong bytes = new AtomicLong(0L);
        AtomicLong time = new AtomicLong(0L);

        private StatisticsIOMonitor() {
        }

        @Override
        public void afterSegmentRead(File file, long msb, long lsb, int length, long elapsed) {
            this.ops.incrementAndGet();
            this.bytes.addAndGet(length);
            this.time.addAndGet(elapsed);
        }
    }

    public static class Builder {
        private File path;
        private boolean mmap;
        private File journal;
        private long debugInterval = Long.MAX_VALUE;
        private boolean checkBinaries;
        private boolean checkHead;
        private Integer revisionsCount;
        private Set<String> checkpoints;
        private Set<String> filterPaths;
        private boolean ioStatistics;
        private RepositoryStatistics repoStatistics;
        private PrintWriter outWriter;
        private PrintWriter errWriter;

        private Builder() {
        }

        public Builder withPath(File path) {
            this.path = (File)Preconditions.checkNotNull((Object)path);
            return this;
        }

        public Builder withMmap(boolean mmap) {
            this.mmap = mmap;
            return this;
        }

        public Builder withJournal(File journal) {
            this.journal = (File)Preconditions.checkNotNull((Object)journal);
            return this;
        }

        public Builder withDebugInterval(long debugInterval) {
            Preconditions.checkArgument((debugInterval >= 0L ? 1 : 0) != 0);
            this.debugInterval = debugInterval;
            return this;
        }

        public Builder withCheckBinaries(boolean checkBinaries) {
            this.checkBinaries = checkBinaries;
            return this;
        }

        public Builder withCheckHead(boolean checkHead) {
            this.checkHead = checkHead;
            return this;
        }

        public Builder withRevisionsCount(Integer revisionsCount) {
            this.revisionsCount = revisionsCount;
            return this;
        }

        public Builder withCheckpoints(Set<String> checkpoints) {
            this.checkpoints = checkpoints;
            return this;
        }

        public Builder withFilterPaths(Set<String> filterPaths) {
            this.filterPaths = filterPaths;
            return this;
        }

        public Builder withIOStatistics(boolean ioStatistics) {
            this.ioStatistics = ioStatistics;
            return this;
        }

        public Builder withRepositoryStatistics(RepositoryStatistics repoStatistics) {
            this.repoStatistics = repoStatistics;
            return this;
        }

        public Builder withOutWriter(PrintWriter outWriter) {
            this.outWriter = outWriter;
            return this;
        }

        public Builder withErrWriter(PrintWriter errWriter) {
            this.errWriter = errWriter;
            return this;
        }

        public Check build() {
            Preconditions.checkNotNull((Object)this.path);
            return new Check(this);
        }
    }
}

