/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure.tool;

import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.azure.AzurePersistence;
import org.apache.jackrabbit.oak.segment.azure.tool.ToolUtils;
import org.apache.jackrabbit.oak.segment.azure.util.Retrier;
import org.apache.jackrabbit.oak.segment.file.tar.TarPersistence;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.monitor.RemoteStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.RemoteStoreMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.persistence.GCJournalFile;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileWriter;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveEntry;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveManager;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentNodeStorePersistence;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentStoreMigrator
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(SegmentStoreMigrator.class);
    private static final int READ_THREADS = 20;
    private static final Retrier RETRIER = Retrier.withParams(16, 5000);
    private final SegmentNodeStorePersistence source;
    private final SegmentNodeStorePersistence target;
    private final String sourceName;
    private final String targetName;
    private final boolean appendMode;
    private final Integer revisionCount;
    private ExecutorService executor = Executors.newFixedThreadPool(21);

    private SegmentStoreMigrator(Builder builder) {
        this.source = builder.source;
        this.target = builder.target;
        this.sourceName = builder.sourceName;
        this.targetName = builder.targetName;
        this.appendMode = builder.appendMode;
        this.revisionCount = builder.revisionCount;
    }

    public void migrate() throws IOException, ExecutionException, InterruptedException {
        RETRIER.execute(this::migrateJournal);
        RETRIER.execute(this::migrateGCJournal);
        RETRIER.execute(this::migrateManifest);
        this.migrateArchives();
    }

    private void migrateJournal() throws IOException {
        if (this.revisionCount == 0) {
            log.info("Number of revisions configured to be copied is 0. Skip copying journal.");
            return;
        }
        log.info("{}/journal.log -> {}", (Object)this.sourceName, (Object)this.targetName);
        if (!this.source.getJournalFile().exists()) {
            log.info("No journal at {}; skipping.", (Object)this.sourceName);
            return;
        }
        ArrayList<String> journal = new ArrayList<String>();
        try (JournalFileReader reader = this.source.getJournalFile().openJournalReader();){
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.length() > 0 && !line.trim().equals("")) {
                    journal.add(line);
                }
                if (journal.size() != this.revisionCount.intValue()) continue;
                break;
            }
        }
        Collections.reverse(journal);
        try (JournalFileWriter writer = this.target.getJournalFile().openJournalWriter();){
            writer.truncate();
            for (String line : journal) {
                writer.writeLine(line);
            }
        }
    }

    private void migrateGCJournal() throws IOException {
        log.info("{}/gc.log -> {}", (Object)this.sourceName, (Object)this.targetName);
        GCJournalFile targetGCJournal = this.target.getGCJournalFile();
        if (this.appendMode) {
            targetGCJournal.truncate();
        }
        for (String line : this.source.getGCJournalFile().readLines()) {
            targetGCJournal.writeLine(line);
        }
    }

    private void migrateManifest() throws IOException {
        log.info("{}/manifest -> {}", (Object)this.sourceName, (Object)this.targetName);
        if (!this.source.getManifestFile().exists()) {
            log.info("No manifest at {}; skipping.", (Object)this.sourceName);
            return;
        }
        Properties manifest = this.source.getManifestFile().load();
        this.target.getManifestFile().save(manifest);
    }

    private void migrateArchives() throws IOException, ExecutionException, InterruptedException {
        if (!this.source.segmentFilesExist()) {
            log.info("No segment archives at {}; skipping.", (Object)this.sourceName);
            return;
        }
        SegmentArchiveManager sourceManager = this.source.createArchiveManager(false, false, (IOMonitor)new IOMonitorAdapter(), (FileStoreMonitor)new FileStoreMonitorAdapter(), (RemoteStoreMonitor)new RemoteStoreMonitorAdapter());
        SegmentArchiveManager targetManager = this.target.createArchiveManager(false, false, (IOMonitor)new IOMonitorAdapter(), (FileStoreMonitor)new FileStoreMonitorAdapter(), (RemoteStoreMonitor)new RemoteStoreMonitorAdapter());
        List targetArchives = targetManager.listArchives();
        if (this.appendMode && !targetArchives.isEmpty()) {
            String lastArchive = (String)targetArchives.get(targetArchives.size() - 1);
            targetArchives.remove(lastArchive);
        }
        for (String archiveName : sourceManager.listArchives()) {
            log.info("{}/{} -> {}", new Object[]{this.sourceName, archiveName, this.targetName});
            if (this.appendMode && targetArchives.contains(archiveName)) {
                log.info("Already exists, skipping.");
                continue;
            }
            SegmentArchiveReader reader = sourceManager.forceOpen(archiveName);
            try (SegmentArchiveWriter writer = targetManager.create(archiveName);){
                this.migrateSegments(reader, writer);
                this.migrateBinaryRef(reader, writer);
                this.migrateGraph(reader, writer);
            }
            finally {
                if (reader == null) continue;
                reader.close();
            }
        }
    }

    private void migrateSegments(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws ExecutionException, InterruptedException, IOException {
        ArrayList<Future<Segment>> futures = new ArrayList<Future<Segment>>();
        for (SegmentArchiveEntry segmentArchiveEntry : reader.listSegments()) {
            futures.add(this.executor.submit(() -> RETRIER.execute(() -> {
                Segment segment = new Segment(entry);
                segment.read(reader);
                return segment;
            })));
        }
        for (Future future : futures) {
            Segment segment = (Segment)future.get();
            RETRIER.execute(() -> segment.write(writer));
        }
    }

    private void migrateBinaryRef(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws IOException, ExecutionException, InterruptedException {
        Future<Buffer> future = this.executor.submit(() -> RETRIER.execute(() -> ((SegmentArchiveReader)reader).getBinaryReferences()));
        Buffer binaryReferences = future.get();
        if (binaryReferences != null) {
            byte[] array = ToolUtils.fetchByteArray(binaryReferences);
            RETRIER.execute(() -> writer.writeBinaryReferences(array));
        }
    }

    private void migrateGraph(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws IOException, ExecutionException, InterruptedException {
        Future<Buffer> future = this.executor.submit(() -> RETRIER.execute(() -> {
            if (reader.hasGraph()) {
                return reader.getGraph();
            }
            return null;
        }));
        Buffer graph = future.get();
        if (graph != null) {
            byte[] array = ToolUtils.fetchByteArray(graph);
            RETRIER.execute(() -> writer.writeGraph(array));
        }
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdown();
        try {
            while (!this.executor.awaitTermination(100L, TimeUnit.MILLISECONDS)) {
            }
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    public static class Builder {
        private SegmentNodeStorePersistence source;
        private SegmentNodeStorePersistence target;
        private String sourceName;
        private String targetName;
        private boolean appendMode;
        private Integer revisionCount = Integer.MAX_VALUE;

        public Builder withSource(File dir) {
            this.source = new TarPersistence(dir);
            this.sourceName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.TAR, dir.getPath());
            return this;
        }

        public Builder withSource(CloudBlobDirectory dir) throws URISyntaxException, StorageException {
            this.source = new AzurePersistence(dir);
            this.sourceName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.AZURE, dir.getContainer().getName() + "/" + dir.getPrefix());
            return this;
        }

        public Builder withSourcePersistence(SegmentNodeStorePersistence source, String sourceName) {
            this.source = source;
            this.sourceName = sourceName;
            return this;
        }

        public Builder withTargetPersistence(SegmentNodeStorePersistence target, String targetName) {
            this.target = target;
            this.targetName = targetName;
            return this;
        }

        public Builder withTarget(File dir) {
            this.target = new TarPersistence(dir);
            this.targetName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.TAR, dir.getPath());
            return this;
        }

        public Builder withTarget(CloudBlobDirectory dir) throws URISyntaxException, StorageException {
            this.target = new AzurePersistence(dir);
            this.targetName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.AZURE, dir.getContainer().getName() + "/" + dir.getPrefix());
            return this;
        }

        public Builder setAppendMode() {
            this.appendMode = true;
            return this;
        }

        public Builder withRevisionCount(Integer revisionCount) {
            this.revisionCount = revisionCount;
            return this;
        }

        public SegmentStoreMigrator build() {
            return new SegmentStoreMigrator(this);
        }
    }

    static class Segment {
        final SegmentArchiveEntry entry;
        volatile Buffer data;

        Segment(SegmentArchiveEntry entry) {
            this.entry = entry;
        }

        void read(SegmentArchiveReader reader) throws IOException {
            this.data = reader.readSegment(this.entry.getMsb(), this.entry.getLsb());
        }

        void write(SegmentArchiveWriter writer) throws IOException {
            byte[] array = this.data.array();
            boolean offset = false;
            writer.writeSegment(this.entry.getMsb(), this.entry.getLsb(), array, 0, this.entry.getLength(), this.entry.getGeneration(), this.entry.getFullGeneration(), this.entry.isCompacted());
        }

        public String toString() {
            return new UUID(this.entry.getMsb(), this.entry.getLsb()).toString();
        }
    }
}

