/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure;

import com.microsoft.azure.storage.AccessCondition;
import com.microsoft.azure.storage.RetryNoRetry;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobRequestOptions;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.segment.remote.WriteAccessController;
import org.apache.jackrabbit.oak.segment.spi.persistence.RepositoryLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureRepositoryLock
implements RepositoryLock {
    private static final Logger log = LoggerFactory.getLogger(AzureRepositoryLock.class);
    private static final int TIMEOUT_SEC = Integer.getInteger("oak.segment.azure.lock.timeout", 0);
    private static final Integer LEASE_RENEWAL_TIMEOUT_MS = 5000;
    public static final String LEASE_DURATION_PROP = "oak.segment.azure.lock.leaseDurationInSec";
    private final int leaseDuration = Integer.getInteger("oak.segment.azure.lock.leaseDurationInSec", 60);
    public static final String RENEWAL_INTERVAL_PROP = "oak.segment.azure.lock.leaseRenewalIntervalInSec";
    private final int renewalInterval = Integer.getInteger("oak.segment.azure.lock.leaseRenewalIntervalInSec", 5);
    public static final String TIME_TO_WAIT_BEFORE_WRITE_BLOCK_PROP = "oak.segment.azure.lock.blockWritesAfterInSec";
    private final int timeToWaitBeforeWriteBlock = Integer.getInteger("oak.segment.azure.lock.blockWritesAfterInSec", 20);
    private final Runnable shutdownHook;
    private final CloudBlockBlob blob;
    private final ExecutorService executor;
    private final int timeoutSec;
    private WriteAccessController writeAccessController;
    private String leaseId;
    private volatile boolean doUpdate;

    public AzureRepositoryLock(CloudBlockBlob blob, Runnable shutdownHook, WriteAccessController writeAccessController) {
        this(blob, shutdownHook, writeAccessController, TIMEOUT_SEC);
    }

    public AzureRepositoryLock(CloudBlockBlob blob, Runnable shutdownHook, WriteAccessController writeAccessController, int timeoutSec) {
        this.shutdownHook = shutdownHook;
        this.blob = blob;
        this.executor = Executors.newSingleThreadExecutor();
        this.timeoutSec = timeoutSec;
        this.writeAccessController = writeAccessController;
        if (this.leaseDuration < this.timeToWaitBeforeWriteBlock || this.timeToWaitBeforeWriteBlock < this.renewalInterval) {
            throw new IllegalStateException(String.format("The value of %s must be greater than %s and the value of %s must be greater than %s", LEASE_DURATION_PROP, TIME_TO_WAIT_BEFORE_WRITE_BLOCK_PROP, TIME_TO_WAIT_BEFORE_WRITE_BLOCK_PROP, RENEWAL_INTERVAL_PROP));
        }
    }

    public AzureRepositoryLock lock() throws IOException {
        long start = System.currentTimeMillis();
        Exception ex = null;
        do {
            try {
                this.blob.openOutputStream().close();
                log.info("{} = {}", (Object)LEASE_DURATION_PROP, (Object)this.leaseDuration);
                log.info("{} = {}", (Object)RENEWAL_INTERVAL_PROP, (Object)this.renewalInterval);
                log.info("{} = {}", (Object)TIME_TO_WAIT_BEFORE_WRITE_BLOCK_PROP, (Object)this.timeToWaitBeforeWriteBlock);
                this.leaseId = this.blob.acquireLease(this.leaseDuration, null);
                this.writeAccessController.enableWriting();
                log.info("Acquired lease {}", (Object)this.leaseId);
            }
            catch (StorageException | IOException e) {
                if (ex == null) {
                    log.info("Can't acquire the lease. Retrying every 1s. Timeout is set to {}s.", (Object)this.timeoutSec);
                }
                ex = e;
                if ((System.currentTimeMillis() - start) / 1000L >= (long)this.timeoutSec) break;
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e1) {
                    throw new IOException(e1);
                }
            }
        } while (this.leaseId == null);
        if (this.leaseId == null) {
            log.error("Can't acquire the lease in {}s.", (Object)this.timeoutSec);
            throw new IOException(ex);
        }
        this.executor.submit(this::refreshLease);
        return this;
    }

    private void refreshLease() {
        this.doUpdate = true;
        long lastUpdate = 0L;
        while (this.doUpdate) {
            long timeSinceLastUpdate = (System.currentTimeMillis() - lastUpdate) / 1000L;
            try {
                if (timeSinceLastUpdate > (long)this.renewalInterval) {
                    BlobRequestOptions requestOptions = new BlobRequestOptions();
                    requestOptions.setMaximumExecutionTimeInMs(LEASE_RENEWAL_TIMEOUT_MS);
                    requestOptions.setRetryPolicyFactory(new RetryNoRetry());
                    this.blob.renewLease(AccessCondition.generateLeaseCondition(this.leaseId), requestOptions, null);
                    this.writeAccessController.enableWriting();
                    lastUpdate = System.currentTimeMillis();
                }
            }
            catch (StorageException e) {
                timeSinceLastUpdate = (System.currentTimeMillis() - lastUpdate) / 1000L;
                if (e.getErrorCode().equals("OperationTimedOut")) {
                    if (timeSinceLastUpdate > (long)this.timeToWaitBeforeWriteBlock) {
                        this.writeAccessController.disableWriting();
                    }
                    log.warn("Could not renew the lease due to the operation timeout. Retry in progress ...", (Throwable)e);
                }
                log.error("Can't renew the lease", (Throwable)e);
                this.shutdownHook.run();
                this.doUpdate = false;
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                log.error("Interrupted the lease renewal loop", (Throwable)e);
            }
        }
    }

    public void unlock() throws IOException {
        this.doUpdate = false;
        this.executor.shutdown();
        try {
            this.executor.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
        finally {
            this.releaseLease();
        }
    }

    private void releaseLease() throws IOException {
        try {
            this.blob.releaseLease(AccessCondition.generateLeaseCondition(this.leaseId));
            this.blob.delete();
            log.info("Released lease {}", (Object)this.leaseId);
            this.leaseId = null;
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }
}

