/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.hudi

import org.apache.hudi.DataSourceReadOptions.INCREMENTAL_READ_HANDLE_HOLLOW_COMMIT
import org.apache.hudi.common.table.timeline.TimelineUtils.HollowCommitHandling
import org.apache.spark.sql.internal.SQLConf

/**
 * Util methods for Hudi Spark and SQL configurations
 */
object HoodieSparkConfUtils {
  /**
   * Gets boolean config value from config properties and SQL conf.
   *
   * @param options      Config properties.
   * @param sqlConf      SQL conf.
   * @param configKey    Config key to fetch.
   * @param defaultValue Default value to return if not configured.
   * @return The config value.
   */
  def getConfigValue(options: Map[String, String],
                     sqlConf: SQLConf,
                     configKey: String,
                     defaultValue: String): String = {
    options.getOrElse(configKey, sqlConf.getConfString(configKey, defaultValue))
  }

  def getHollowCommitHandling(options: Map[String, String]): HollowCommitHandling = {
    options.get(INCREMENTAL_READ_HANDLE_HOLLOW_COMMIT.key)
      .map(HollowCommitHandling.valueOf)
      .getOrElse(HollowCommitHandling.valueOf(INCREMENTAL_READ_HANDLE_HOLLOW_COMMIT.defaultValue))
  }
}
