/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.ql.security.authorization.plugin;

import org.apache.hadoop.hive.common.classification.InterfaceAudience.LimitedPrivate;
import org.apache.hadoop.hive.common.classification.InterfaceStability.Evolving;

/**
 * Represents the user or role in grant/revoke statements
 */
@LimitedPrivate(value = { "Apache Argus (incubating)" })
@Evolving
public class HivePrincipal implements Comparable<HivePrincipal> {

  @Override
  public int compareTo(HivePrincipal o) {
    int compare = name.compareTo(o.name);
    if (compare == 0) {
      compare = type.compareTo(o.type);
    }
    return compare;
  }

  public enum HivePrincipalType{
    USER, GROUP, ROLE, UNKNOWN
  }

  @Override
  public String toString() {
    return "Principal [name=" + name + ", type=" + type + "]";
  }

  private final String name;
  private final HivePrincipalType type;

  public HivePrincipal(String name, HivePrincipalType type){
    this.type = type;
    this.name = name;
  }

  public String getName() {
    return name;
  }
  public HivePrincipalType getType() {
    return type;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((name == null) ? 0 : name.hashCode());
    result = prime * result + ((type == null) ? 0 : type.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    HivePrincipal other = (HivePrincipal) obj;
    if (name == null) {
      if (other.name != null)
        return false;
    } else if (!name.equals(other.name))
      return false;
    if (type != other.type)
      return false;
    return true;
  }

}
