/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.org.apache.hadoop.hbase.regionserver;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.util.StringUtils;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.Chunk;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.CompactingMemStore;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.HeapMemoryManager;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.OffheapChunk;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.OnheapChunk;
import org.apache.hudi.org.apache.hbase.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ChunkCreator {
    private static final Logger LOG = LoggerFactory.getLogger(ChunkCreator.class);
    private AtomicInteger chunkID = new AtomicInteger(1);
    public static final int SIZEOF_CHUNK_HEADER = 4;
    private Map<Integer, Chunk> chunkIdMap = new ConcurrentHashMap<Integer, Chunk>();
    private final boolean offheap;
    static ChunkCreator instance;
    static boolean chunkPoolDisabled;
    private MemStoreChunkPool dataChunksPool;
    private final int chunkSize;
    private int indexChunkSize;
    private MemStoreChunkPool indexChunksPool;

    ChunkCreator(int chunkSize, boolean offheap, long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage, HeapMemoryManager heapMemoryManager, float indexChunkSizePercentage) {
        this.offheap = offheap;
        this.chunkSize = chunkSize;
        this.initializePools(chunkSize, globalMemStoreSize, poolSizePercentage, indexChunkSizePercentage, initialCountPercentage, heapMemoryManager);
    }

    private void initializePools(int chunkSize, long globalMemStoreSize, float poolSizePercentage, float indexChunkSizePercentage, float initialCountPercentage, HeapMemoryManager heapMemoryManager) {
        this.dataChunksPool = this.initializePool("data", globalMemStoreSize, (1.0f - indexChunkSizePercentage) * poolSizePercentage, initialCountPercentage, chunkSize, ChunkType.DATA_CHUNK, heapMemoryManager);
        this.indexChunkSize = (int)(indexChunkSizePercentage * (float)chunkSize);
        this.indexChunksPool = this.initializePool("index", globalMemStoreSize, indexChunkSizePercentage * poolSizePercentage, initialCountPercentage, this.indexChunkSize, ChunkType.INDEX_CHUNK, heapMemoryManager);
    }

    @SuppressWarnings(value={"LI_LAZY_INIT_STATIC"}, justification="Method is called by single thread at the starting of RS")
    public static ChunkCreator initialize(int chunkSize, boolean offheap, long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage, HeapMemoryManager heapMemoryManager, float indexChunkSizePercent) {
        if (instance != null) {
            return instance;
        }
        instance = new ChunkCreator(chunkSize, offheap, globalMemStoreSize, poolSizePercentage, initialCountPercentage, heapMemoryManager, indexChunkSizePercent);
        return instance;
    }

    public static ChunkCreator getInstance() {
        return instance;
    }

    Chunk getChunk(ChunkType chunkType) {
        return this.getChunk(CompactingMemStore.IndexType.ARRAY_MAP, chunkType);
    }

    Chunk getChunk() {
        return this.getChunk(CompactingMemStore.IndexType.ARRAY_MAP, ChunkType.DATA_CHUNK);
    }

    Chunk getChunk(CompactingMemStore.IndexType chunkIndexType) {
        return this.getChunk(chunkIndexType, ChunkType.DATA_CHUNK);
    }

    Chunk getChunk(CompactingMemStore.IndexType chunkIndexType, ChunkType chunkType) {
        switch (chunkType) {
            case INDEX_CHUNK: {
                if (this.indexChunksPool == null) {
                    if (this.indexChunkSize <= 0) {
                        throw new IllegalArgumentException("chunkType is INDEX_CHUNK but indexChunkSize is:[" + this.indexChunkSize + "]");
                    }
                    return this.getChunk(chunkIndexType, chunkType, this.indexChunkSize);
                }
                return this.getChunk(chunkIndexType, chunkType, this.indexChunksPool.getChunkSize());
            }
            case DATA_CHUNK: {
                if (this.dataChunksPool == null) {
                    return this.getChunk(chunkIndexType, chunkType, this.chunkSize);
                }
                return this.getChunk(chunkIndexType, chunkType, this.dataChunksPool.getChunkSize());
            }
        }
        throw new IllegalArgumentException("chunkType must either be INDEX_CHUNK or DATA_CHUNK");
    }

    Chunk getChunk(CompactingMemStore.IndexType chunkIndexType, ChunkType chunkType, int size) {
        Chunk chunk = null;
        MemStoreChunkPool pool = null;
        if (this.dataChunksPool != null && chunkType == ChunkType.DATA_CHUNK) {
            pool = this.dataChunksPool;
        } else if (this.indexChunksPool != null && chunkType == ChunkType.INDEX_CHUNK) {
            pool = this.indexChunksPool;
        }
        if (pool != null && (chunk = pool.getChunk()) == null && LOG.isTraceEnabled()) {
            LOG.trace("The chunk pool is full. Reached maxCount= " + pool.getMaxCount() + ". Creating chunk onheap.");
        }
        if (chunk == null) {
            chunk = this.createChunk(false, chunkIndexType, chunkType, size);
        }
        chunk.init();
        return chunk;
    }

    Chunk getJumboChunk(int jumboSize) {
        int allocSize = jumboSize + 4;
        if (allocSize <= this.getChunkSize(ChunkType.DATA_CHUNK)) {
            LOG.warn("Jumbo chunk size " + jumboSize + " must be more than regular chunk size " + this.getChunkSize(ChunkType.DATA_CHUNK) + ". Converting to regular chunk.");
            return this.getChunk(CompactingMemStore.IndexType.CHUNK_MAP);
        }
        return this.getChunk(CompactingMemStore.IndexType.CHUNK_MAP, ChunkType.JUMBO_CHUNK, allocSize);
    }

    private Chunk createChunk(boolean pool, CompactingMemStore.IndexType chunkIndexType, ChunkType chunkType, int size) {
        Chunk chunk = null;
        int id = this.chunkID.getAndIncrement();
        assert (id > 0);
        chunk = pool && this.offheap ? new OffheapChunk(size, id, chunkType, pool) : new OnheapChunk(size, id, chunkType, pool);
        if (pool || chunkIndexType == CompactingMemStore.IndexType.CHUNK_MAP) {
            this.chunkIdMap.put(chunk.getId(), chunk);
        }
        return chunk;
    }

    private Chunk createChunkForPool(CompactingMemStore.IndexType chunkIndexType, ChunkType chunkType, int chunkSize) {
        if (chunkSize != this.dataChunksPool.getChunkSize() && chunkSize != this.indexChunksPool.getChunkSize()) {
            return null;
        }
        return this.createChunk(true, chunkIndexType, chunkType, chunkSize);
    }

    Chunk getChunk(int id) {
        return this.chunkIdMap.get(id);
    }

    boolean isOffheap() {
        return this.offheap;
    }

    private void removeChunks(Set<Integer> chunkIDs) {
        this.chunkIdMap.keySet().removeAll(chunkIDs);
    }

    Chunk removeChunk(int chunkId) {
        return this.chunkIdMap.remove(chunkId);
    }

    int numberOfMappedChunks() {
        return this.chunkIdMap.size();
    }

    void clearChunkIds() {
        this.chunkIdMap.clear();
    }

    static void clearDisableFlag() {
        chunkPoolDisabled = false;
    }

    private MemStoreChunkPool initializePool(String label, long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage, int chunkSize, ChunkType chunkType, HeapMemoryManager heapMemoryManager) {
        if (poolSizePercentage <= 0.0f) {
            LOG.info("{} poolSizePercentage is less than 0. So not using pool", (Object)label);
            return null;
        }
        if (chunkPoolDisabled) {
            return null;
        }
        if ((double)poolSizePercentage > 1.0) {
            throw new IllegalArgumentException("hbase.hregion.memstore.chunkpool.maxsize must be between 0.0 and 1.0");
        }
        int maxCount = (int)((float)globalMemStoreSize * poolSizePercentage / (float)chunkSize);
        if ((double)initialCountPercentage > 1.0 || initialCountPercentage < 0.0f) {
            throw new IllegalArgumentException(label + " " + "hbase.hregion.memstore.chunkpool.initialsize" + " must be between 0.0 and 1.0");
        }
        int initialCount = (int)(initialCountPercentage * (float)maxCount);
        LOG.info("Allocating {} MemStoreChunkPool with chunk size {}, max count {}, initial count {}", new Object[]{label, StringUtils.byteDesc((long)chunkSize), maxCount, initialCount});
        MemStoreChunkPool memStoreChunkPool = new MemStoreChunkPool(label, chunkSize, chunkType, maxCount, initialCount, poolSizePercentage);
        if (heapMemoryManager != null && memStoreChunkPool != null) {
            heapMemoryManager.registerTuneObserver(memStoreChunkPool);
        }
        return memStoreChunkPool;
    }

    int getMaxCount() {
        return this.getMaxCount(ChunkType.DATA_CHUNK);
    }

    int getMaxCount(ChunkType chunkType) {
        switch (chunkType) {
            case INDEX_CHUNK: {
                if (this.indexChunksPool == null) break;
                return this.indexChunksPool.getMaxCount();
            }
            case DATA_CHUNK: {
                if (this.dataChunksPool == null) break;
                return this.dataChunksPool.getMaxCount();
            }
            default: {
                throw new IllegalArgumentException("chunkType must either be INDEX_CHUNK or DATA_CHUNK");
            }
        }
        return 0;
    }

    int getPoolSize() {
        return this.getPoolSize(ChunkType.DATA_CHUNK);
    }

    int getPoolSize(ChunkType chunkType) {
        switch (chunkType) {
            case INDEX_CHUNK: {
                if (this.indexChunksPool == null) break;
                return this.indexChunksPool.reclaimedChunks.size();
            }
            case DATA_CHUNK: {
                if (this.dataChunksPool == null) break;
                return this.dataChunksPool.reclaimedChunks.size();
            }
            default: {
                throw new IllegalArgumentException("chunkType must either be INDEX_CHUNK or DATA_CHUNK");
            }
        }
        return 0;
    }

    boolean isChunkInPool(int chunkId) {
        Chunk c = this.getChunk(chunkId);
        if (c == null) {
            return false;
        }
        if (this.dataChunksPool != null && this.dataChunksPool.reclaimedChunks.contains(c)) {
            return true;
        }
        return this.indexChunksPool != null && this.indexChunksPool.reclaimedChunks.contains(c);
    }

    void clearChunksInPool() {
        if (this.dataChunksPool != null) {
            this.dataChunksPool.reclaimedChunks.clear();
        }
        if (this.indexChunksPool != null) {
            this.indexChunksPool.reclaimedChunks.clear();
        }
    }

    int getChunkSize() {
        return this.getChunkSize(ChunkType.DATA_CHUNK);
    }

    int getChunkSize(ChunkType chunkType) {
        switch (chunkType) {
            case INDEX_CHUNK: {
                if (this.indexChunksPool != null) {
                    return this.indexChunksPool.getChunkSize();
                }
                return this.indexChunkSize;
            }
            case DATA_CHUNK: {
                if (this.dataChunksPool != null) {
                    return this.dataChunksPool.getChunkSize();
                }
                return this.chunkSize;
            }
        }
        throw new IllegalArgumentException("chunkType must either be INDEX_CHUNK or DATA_CHUNK");
    }

    synchronized void putbackChunks(Set<Integer> chunks) {
        if (this.dataChunksPool == null && this.indexChunksPool == null) {
            this.removeChunks(chunks);
            return;
        }
        for (int chunkID : chunks) {
            Chunk chunk = this.getChunk(chunkID);
            if (chunk == null) continue;
            if (chunk.isFromPool() && chunk.isIndexChunk()) {
                this.indexChunksPool.putbackChunks(chunk);
                continue;
            }
            if (chunk.isFromPool() && chunk.isDataChunk()) {
                this.dataChunksPool.putbackChunks(chunk);
                continue;
            }
            this.removeChunk(chunkID);
        }
    }

    MemStoreChunkPool getIndexChunksPool() {
        return this.indexChunksPool;
    }

    MemStoreChunkPool getDataChunksPool() {
        return this.dataChunksPool;
    }

    static {
        chunkPoolDisabled = false;
    }

    private class MemStoreChunkPool
    implements HeapMemoryManager.HeapMemoryTuneObserver {
        private final int chunkSize;
        private final ChunkType chunkType;
        private int maxCount;
        private final BlockingQueue<Chunk> reclaimedChunks;
        private final float poolSizePercentage;
        private final ScheduledExecutorService scheduleThreadPool;
        private static final int statThreadPeriod = 300;
        private final AtomicLong chunkCount = new AtomicLong();
        private final LongAdder reusedChunkCount = new LongAdder();
        private final String label;

        MemStoreChunkPool(String label, int chunkSize, ChunkType chunkType, int maxCount, int initialCount, float poolSizePercentage) {
            this.label = label;
            this.chunkSize = chunkSize;
            this.chunkType = chunkType;
            this.maxCount = maxCount;
            this.poolSizePercentage = poolSizePercentage;
            this.reclaimedChunks = new LinkedBlockingQueue<Chunk>();
            for (int i = 0; i < initialCount; ++i) {
                Chunk chunk = ChunkCreator.this.createChunk(true, CompactingMemStore.IndexType.ARRAY_MAP, chunkType, chunkSize);
                chunk.init();
                this.reclaimedChunks.add(chunk);
            }
            this.chunkCount.set(initialCount);
            String n = Thread.currentThread().getName();
            this.scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(n + "-MemStoreChunkPool Statistics").setDaemon(true).build());
            this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(), 300L, 300L, TimeUnit.SECONDS);
        }

        Chunk getChunk() {
            return this.getChunk(CompactingMemStore.IndexType.ARRAY_MAP);
        }

        Chunk getChunk(CompactingMemStore.IndexType chunkIndexType) {
            Chunk chunk = (Chunk)this.reclaimedChunks.poll();
            if (chunk != null) {
                chunk.reset();
                this.reusedChunkCount.increment();
            } else {
                long created;
                while ((created = this.chunkCount.get()) < (long)this.maxCount) {
                    if (!this.chunkCount.compareAndSet(created, created + 1L)) continue;
                    chunk = ChunkCreator.this.createChunkForPool(chunkIndexType, this.chunkType, this.chunkSize);
                    break;
                }
            }
            return chunk;
        }

        int getChunkSize() {
            return this.chunkSize;
        }

        private void putbackChunks(Chunk c) {
            int toAdd = this.maxCount - this.reclaimedChunks.size();
            if (c.isFromPool() && c.size == this.chunkSize && toAdd > 0) {
                this.reclaimedChunks.add(c);
            } else {
                ChunkCreator.this.removeChunk(c.getId());
            }
        }

        private int getMaxCount() {
            return this.maxCount;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onHeapMemoryTune(long newMemstoreSize, long newBlockCacheSize) {
            if (ChunkCreator.this.isOffheap()) {
                LOG.warn("{} not tuning the chunk pool as it is offheap", (Object)this.label);
                return;
            }
            int newMaxCount = (int)((float)newMemstoreSize * this.poolSizePercentage / (float)this.getChunkSize());
            if (newMaxCount != this.maxCount) {
                if (newMaxCount > this.maxCount) {
                    LOG.info("{} max count for chunks increased from {} to {}", new Object[]{this.label, this.maxCount, newMaxCount});
                    this.maxCount = newMaxCount;
                } else {
                    LOG.info("{} max count for chunks decreased from {} to {}", new Object[]{this.label, this.maxCount, newMaxCount});
                    this.maxCount = newMaxCount;
                    if (this.reclaimedChunks.size() > newMaxCount) {
                        MemStoreChunkPool memStoreChunkPool = this;
                        synchronized (memStoreChunkPool) {
                            while (this.reclaimedChunks.size() > newMaxCount) {
                                this.reclaimedChunks.poll();
                            }
                        }
                    }
                }
            }
        }

        private class StatisticsThread
        extends Thread {
            StatisticsThread() {
                super("MemStoreChunkPool.StatisticsThread");
                this.setDaemon(true);
            }

            @Override
            public void run() {
                this.logStats();
            }

            private void logStats() {
                if (!LOG.isDebugEnabled()) {
                    return;
                }
                long created = MemStoreChunkPool.this.chunkCount.get();
                long reused = MemStoreChunkPool.this.reusedChunkCount.sum();
                long total = created + reused;
                LOG.debug("{} stats (chunk size={}): current pool size={}, created chunk count={}, reused chunk count={}, reuseRatio={}", new Object[]{MemStoreChunkPool.this.label, MemStoreChunkPool.this.chunkSize, MemStoreChunkPool.this.reclaimedChunks.size(), created, reused, total == 0L ? "0" : StringUtils.formatPercent((double)((float)reused / (float)total), (int)2)});
            }
        }
    }

    public static enum ChunkType {
        INDEX_CHUNK,
        DATA_CHUNK,
        JUMBO_CHUNK;

    }
}

