/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import org.apache.hudi.common.util.ReflectionUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.exception.HoodieException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class RetryHelper<T>
implements Serializable {
    private static final Logger LOG = LogManager.getLogger(RetryHelper.class);
    private transient CheckedFunction<T> func;
    private final int num;
    private final long maxIntervalTime;
    private final long initialIntervalTime;
    private String taskInfo = "N/A";
    private List<? extends Class<? extends Exception>> retryExceptionsClasses;

    public RetryHelper(long maxRetryIntervalMs, int maxRetryNumbers, long initialRetryIntervalMs, String retryExceptions) {
        this.num = maxRetryNumbers;
        this.initialIntervalTime = initialRetryIntervalMs;
        this.maxIntervalTime = maxRetryIntervalMs;
        if (StringUtils.isNullOrEmpty(retryExceptions)) {
            this.retryExceptionsClasses = new ArrayList<Class<? extends Exception>>();
        } else {
            try {
                this.retryExceptionsClasses = Arrays.stream(retryExceptions.split(",")).map(exception -> (Exception)ReflectionUtils.loadClass(exception, "")).map(Object::getClass).collect(Collectors.toList());
            }
            catch (HoodieException e) {
                LOG.error((Object)("Exception while loading retry exceptions classes '" + retryExceptions + "'."), (Throwable)e);
                this.retryExceptionsClasses = new ArrayList<Class<? extends Exception>>();
            }
        }
    }

    public RetryHelper(long maxRetryIntervalMs, int maxRetryNumbers, long initialRetryIntervalMs, String retryExceptions, String taskInfo) {
        this(maxRetryIntervalMs, maxRetryNumbers, initialRetryIntervalMs, retryExceptions);
        this.taskInfo = taskInfo;
    }

    public RetryHelper<T> tryWith(CheckedFunction<T> func) {
        this.func = func;
        return this;
    }

    public T start(CheckedFunction<T> func) throws IOException {
        int retries = 0;
        T functionResult = null;
        while (true) {
            long waitTime = Math.min(this.getWaitTimeExp(retries), this.maxIntervalTime);
            try {
                functionResult = func.get();
            }
            catch (IOException | RuntimeException e) {
                if (!this.checkIfExceptionInRetryList(e)) {
                    throw e;
                }
                if (retries++ >= this.num) {
                    String message = "Still failed to " + this.taskInfo + " after retried " + this.num + " times.";
                    LOG.error((Object)message, (Throwable)e);
                    if (e instanceof IOException) {
                        throw new IOException(message, e);
                    }
                    throw e;
                }
                LOG.warn((Object)("Catch Exception for " + this.taskInfo + ", will retry after " + waitTime + " ms."), (Throwable)e);
                try {
                    Thread.sleep(waitTime);
                }
                catch (InterruptedException interruptedException) {}
                continue;
            }
            break;
        }
        if (retries > 0) {
            LOG.info((Object)("Success to " + this.taskInfo + " after retried " + retries + " times."));
        }
        return functionResult;
    }

    public T start() throws IOException {
        return this.start(this.func);
    }

    private boolean checkIfExceptionInRetryList(Exception e) {
        boolean inRetryList = false;
        if (this.retryExceptionsClasses.isEmpty()) {
            return true;
        }
        for (Class<? extends Exception> clazz : this.retryExceptionsClasses) {
            if (!clazz.isInstance(e)) continue;
            inRetryList = true;
            break;
        }
        return inRetryList;
    }

    private long getWaitTimeExp(int retryCount) {
        Random random = new Random();
        if (0 == retryCount) {
            return this.initialIntervalTime;
        }
        return (long)Math.pow(2.0, retryCount) * this.initialIntervalTime + (long)random.nextInt(100);
    }

    @FunctionalInterface
    public static interface CheckedFunction<T>
    extends Serializable {
        public T get() throws IOException;
    }
}

