/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hive.ql.exec.vector.expressions.gen;

import java.util.Arrays;

import org.apache.hadoop.hive.ql.exec.vector.Decimal64ColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.VectorExpressionDescriptor;
import org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch;
import org.apache.hadoop.hive.ql.exec.vector.expressions.Decimal64Util;
import org.apache.hadoop.hive.ql.exec.vector.expressions.VectorExpression;
import org.apache.hadoop.hive.serde2.io.HiveDecimalWritable;
import org.apache.hadoop.hive.serde2.typeinfo.DecimalTypeInfo;
import org.apache.hadoop.hive.ql.metadata.HiveException;

/**
 * Generated from template Decimal64ScalarArithmeticDecimal64Column.txt.
 * Implements a vectorized arithmetic operator with a scalar on the left and a
 * column vector on the right. The result is output to an output column vector.
 */
public class Decimal64ScalarAddDecimal64Column extends VectorExpression {

  private static final long serialVersionUID = 1L;

  protected final long value;

  public Decimal64ScalarAddDecimal64Column(long value, int colNum, int outputColumnNum) {
    super(colNum, outputColumnNum);
    this.value = value;
  }

  public Decimal64ScalarAddDecimal64Column() {
    super();

    // Dummy final assignments.
    value = 0;
  }

  @Override
  public void evaluate(VectorizedRowBatch batch) throws HiveException {

    // return immediately if batch is empty
    final int n = batch.size;
    if (n == 0) {
      return;
    }

    if (childExpressions != null) {
      super.evaluateChildren(batch);
    }

    Decimal64ColumnVector inputColVector = (Decimal64ColumnVector) batch.cols[inputColumnNum[0]];
    Decimal64ColumnVector outputColVector = (Decimal64ColumnVector) batch.cols[outputColumnNum];
    int[] sel = batch.selected;
    boolean[] inputIsNull = inputColVector.isNull;
    boolean[] outputIsNull = outputColVector.isNull;

    // We do not need to do a column reset since we are carefully changing the output.
    outputColVector.isRepeating = false;

    long[] vector = inputColVector.vector;
    long[] outputVector = outputColVector.vector;

    final long outputDecimal64AbsMax =
        HiveDecimalWritable.getDecimal64AbsMax(outputColVector.precision);

    if (inputColVector.isRepeating) {
      if (inputColVector.noNulls || !inputIsNull[0]) {
        outputIsNull[0] = false;
        // The following may override a "false" null setting if an error or overflow occurs.
        final long result = value + vector[0];
        if (Math.abs(result) > outputDecimal64AbsMax) {
          outputIsNull[0] = true;
          outputColVector.noNulls = false;
        } else {
          outputVector[0] = result;
        }
      } else {
        outputIsNull[0] = true;
        outputColVector.noNulls = false;
      }
      outputColVector.isRepeating = true;
      return;
    }

    if (inputColVector.noNulls) {
      if (batch.selectedInUse) {

         // CONSIDER: For large n, fill n or all of isNull array and use the tighter ELSE loop.

         if (!outputColVector.noNulls) {
           for(int j = 0; j != n; j++) {
            final int i = sel[j];
            outputIsNull[i] = false;
            final long result = value + vector[i];
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputIsNull[i] = true;
              outputColVector.noNulls = false;
            }
          }
         } else {
           for(int j = 0; j != n; j++) {
             final int i = sel[j];
             final long result = value + vector[i];
             outputVector[i] = result;
             if (Math.abs(result) > outputDecimal64AbsMax) {
               outputIsNull[i] = true;
               outputColVector.noNulls = false;
             }
           }
         }
      } else {
        if (!outputColVector.noNulls) {

          // Assume it is almost always a performance win to fill all of isNull so we can
          // safely reset noNulls.
          Arrays.fill(outputIsNull, false);
          outputColVector.noNulls = true;
        }
        for(int i = 0; i != n; i++) {
          final long result = value + vector[i];
          outputVector[i] = result;
          if (Math.abs(result) > outputDecimal64AbsMax) {
            outputIsNull[i] = true;
            outputColVector.noNulls = false;
          }
        }
      }
    } else /* there are NULLs in the inputColVector */ {

      /*
       * Do careful maintenance of the outputColVector.noNulls flag.
       */

      if (batch.selectedInUse) {
        for(int j = 0; j != n; j++) {
          int i = sel[j];
          if (!inputIsNull[i]) {
            outputIsNull[i] = false;
            // The following may override a "false" null setting if an error or overflow occurs.
            final long result = value + vector[i];
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputIsNull[i] = true;
              outputColVector.noNulls = false;
            }
          } else {
            outputIsNull[i] = true;
            outputColVector.noNulls = false;
          }
        }
      } else {
        for(int i = 0; i != n; i++) {
          if (!inputIsNull[i]) {
            outputIsNull[i] = false;
            // The following may override a "false" null setting if an error or overflow occurs.
            final long result = value + vector[i];
            outputVector[i] = result;
            if (Math.abs(result) > outputDecimal64AbsMax) {
              outputIsNull[i] = true;
              outputColVector.noNulls = false;
            }
          } else {
            outputIsNull[i] = true;
            outputColVector.noNulls = false;
          }
        }
      }
    }
  }

  @Override
  public String vectorExpressionParameters() {
    DecimalTypeInfo decimalTypeInfo = (DecimalTypeInfo) inputTypeInfos[1];
    HiveDecimalWritable writable = new HiveDecimalWritable();
    writable.deserialize64(value, decimalTypeInfo.scale());
    return "decimal64Val " + value + ", decimalVal " + writable.toString() +
        ", " + getColumnParamString(1, inputColumnNum[0]);
  }

  @Override
  public VectorExpressionDescriptor.Descriptor getDescriptor() {
    return (new VectorExpressionDescriptor.Builder())
        .setMode(
            VectorExpressionDescriptor.Mode.PROJECTION)
        .setNumArguments(2)
        .setArgumentTypes(
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64,
            VectorExpressionDescriptor.ArgumentType.DECIMAL_64)
        .setInputExpressionTypes(
            VectorExpressionDescriptor.InputExpressionType.SCALAR,
            VectorExpressionDescriptor.InputExpressionType.COLUMN).build();
  }
}
