/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.hbase.shaded.com.google.common.collect.Lists;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.hdfs.server.namenode.FSDirXAttrOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EncryptionZoneManager {
    public static Logger LOG = LoggerFactory.getLogger(EncryptionZoneManager.class);
    private TreeMap<Long, EncryptionZoneInt> encryptionZones = null;
    private final FSDirectory dir;
    private final int maxListEncryptionZonesResponses;

    public EncryptionZoneManager(FSDirectory dir, Configuration conf) {
        this.dir = dir;
        this.maxListEncryptionZonesResponses = conf.getInt("dfs.namenode.list.encryption.zones.num.responses", 100);
        Preconditions.checkArgument(this.maxListEncryptionZonesResponses >= 0, "dfs.namenode.list.encryption.zones.num.responses must be a positive integer.");
    }

    void addEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        assert (this.dir.hasWriteLock());
        this.unprotectedAddEncryptionZone(inodeId, suite, version, keyName);
    }

    void unprotectedAddEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        EncryptionZoneInt ez = new EncryptionZoneInt(inodeId, suite, version, keyName);
        if (this.encryptionZones == null) {
            this.encryptionZones = new TreeMap();
        }
        this.encryptionZones.put(inodeId, ez);
    }

    void removeEncryptionZone(Long inodeId) {
        assert (this.dir.hasWriteLock());
        if (this.hasCreatedEncryptionZone()) {
            this.encryptionZones.remove(inodeId);
        }
    }

    boolean isInAnEZ(INodesInPath iip) throws UnresolvedLinkException, SnapshotAccessControlException {
        assert (this.dir.hasReadLock());
        return this.getEncryptionZoneForPath(iip) != null;
    }

    private String getFullPathName(EncryptionZoneInt ezi) {
        assert (this.dir.hasReadLock());
        return this.dir.getInode(ezi.getINodeId()).getFullPathName();
    }

    String getKeyName(INodesInPath iip) {
        assert (this.dir.hasReadLock());
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return ezi.getKeyName();
    }

    private EncryptionZoneInt getEncryptionZoneForPath(INodesInPath iip) {
        assert (this.dir.hasReadLock());
        Preconditions.checkNotNull(iip);
        if (!this.hasCreatedEncryptionZone()) {
            return null;
        }
        for (int i = iip.length() - 1; i >= 0; --i) {
            EncryptionZoneInt ezi;
            INode inode = iip.getINode(i);
            if (inode == null || (ezi = this.encryptionZones.get(inode.getId())) == null) continue;
            return ezi;
        }
        return null;
    }

    private EncryptionZoneInt getParentEncryptionZoneForPath(INodesInPath iip) {
        assert (this.dir.hasReadLock());
        Preconditions.checkNotNull(iip);
        INodesInPath parentIIP = iip.getParentINodesInPath();
        return parentIIP == null ? null : this.getEncryptionZoneForPath(parentIIP);
    }

    EncryptionZone getEZINodeForPath(INodesInPath iip) {
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return new EncryptionZone(ezi.getINodeId(), this.getFullPathName(ezi), ezi.getSuite(), ezi.getVersion(), ezi.getKeyName());
    }

    void checkMoveValidity(INodesInPath srcIIP, INodesInPath dstIIP) throws IOException {
        boolean dstInEZ;
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return;
        }
        EncryptionZoneInt srcParentEZI = this.getParentEncryptionZoneForPath(srcIIP);
        EncryptionZoneInt dstParentEZI = this.getParentEncryptionZoneForPath(dstIIP);
        boolean srcInEZ = srcParentEZI != null;
        boolean bl = dstInEZ = dstParentEZI != null;
        if (srcInEZ && !dstInEZ) {
            throw new IOException(srcIIP.getPath() + " can't be moved from an encryption zone.");
        }
        if (dstInEZ && !srcInEZ) {
            throw new IOException(srcIIP.getPath() + " can't be moved into an encryption zone.");
        }
        if (srcInEZ && srcParentEZI != dstParentEZI) {
            String srcEZPath = this.getFullPathName(srcParentEZI);
            String dstEZPath = this.getFullPathName(dstParentEZI);
            StringBuilder sb = new StringBuilder(srcIIP.getPath());
            sb.append(" can't be moved from encryption zone ");
            sb.append(srcEZPath);
            sb.append(" to encryption zone ");
            sb.append(dstEZPath);
            sb.append(".");
            throw new IOException(sb.toString());
        }
    }

    XAttr createEncryptionZone(INodesInPath srcIIP, CipherSuite suite, CryptoProtocolVersion version, String keyName) throws IOException {
        assert (this.dir.hasWriteLock());
        if (srcIIP.getLastINode() == null) {
            throw new FileNotFoundException("cannot find " + srcIIP.getPath());
        }
        if (this.dir.isNonEmptyDirectory(srcIIP)) {
            throw new IOException("Attempt to create an encryption zone for a non-empty directory.");
        }
        INode srcINode = srcIIP.getLastINode();
        if (!srcINode.isDirectory()) {
            throw new IOException("Attempt to create an encryption zone for a file.");
        }
        if (this.hasCreatedEncryptionZone() && this.encryptionZones.get(srcINode.getId()) != null) {
            throw new IOException("Directory " + srcIIP.getPath() + " is already an encryption zone.");
        }
        HdfsProtos.ZoneEncryptionInfoProto proto = PBHelperClient.convert(suite, version, keyName);
        XAttr ezXAttr = XAttrHelper.buildXAttr("raw.hdfs.crypto.encryption.zone", proto.toByteArray());
        ArrayList<XAttr> xattrs = Lists.newArrayListWithCapacity(1);
        xattrs.add(ezXAttr);
        FSDirXAttrOp.unprotectedSetXAttrs(this.dir, srcIIP, xattrs, EnumSet.of(XAttrSetFlag.CREATE));
        return ezXAttr;
    }

    BatchedRemoteIterator.BatchedListEntries<EncryptionZone> listEncryptionZones(long prevId) throws IOException {
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            ArrayList emptyZones = Lists.newArrayList();
            return new BatchedRemoteIterator.BatchedListEntries<EncryptionZone>(emptyZones, false);
        }
        NavigableMap<Long, EncryptionZoneInt> tailMap = this.encryptionZones.tailMap(prevId, false);
        int numResponses = Math.min(this.maxListEncryptionZonesResponses, tailMap.size());
        ArrayList<EncryptionZone> zones = Lists.newArrayListWithExpectedSize(numResponses);
        int count = 0;
        for (EncryptionZoneInt ezi : tailMap.values()) {
            String pathName = this.getFullPathName(ezi);
            INode inode = this.dir.getInode(ezi.getINodeId());
            INode lastINode = null;
            if (INode.isValidAbsolutePath(pathName)) {
                INodesInPath iip = this.dir.getINodesInPath(pathName, FSDirectory.DirOp.READ_LINK);
                lastINode = iip.getLastINode();
            }
            if (lastINode == null || lastINode.getId() != ezi.getINodeId()) continue;
            zones.add(new EncryptionZone(ezi.getINodeId(), pathName, ezi.getSuite(), ezi.getVersion(), ezi.getKeyName()));
            if (++count < numResponses) continue;
            break;
        }
        boolean hasMore = numResponses < tailMap.size();
        return new BatchedRemoteIterator.BatchedListEntries<EncryptionZone>(zones, hasMore);
    }

    public int getNumEncryptionZones() {
        return this.hasCreatedEncryptionZone() ? this.encryptionZones.size() : 0;
    }

    public boolean hasCreatedEncryptionZone() {
        return this.encryptionZones != null;
    }

    String[] getKeyNames() {
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return new String[0];
        }
        String[] ret = new String[this.encryptionZones.size()];
        int index = 0;
        for (Map.Entry<Long, EncryptionZoneInt> entry : this.encryptionZones.entrySet()) {
            ret[index++] = entry.getValue().getKeyName();
        }
        return ret;
    }

    private static class EncryptionZoneInt {
        private final long inodeId;
        private final CipherSuite suite;
        private final CryptoProtocolVersion version;
        private final String keyName;

        EncryptionZoneInt(long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
            Preconditions.checkArgument(suite != CipherSuite.UNKNOWN);
            Preconditions.checkArgument(version != CryptoProtocolVersion.UNKNOWN);
            this.inodeId = inodeId;
            this.suite = suite;
            this.version = version;
            this.keyName = keyName;
        }

        long getINodeId() {
            return this.inodeId;
        }

        CipherSuite getSuite() {
            return this.suite;
        }

        CryptoProtocolVersion getVersion() {
            return this.version;
        }

        String getKeyName() {
            return this.keyName;
        }
    }
}

