/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.retry;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.io.retry.AsyncCallHandler;
import org.apache.hadoop.io.retry.AtMostOnce;
import org.apache.hadoop.io.retry.CallReturn;
import org.apache.hadoop.io.retry.FailoverProxyProvider;
import org.apache.hadoop.io.retry.Idempotent;
import org.apache.hadoop.io.retry.MultiException;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.ipc.Client;
import org.apache.hadoop.ipc.ProtocolTranslator;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.ipc.RpcInvocationHandler;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class RetryInvocationHandler<T>
implements RpcInvocationHandler {
    public static final Logger LOG = LoggerFactory.getLogger(RetryInvocationHandler.class);
    private final ProxyDescriptor<T> proxyDescriptor;
    private volatile boolean hasSuccessfulCall = false;
    private final RetryPolicy defaultPolicy;
    private final Map<String, RetryPolicy> methodNameToPolicyMap;
    private final AsyncCallHandler asyncCallHandler = new AsyncCallHandler();

    protected RetryInvocationHandler(FailoverProxyProvider<T> proxyProvider, RetryPolicy retryPolicy) {
        this(proxyProvider, retryPolicy, Collections.emptyMap());
    }

    protected RetryInvocationHandler(FailoverProxyProvider<T> proxyProvider, RetryPolicy defaultPolicy, Map<String, RetryPolicy> methodNameToPolicyMap) {
        this.proxyDescriptor = new ProxyDescriptor<T>(proxyProvider);
        this.defaultPolicy = defaultPolicy;
        this.methodNameToPolicyMap = methodNameToPolicyMap;
    }

    private RetryPolicy getRetryPolicy(Method method) {
        RetryPolicy policy = this.methodNameToPolicyMap.get(method.getName());
        return policy != null ? policy : this.defaultPolicy;
    }

    private long getFailoverCount() {
        return this.proxyDescriptor.getFailoverCount();
    }

    private Call newCall(Method method, Object[] args, boolean isRpc, int callId) {
        if (Client.isAsynchronousMode()) {
            return this.asyncCallHandler.newAsyncCall(method, args, isRpc, callId, this);
        }
        return new Call(method, args, isRpc, callId, this);
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        CallReturn c;
        boolean isRpc = RetryInvocationHandler.isRpcInvocation(this.proxyDescriptor.getProxy());
        int callId = isRpc ? Client.nextCallId() : -2;
        Call call = this.newCall(method, args, isRpc, callId);
        do {
            CallReturn.State state;
            if ((state = (c = call.invokeOnce()).getState()) != CallReturn.State.ASYNC_INVOKED) continue;
            return null;
        } while (c.getState() == CallReturn.State.RETRY);
        return c.getReturnValue();
    }

    private RetryInfo handleException(Method method, int callId, RetryPolicy policy, Counters counters, long expectFailoverCount, Exception e) throws Exception {
        RetryInfo retryInfo = RetryInfo.newRetryInfo(policy, e, counters, this.proxyDescriptor.idempotentOrAtMostOnce(method), expectFailoverCount);
        if (retryInfo.isFail()) {
            if (((RetryInfo)retryInfo).action.reason != null && LOG.isDebugEnabled()) {
                LOG.debug("Exception while invoking call #" + callId + " " + this.proxyDescriptor.getProxyInfo().getString(method.getName()) + ". Not retrying because " + ((RetryInfo)retryInfo).action.reason, (Throwable)e);
            }
            throw retryInfo.getFailException();
        }
        this.log(method, retryInfo.isFailover(), counters.failovers, retryInfo.delay, e);
        return retryInfo;
    }

    private void log(Method method, boolean isFailover, int failovers, long delay, Exception ex) {
        boolean info;
        boolean bl = info = this.hasSuccessfulCall || failovers != 0 || this.asyncCallHandler.hasSuccessfulCall();
        if (!info && !LOG.isDebugEnabled()) {
            return;
        }
        StringBuilder b = new StringBuilder().append(ex + ", while invoking ").append(this.proxyDescriptor.getProxyInfo().getString(method.getName()));
        if (failovers > 0) {
            b.append(" after ").append(failovers).append(" failover attempts");
        }
        b.append(isFailover ? ". Trying to failover " : ". Retrying ");
        b.append(delay > 0L ? "after sleeping for " + delay + "ms." : "immediately.");
        if (info) {
            LOG.info(b.toString());
        } else {
            LOG.debug(b.toString(), (Throwable)ex);
        }
    }

    protected Object invokeMethod(Method method, Object[] args) throws Throwable {
        try {
            if (!method.isAccessible()) {
                method.setAccessible(true);
            }
            Object r = method.invoke(this.proxyDescriptor.getProxy(), args);
            this.hasSuccessfulCall = true;
            return r;
        }
        catch (InvocationTargetException e) {
            throw e.getCause();
        }
    }

    @VisibleForTesting
    static boolean isRpcInvocation(Object proxy) {
        if (proxy instanceof ProtocolTranslator) {
            proxy = ((ProtocolTranslator)proxy).getUnderlyingProxyObject();
        }
        if (!Proxy.isProxyClass(proxy.getClass())) {
            return false;
        }
        InvocationHandler ih = Proxy.getInvocationHandler(proxy);
        return ih instanceof RpcInvocationHandler;
    }

    @Override
    public void close() throws IOException {
        this.proxyDescriptor.close();
    }

    @Override
    public Client.ConnectionId getConnectionId() {
        return RPC.getConnectionIdForProxy(this.proxyDescriptor.getProxy());
    }

    private static class RetryInfo {
        private final long retryTime;
        private final long delay;
        private final RetryPolicy.RetryAction action;
        private final long expectedFailoverCount;
        private final Exception failException;

        RetryInfo(long delay, RetryPolicy.RetryAction action, long expectedFailoverCount, Exception failException) {
            this.delay = delay;
            this.retryTime = Time.monotonicNow() + delay;
            this.action = action;
            this.expectedFailoverCount = expectedFailoverCount;
            this.failException = failException;
        }

        boolean isFailover() {
            return this.action != null && this.action.action == RetryPolicy.RetryAction.RetryDecision.FAILOVER_AND_RETRY;
        }

        boolean isFail() {
            return this.action != null && this.action.action == RetryPolicy.RetryAction.RetryDecision.FAIL;
        }

        Exception getFailException() {
            return this.failException;
        }

        static RetryInfo newRetryInfo(RetryPolicy policy, Exception e, Counters counters, boolean idempotentOrAtMostOnce, long expectedFailoverCount) throws Exception {
            RetryPolicy.RetryAction max = null;
            long maxRetryDelay = 0L;
            Exception ex = null;
            List<Exception> exceptions = e instanceof MultiException ? ((MultiException)e).getExceptions().values() : Collections.singletonList(e);
            for (Exception exception : exceptions) {
                RetryPolicy.RetryAction a = policy.shouldRetry(exception, counters.retries, counters.failovers, idempotentOrAtMostOnce);
                if (a.action != RetryPolicy.RetryAction.RetryDecision.FAIL && a.delayMillis > maxRetryDelay) {
                    maxRetryDelay = a.delayMillis;
                }
                if (max != null && max.action.compareTo(a.action) >= 0) continue;
                max = a;
                if (a.action != RetryPolicy.RetryAction.RetryDecision.FAIL) continue;
                ex = exception;
            }
            return new RetryInfo(maxRetryDelay, max, expectedFailoverCount, ex);
        }

        public String toString() {
            return "RetryInfo{retryTime=" + this.retryTime + ", delay=" + this.delay + ", action=" + this.action + ", expectedFailoverCount=" + this.expectedFailoverCount + ", failException=" + this.failException + '}';
        }
    }

    private static class ProxyDescriptor<T> {
        private final FailoverProxyProvider<T> fpp;
        private long failoverCount = 0L;
        private FailoverProxyProvider.ProxyInfo<T> proxyInfo;

        ProxyDescriptor(FailoverProxyProvider<T> fpp) {
            this.fpp = fpp;
            this.proxyInfo = fpp.getProxy();
        }

        synchronized FailoverProxyProvider.ProxyInfo<T> getProxyInfo() {
            return this.proxyInfo;
        }

        synchronized T getProxy() {
            return this.proxyInfo.proxy;
        }

        synchronized long getFailoverCount() {
            return this.failoverCount;
        }

        synchronized void failover(long expectedFailoverCount, Method method, int callId) {
            if (this.failoverCount == expectedFailoverCount) {
                this.fpp.performFailover(this.proxyInfo.proxy);
                ++this.failoverCount;
            } else {
                LOG.warn("A failover has occurred since the start of call #" + callId + " " + this.proxyInfo.getString(method.getName()));
            }
            this.proxyInfo = this.fpp.getProxy();
        }

        boolean idempotentOrAtMostOnce(Method method) throws NoSuchMethodException {
            Method m = this.fpp.getInterface().getMethod(method.getName(), method.getParameterTypes());
            return m.isAnnotationPresent(Idempotent.class) || m.isAnnotationPresent(AtMostOnce.class);
        }

        void close() throws IOException {
            this.fpp.close();
        }
    }

    static class Counters {
        private int retries;
        private int failovers;

        Counters() {
        }

        boolean isZeros() {
            return this.retries == 0 && this.failovers == 0;
        }
    }

    static class Call {
        private final Method method;
        private final Object[] args;
        private final boolean isRpc;
        private final int callId;
        private final Counters counters = new Counters();
        private final RetryPolicy retryPolicy;
        private final RetryInvocationHandler<?> retryInvocationHandler;
        private RetryInfo retryInfo;

        Call(Method method, Object[] args, boolean isRpc, int callId, RetryInvocationHandler<?> retryInvocationHandler) {
            this.method = method;
            this.args = args;
            this.isRpc = isRpc;
            this.callId = callId;
            this.retryPolicy = ((RetryInvocationHandler)retryInvocationHandler).getRetryPolicy(method);
            this.retryInvocationHandler = retryInvocationHandler;
        }

        int getCallId() {
            return this.callId;
        }

        Counters getCounters() {
            return this.counters;
        }

        synchronized Long getWaitTime(long now) {
            return this.retryInfo == null ? null : Long.valueOf(this.retryInfo.retryTime - now);
        }

        synchronized CallReturn invokeOnce() {
            try {
                if (this.retryInfo != null) {
                    return this.processWaitTimeAndRetryInfo();
                }
                long failoverCount = ((RetryInvocationHandler)this.retryInvocationHandler).getFailoverCount();
                try {
                    return this.invoke();
                }
                catch (Exception e) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace(this.toString(), (Throwable)e);
                    }
                    if (Thread.currentThread().isInterrupted()) {
                        throw e;
                    }
                    this.retryInfo = ((RetryInvocationHandler)this.retryInvocationHandler).handleException(this.method, this.callId, this.retryPolicy, this.counters, failoverCount, e);
                    return this.processWaitTimeAndRetryInfo();
                }
            }
            catch (Throwable t) {
                return new CallReturn(t);
            }
        }

        CallReturn processWaitTimeAndRetryInfo() throws InterruptedIOException {
            Long waitTime = this.getWaitTime(Time.monotonicNow());
            LOG.trace("#{} processRetryInfo: retryInfo={}, waitTime={}", new Object[]{this.callId, this.retryInfo, waitTime});
            if (waitTime != null && waitTime > 0L) {
                try {
                    Thread.sleep(this.retryInfo.delay);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Interrupted while waiting to retry", (Throwable)e);
                    }
                    InterruptedIOException intIOE = new InterruptedIOException("Retry interrupted");
                    intIOE.initCause(e);
                    throw intIOE;
                }
            }
            this.processRetryInfo();
            return CallReturn.RETRY;
        }

        synchronized void processRetryInfo() {
            this.counters.retries++;
            if (this.retryInfo.isFailover()) {
                ((RetryInvocationHandler)this.retryInvocationHandler).proxyDescriptor.failover(this.retryInfo.expectedFailoverCount, this.method, this.callId);
                this.counters.failovers++;
            }
            this.retryInfo = null;
        }

        CallReturn invoke() throws Throwable {
            return new CallReturn(this.invokeMethod());
        }

        Object invokeMethod() throws Throwable {
            if (this.isRpc) {
                Client.setCallIdAndRetryCount(this.callId, this.counters.retries, ((RetryInvocationHandler)this.retryInvocationHandler).asyncCallHandler);
            }
            return this.retryInvocationHandler.invokeMethod(this.method, this.args);
        }

        public String toString() {
            return this.getClass().getSimpleName() + "#" + this.callId + ": " + this.method.getDeclaringClass().getSimpleName() + "." + this.method.getName() + "(" + (this.args == null || this.args.length == 0 ? "" : Arrays.toString(this.args)) + ")";
        }
    }
}

