/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.context.Scope;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.AsyncConnectionImpl;
import org.apache.hadoop.hbase.client.AsyncMetaRegionLocator;
import org.apache.hadoop.hbase.client.AsyncNonMetaRegionLocator;
import org.apache.hadoop.hbase.client.MetricsConnection;
import org.apache.hadoop.hbase.client.RegionLocateType;
import org.apache.hadoop.hbase.client.RegionOfflineException;
import org.apache.hadoop.hbase.exceptions.TimeoutIOException;
import org.apache.hadoop.hbase.trace.HBaseSemanticAttributes;
import org.apache.hadoop.hbase.trace.TraceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hbase.thirdparty.io.netty.util.HashedWheelTimer;
import org.apache.hbase.thirdparty.io.netty.util.Timeout;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class AsyncRegionLocator {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncRegionLocator.class);
    private final HashedWheelTimer retryTimer;
    private final AsyncConnectionImpl conn;
    private final AsyncMetaRegionLocator metaRegionLocator;
    private final AsyncNonMetaRegionLocator nonMetaRegionLocator;

    AsyncRegionLocator(AsyncConnectionImpl conn, HashedWheelTimer retryTimer) {
        this.conn = conn;
        this.metaRegionLocator = new AsyncMetaRegionLocator(conn.registry);
        this.nonMetaRegionLocator = new AsyncNonMetaRegionLocator(conn);
        this.retryTimer = retryTimer;
    }

    private <T> CompletableFuture<T> withTimeout(CompletableFuture<T> future, long timeoutNs, Supplier<String> timeoutMsg) {
        if (future.isDone() || timeoutNs <= 0L) {
            return future;
        }
        Timeout timeoutTask = this.retryTimer.newTimeout(t -> {
            if (future.isDone()) {
                return;
            }
            future.completeExceptionally((Throwable)new TimeoutIOException((String)timeoutMsg.get()));
        }, timeoutNs, TimeUnit.NANOSECONDS);
        FutureUtils.addListener(future, (loc, error) -> {
            if (error != null && error.getClass() != TimeoutIOException.class) {
                timeoutTask.cancel();
            }
        });
        return future;
    }

    private boolean isMeta(TableName tableName) {
        return TableName.isMetaTableName((TableName)tableName);
    }

    private <T> CompletableFuture<T> tracedLocationFuture(Supplier<CompletableFuture<T>> action, Function<T, List<String>> getRegionNames, TableName tableName, String methodName) {
        Span span = TraceUtil.createTableSpan((String)("AsyncRegionLocator." + methodName), (TableName)tableName);
        try (Scope scope = span.makeCurrent();){
            CompletableFuture<T> future = action.get();
            FutureUtils.addListener(future, (resp, error) -> {
                if (error != null) {
                    TraceUtil.setError((Span)span, (Throwable)error);
                } else {
                    List regionNames = (List)getRegionNames.apply(resp);
                    if (!regionNames.isEmpty()) {
                        span.setAttribute(HBaseSemanticAttributes.REGION_NAMES_KEY, (Object)regionNames);
                    }
                    span.setStatus(StatusCode.OK);
                }
                span.end();
            });
            CompletableFuture<T> completableFuture = future;
            return completableFuture;
        }
    }

    private List<String> getRegionName(RegionLocations locs) {
        ArrayList<String> names = new ArrayList<String>();
        for (HRegionLocation loc : locs.getRegionLocations()) {
            if (loc == null) continue;
            names.add(loc.getRegion().getRegionNameAsString());
        }
        return names;
    }

    CompletableFuture<RegionLocations> getRegionLocations(TableName tableName, byte[] row, RegionLocateType type, boolean reload, long timeoutNs) {
        return this.tracedLocationFuture(() -> {
            CompletableFuture<RegionLocations> future = this.isMeta(tableName) ? this.metaRegionLocator.getRegionLocations(0, reload) : this.nonMetaRegionLocator.getRegionLocations(tableName, row, 0, type, reload);
            return this.withTimeout(future, timeoutNs, () -> "Timeout(" + TimeUnit.NANOSECONDS.toMillis(timeoutNs) + "ms) waiting for region locations for " + tableName + ", row='" + Bytes.toStringBinary((byte[])row) + "'");
        }, this::getRegionName, tableName, "getRegionLocations");
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, int replicaId, RegionLocateType type, boolean reload, long timeoutNs) {
        return this.tracedLocationFuture(() -> {
            CompletableFuture future = new CompletableFuture();
            CompletableFuture<RegionLocations> locsFuture = this.isMeta(tableName) ? this.metaRegionLocator.getRegionLocations(replicaId, reload) : this.nonMetaRegionLocator.getRegionLocations(tableName, row, replicaId, type, reload);
            FutureUtils.addListener(locsFuture, (locs, error) -> {
                if (error != null) {
                    future.completeExceptionally((Throwable)error);
                    return;
                }
                HRegionLocation loc = locs.getRegionLocation(replicaId);
                if (loc == null) {
                    future.completeExceptionally((Throwable)((Object)new RegionOfflineException("No location for " + tableName + ", row='" + Bytes.toStringBinary((byte[])row) + "', locateType=" + (Object)((Object)type) + ", replicaId=" + replicaId)));
                } else if (loc.getServerName() == null) {
                    future.completeExceptionally((Throwable)((Object)new RegionOfflineException("No server address listed for region '" + loc.getRegion().getRegionNameAsString() + ", row='" + Bytes.toStringBinary((byte[])row) + "', locateType=" + (Object)((Object)type) + ", replicaId=" + replicaId)));
                } else {
                    future.complete(loc);
                }
            });
            return this.withTimeout(future, timeoutNs, () -> "Timeout(" + TimeUnit.NANOSECONDS.toMillis(timeoutNs) + "ms) waiting for region location for " + tableName + ", row='" + Bytes.toStringBinary((byte[])row) + "', replicaId=" + replicaId);
        }, loc -> Arrays.asList(loc.getRegion().getRegionNameAsString()), tableName, "getRegionLocation");
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, int replicaId, RegionLocateType type, long timeoutNs) {
        return this.getRegionLocation(tableName, row, replicaId, type, false, timeoutNs);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, RegionLocateType type, boolean reload, long timeoutNs) {
        return this.getRegionLocation(tableName, row, 0, type, reload, timeoutNs);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, RegionLocateType type, long timeoutNs) {
        return this.getRegionLocation(tableName, row, type, false, timeoutNs);
    }

    void updateCachedLocationOnError(HRegionLocation loc, Throwable exception) {
        if (loc.getRegion().isMetaRegion()) {
            this.metaRegionLocator.updateCachedLocationOnError(loc, exception);
        } else {
            this.nonMetaRegionLocator.updateCachedLocationOnError(loc, exception);
        }
    }

    void clearCache(TableName tableName) {
        TraceUtil.trace(() -> {
            LOG.debug("Clear meta cache for {}", (Object)tableName);
            if (tableName.equals((Object)TableName.META_TABLE_NAME)) {
                this.metaRegionLocator.clearCache();
            } else {
                this.nonMetaRegionLocator.clearCache(tableName);
            }
        }, () -> TraceUtil.createTableSpan((String)"AsyncRegionLocator.clearCache", (TableName)tableName));
    }

    void clearCache(ServerName serverName) {
        TraceUtil.trace(() -> {
            LOG.debug("Clear meta cache for {}", (Object)serverName);
            this.metaRegionLocator.clearCache(serverName);
            this.nonMetaRegionLocator.clearCache(serverName);
            this.conn.getConnectionMetrics().ifPresent(MetricsConnection::incrMetaCacheNumClearServer);
        }, () -> TraceUtil.createSpan((String)"AsyncRegionLocator.clearCache").setAttribute(HBaseSemanticAttributes.SERVER_NAME_KEY, (Object)serverName.getServerName()));
    }

    void clearCache() {
        TraceUtil.trace(() -> {
            this.metaRegionLocator.clearCache();
            this.nonMetaRegionLocator.clearCache();
        }, (String)"AsyncRegionLocator.clearCache");
    }

    AsyncNonMetaRegionLocator getNonMetaRegionLocator() {
        return this.nonMetaRegionLocator;
    }

    RegionLocations getRegionLocationInCache(TableName tableName, byte[] row) {
        if (TableName.isMetaTableName((TableName)tableName)) {
            return this.metaRegionLocator.getRegionLocationInCache();
        }
        return this.nonMetaRegionLocator.getRegionLocationInCache(tableName, row);
    }

    int getNumberOfCachedRegionLocations(TableName tableName) {
        if (TableName.isMetaTableName((TableName)tableName)) {
            return this.metaRegionLocator.getNumberOfCachedRegionLocations();
        }
        return this.nonMetaRegionLocator.getNumberOfCachedRegionLocations(tableName);
    }
}

