/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.examples;

import com.google.common.base.Charsets;
import java.io.BufferedOutputStream;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.BytesWritable;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.mapreduce.InputFormat;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.RecordReader;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BaileyBorweinPlouffe
extends Configured
implements Tool {
    public static final String DESCRIPTION = "A map/reduce program that uses Bailey-Borwein-Plouffe to compute exact digits of Pi.";
    private static final String NAME = "mapreduce." + BaileyBorweinPlouffe.class.getSimpleName();
    private static final String WORKING_DIR_PROPERTY = NAME + ".dir";
    private static final String HEX_FILE_PROPERTY = NAME + ".hex.file";
    private static final String DIGIT_START_PROPERTY = NAME + ".digit.start";
    private static final String DIGIT_SIZE_PROPERTY = NAME + ".digit.size";
    private static final String DIGIT_PARTS_PROPERTY = NAME + ".digit.parts";
    private static final Logger LOG = LoggerFactory.getLogger(BaileyBorweinPlouffe.class);
    private static final long IMPLEMENTATION_LIMIT = 100000000L;
    private static final long ACCURACY_BIT = 32L;
    private static final long BBP_HEX_DIGITS = 4L;
    private static final long BBP_MULTIPLIER = 65536L;
    private static final long MAX_N = 0xFFFFFFFFL;

    private static <T> void print(PrintWriter out, Iterator<T> iterator, String prefix, String format, int elementsPerGroup, int groupsPerLine) {
        StringBuilder sb = new StringBuilder("\n");
        for (int i = 0; i < prefix.length(); ++i) {
            sb.append(" ");
        }
        String spaces = sb.toString();
        out.print("\n" + prefix);
        int i = 0;
        while (iterator.hasNext()) {
            if (i > 0 && i % elementsPerGroup == 0) {
                out.print(i / elementsPerGroup % groupsPerLine == 0 ? spaces : " ");
            }
            out.print(String.format(format, iterator.next()));
            ++i;
        }
        out.println();
    }

    private static Job createJob(String name, Configuration conf) throws IOException {
        Job job = Job.getInstance((Configuration)conf, (String)(NAME + "_" + name));
        Configuration jobconf = job.getConfiguration();
        job.setJarByClass(BaileyBorweinPlouffe.class);
        job.setMapperClass(BbpMapper.class);
        job.setMapOutputKeyClass(LongWritable.class);
        job.setMapOutputValueClass(BytesWritable.class);
        job.setReducerClass(BbpReducer.class);
        job.setOutputKeyClass(LongWritable.class);
        job.setOutputValueClass(BytesWritable.class);
        job.setNumReduceTasks(1);
        job.setInputFormatClass(BbpInputFormat.class);
        jobconf.setLong("mapreduce.task.timeout", 0L);
        jobconf.setBoolean("mapreduce.map.speculative", false);
        jobconf.setBoolean("mapreduce.reduce.speculative", false);
        return job;
    }

    private static void compute(int startDigit, int nDigits, int nMaps, String workingDir, Configuration conf, PrintStream out) throws IOException {
        String name = startDigit + "_" + nDigits;
        out.println("Working Directory = " + workingDir);
        out.println();
        FileSystem fs = FileSystem.get((Configuration)conf);
        Path dir = fs.makeQualified(new Path(workingDir));
        if (fs.exists(dir)) {
            throw new IOException("Working directory " + dir + " already exists.  Please remove it first.");
        }
        if (!fs.mkdirs(dir)) {
            throw new IOException("Cannot create working directory " + dir);
        }
        out.println("Start Digit      = " + startDigit);
        out.println("Number of Digits = " + nDigits);
        out.println("Number of Maps   = " + nMaps);
        Job job = BaileyBorweinPlouffe.createJob(name, conf);
        Path hexfile = new Path(dir, "pi_" + name + ".hex");
        FileOutputFormat.setOutputPath((Job)job, (Path)new Path(dir, "out"));
        job.getConfiguration().set(WORKING_DIR_PROPERTY, dir.toString());
        job.getConfiguration().set(HEX_FILE_PROPERTY, hexfile.toString());
        job.getConfiguration().setInt(DIGIT_START_PROPERTY, startDigit);
        job.getConfiguration().setInt(DIGIT_SIZE_PROPERTY, nDigits);
        job.getConfiguration().setInt(DIGIT_PARTS_PROPERTY, nMaps);
        out.println("\nStarting Job ...");
        long startTime = Time.monotonicNow();
        try {
            if (!job.waitForCompletion(true)) {
                out.println("Job failed.");
                System.exit(1);
            }
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        finally {
            double duration = (double)(Time.monotonicNow() - startTime) / 1000.0;
            out.println("Duration is " + duration + " seconds.");
        }
        out.println("Output file: " + hexfile);
    }

    public int run(String[] args) throws IOException {
        if (args.length != 4) {
            System.err.println("Usage: bbp  <startDigit> <nDigits> <nMaps> <workingDir>");
            ToolRunner.printGenericCommandUsage((PrintStream)System.err);
            return -1;
        }
        int startDigit = Integer.parseInt(args[0]);
        int nDigits = Integer.parseInt(args[1]);
        int nMaps = Integer.parseInt(args[2]);
        String workingDir = args[3];
        if (startDigit <= 0) {
            throw new IllegalArgumentException("startDigit = " + startDigit + " <= 0");
        }
        if (nDigits <= 0) {
            throw new IllegalArgumentException("nDigits = " + nDigits + " <= 0");
        }
        if ((long)nDigits % 4L != 0L) {
            throw new IllegalArgumentException("nDigits = " + nDigits + " is not a multiple of " + 4L);
        }
        if ((long)nDigits - 1L + (long)startDigit > 100000004L) {
            throw new UnsupportedOperationException("nDigits - 1 + startDigit = " + ((long)nDigits - 1L + (long)startDigit) + " > IMPLEMENTATION_LIMIT + BBP_HEX_DIGITS," + ", where IMPLEMENTATION_LIMIT=" + 100000000L + "and BBP_HEX_DIGITS=" + 4L);
        }
        if (nMaps <= 0) {
            throw new IllegalArgumentException("nMaps = " + nMaps + " <= 0");
        }
        BaileyBorweinPlouffe.compute(startDigit, nDigits, nMaps, workingDir, this.getConf(), System.out);
        return 0;
    }

    public static void main(String[] argv) throws Exception {
        System.exit(ToolRunner.run(null, (Tool)new BaileyBorweinPlouffe(), (String[])argv));
    }

    static long hexDigits(long d) {
        if (d < 0L) {
            throw new IllegalArgumentException("d = " + d + " < 0");
        }
        if (d > 100000000L) {
            throw new IllegalArgumentException("d = " + d + " > IMPLEMENTATION_LIMIT = " + 100000000L);
        }
        double s1 = BaileyBorweinPlouffe.sum(1L, d);
        double s4 = BaileyBorweinPlouffe.sum(4L, d);
        double s5 = BaileyBorweinPlouffe.sum(5L, d);
        double s6 = BaileyBorweinPlouffe.sum(6L, d);
        double pi = s1 + s1;
        if (pi >= 1.0) {
            pi -= 1.0;
        }
        if ((pi *= 2.0) >= 1.0) {
            pi -= 1.0;
        }
        if ((pi -= s4) < 0.0) {
            pi += 1.0;
        }
        if ((pi -= s4) < 0.0) {
            pi += 1.0;
        }
        if ((pi -= s5) < 0.0) {
            pi += 1.0;
        }
        if ((pi -= s6) < 0.0) {
            pi += 1.0;
        }
        return (long)(pi * 65536.0);
    }

    private static double sum(long j, long d) {
        long n;
        long k;
        double s = 0.0;
        if (k <= d) {
            s = 1.0 / (double)(d << 3 | j);
            for (k = j == 1L ? 1L : 0L; k < d; ++k) {
                n = k << 3 | j;
                if (!((s += (double)BaileyBorweinPlouffe.mod(d - k << 2, n) * 1.0 / (double)n) >= 1.0)) continue;
                s -= 1.0;
            }
            ++k;
        }
        if (k >= 0x2000000L) {
            return s;
        }
        while (true) {
            n = k << 3 | j;
            long shift = k - d << 2;
            if (32L <= shift || 1L << (int)(32L - shift) < n) {
                return s;
            }
            if ((s += 1.0 / (double)(n << (int)shift)) >= 1.0) {
                s -= 1.0;
            }
            ++k;
        }
    }

    static long mod(long e, long n) {
        long mask = (e & 0xFFFFFFFF00000000L) == 0L ? 0xFFFFFFFFL : -4294967296L;
        mask &= (e & 0xAAAAAAAAAAAAAAAAL & (mask &= (e & 0xCCCCCCCCCCCCCCCCL & (mask &= (e & 0xF0F0F0F0F0F0F0F0L & (mask &= (e & 0xFF00FF00FF00FF00L & (mask &= (e & 0xFFFF0000FFFF0000L & mask) == 0L ? 0xFFFF0000FFFFL : -281470681808896L)) == 0L ? 0xFF00FF00FF00FFL : -71777214294589696L)) == 0L ? 0xF0F0F0F0F0F0F0FL : -1085102592571150096L)) == 0L ? 0x3333333333333333L : -3689348814741910324L)) == 0L ? 0x5555555555555555L : -6148914691236517206L;
        long r = 2L;
        mask >>= 1;
        while (mask > 0L) {
            r *= r;
            r %= n;
            if ((e & mask) != 0L && (r += r) >= n) {
                r -= n;
            }
            mask >>= 1;
        }
        return r;
    }

    static int[] partition(int offset, int size, int nParts) {
        int[] parts = new int[nParts];
        long total = BaileyBorweinPlouffe.workload(offset, size);
        int remainder = offset % 4;
        parts[0] = offset;
        for (int i = 1; i < nParts; ++i) {
            long target = (long)offset + (long)i * (total / (long)nParts) + (long)i * (total % (long)nParts) / (long)nParts;
            int low = parts[i - 1];
            int high = offset + size;
            while (high > low + 4) {
                int mid = (high + low - 2 * remainder) / 8 * 4 + remainder;
                long midvalue = BaileyBorweinPlouffe.workload(mid);
                if (midvalue == target) {
                    high = low = mid;
                    continue;
                }
                if (midvalue > target) {
                    high = mid;
                    continue;
                }
                low = mid;
            }
            parts[i] = high == low ? high : (BaileyBorweinPlouffe.workload(high) - target > target - BaileyBorweinPlouffe.workload(low) ? low : high);
        }
        return parts;
    }

    private static long workload(long n) {
        if (n < 0L) {
            throw new IllegalArgumentException("n = " + n + " < 0");
        }
        if (n > 0xFFFFFFFFL) {
            throw new IllegalArgumentException("n = " + n + " > MAX_N = " + 0xFFFFFFFFL);
        }
        return (n & 1L) == 0L ? (n >> 1) * (n + 1L) : n * (n + 1L >> 1);
    }

    private static long workload(long offset, long size) {
        return BaileyBorweinPlouffe.workload(offset + size) - BaileyBorweinPlouffe.workload(offset);
    }

    private static class Fraction {
        private final int[] integers;
        private int first = 0;

        Fraction(List<Byte> bytes) {
            this.integers = new int[(bytes.size() + 2) / 3];
            for (int i = 0; i < bytes.size(); ++i) {
                int b = 0xFF & bytes.get(i);
                int n = this.integers.length - 1 - i / 3;
                this.integers[n] = this.integers[n] | b << (2 - i % 3 << 3);
            }
            this.skipZeros();
        }

        int times10() {
            int carry = 0;
            int i = this.first;
            while (i < this.integers.length) {
                int n = i;
                this.integers[n] = this.integers[n] << 1;
                int n2 = i;
                this.integers[n2] = this.integers[n2] + (carry + (this.integers[i] << 2));
                carry = this.integers[i] >> 24;
                int n3 = i++;
                this.integers[n3] = this.integers[n3] & 0xFFFFFF;
            }
            this.skipZeros();
            return carry;
        }

        private void skipZeros() {
            while (this.first < this.integers.length && this.integers[this.first] == 0) {
                ++this.first;
            }
        }
    }

    public static class BbpInputFormat
    extends InputFormat<LongWritable, IntWritable> {
        public List<InputSplit> getSplits(JobContext context) {
            int startDigit = context.getConfiguration().getInt(DIGIT_START_PROPERTY, 1);
            int nDigits = context.getConfiguration().getInt(DIGIT_SIZE_PROPERTY, 100);
            int nMaps = context.getConfiguration().getInt(DIGIT_PARTS_PROPERTY, 1);
            ArrayList<InputSplit> splits = new ArrayList<InputSplit>(nMaps);
            int[] parts = BaileyBorweinPlouffe.partition(startDigit - 1, nDigits, nMaps);
            for (int i = 0; i < parts.length; ++i) {
                int k = i < parts.length - 1 ? parts[i + 1] : nDigits + startDigit - 1;
                splits.add(new BbpSplit(i, parts[i], k - parts[i]));
            }
            return splits;
        }

        public RecordReader<LongWritable, IntWritable> createRecordReader(InputSplit generic, TaskAttemptContext context) {
            final BbpSplit split = (BbpSplit)generic;
            return new RecordReader<LongWritable, IntWritable>(){
                boolean done = false;

                public void initialize(InputSplit split2, TaskAttemptContext context) {
                }

                public boolean nextKeyValue() {
                    return !this.done ? (this.done = true) : false;
                }

                public LongWritable getCurrentKey() {
                    return new LongWritable(split.getOffset());
                }

                public IntWritable getCurrentValue() {
                    return new IntWritable((int)split.getLength());
                }

                public float getProgress() {
                    return this.done ? 1.0f : 0.0f;
                }

                public void close() {
                }
            };
        }
    }

    public static class BbpSplit
    extends InputSplit
    implements Writable {
        private static final String[] EMPTY = new String[0];
        private long offset;
        private int size;

        public BbpSplit() {
        }

        private BbpSplit(int i, long offset, int size) {
            LOG.info("Map #" + i + ": workload=" + BaileyBorweinPlouffe.workload(offset, size) + ", offset=" + offset + ", size=" + size);
            this.offset = offset;
            this.size = size;
        }

        private long getOffset() {
            return this.offset;
        }

        public long getLength() {
            return this.size;
        }

        public String[] getLocations() {
            return EMPTY;
        }

        public void readFields(DataInput in) throws IOException {
            this.offset = in.readLong();
            this.size = in.readInt();
        }

        public void write(DataOutput out) throws IOException {
            out.writeLong(this.offset);
            out.writeInt(this.size);
        }
    }

    public static class BbpReducer
    extends Reducer<LongWritable, BytesWritable, LongWritable, BytesWritable> {
        private final List<Byte> hex = new ArrayList<Byte>();

        protected void reduce(LongWritable offset, Iterable<BytesWritable> values, Reducer.Context context) throws IOException, InterruptedException {
            for (BytesWritable bytes : values) {
                for (int i = 0; i < bytes.getLength(); ++i) {
                    this.hex.add(bytes.getBytes()[i]);
                }
            }
            LOG.info("hex.size() = " + this.hex.size());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void cleanup(Reducer.Context context) throws IOException, InterruptedException {
            Configuration conf = context.getConfiguration();
            Path dir = new Path(conf.get(WORKING_DIR_PROPERTY));
            FileSystem fs = dir.getFileSystem(conf);
            Path hexfile = new Path(conf.get(HEX_FILE_PROPERTY));
            try (BufferedOutputStream out = new BufferedOutputStream((OutputStream)fs.create(hexfile));){
                for (byte b : this.hex) {
                    ((OutputStream)out).write(b);
                }
            }
            if (conf.getInt(DIGIT_START_PROPERTY, 1) == 1) {
                Path outfile = new Path(dir, "pi.txt");
                LOG.info("Writing text output to " + outfile);
                try (FSDataOutputStream outputstream = fs.create(outfile);){
                    PrintWriter out = new PrintWriter((Writer)new OutputStreamWriter((OutputStream)outputstream, Charsets.UTF_8), true);
                    BaileyBorweinPlouffe.print(out, this.hex.iterator(), "Pi = 0x3.", "%02X", 5, 5);
                    out.println("Total number of hexadecimal digits is " + 2 * this.hex.size() + ".");
                    final Fraction dec = new Fraction(this.hex);
                    final int decDigits = 2 * this.hex.size();
                    BaileyBorweinPlouffe.print(out, new Iterator<Integer>(){
                        private int i = 0;

                        @Override
                        public boolean hasNext() {
                            return this.i < decDigits;
                        }

                        @Override
                        public Integer next() {
                            ++this.i;
                            return dec.times10();
                        }

                        @Override
                        public void remove() {
                        }
                    }, "Pi = 3.", "%d", 10, 5);
                    out.println("Total number of decimal digits is " + decDigits + ".");
                }
            }
        }
    }

    public static class BbpMapper
    extends Mapper<LongWritable, IntWritable, LongWritable, BytesWritable> {
        protected void map(LongWritable offset, IntWritable length, Mapper.Context context) throws IOException, InterruptedException {
            LOG.info("offset=" + offset + ", length=" + length);
            byte[] bytes = new byte[length.get() >> 1];
            long d = offset.get();
            int i = 0;
            while (i < bytes.length) {
                long digits = BaileyBorweinPlouffe.hexDigits(d);
                bytes[i++] = (byte)(digits >> 8);
                bytes[i++] = (byte)digits;
                d += 4L;
            }
            context.write((Object)offset, (Object)new BytesWritable(bytes));
        }
    }
}

