/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.checker;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.nio.channels.ClosedChannelException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.datanode.checker.AsyncChecker;
import org.apache.hadoop.hdfs.server.datanode.checker.ThrottledAsyncChecker;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeReference;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatasetVolumeChecker {
    public static final Logger LOG = LoggerFactory.getLogger(DatasetVolumeChecker.class);
    private AsyncChecker<FsVolumeSpi.VolumeCheckContext, VolumeCheckResult> delegateChecker;
    private final AtomicLong numVolumeChecks = new AtomicLong(0L);
    private final AtomicLong numSyncDatasetChecks = new AtomicLong(0L);
    private final AtomicLong numAsyncDatasetChecks = new AtomicLong(0L);
    private final AtomicLong numSkippedChecks = new AtomicLong(0L);
    private final long maxAllowedTimeForCheckMs;
    private final int maxVolumeFailuresTolerated;
    private final long minDiskCheckGapMs;
    private long lastAllVolumesCheck;
    private final Timer timer;
    private static final FsVolumeSpi.VolumeCheckContext IGNORED_CONTEXT = new FsVolumeSpi.VolumeCheckContext();

    public DatasetVolumeChecker(Configuration conf, Timer timer) throws DiskChecker.DiskErrorException {
        this.maxAllowedTimeForCheckMs = conf.getTimeDuration("dfs.datanode.disk.check.timeout", "10m", TimeUnit.MILLISECONDS);
        if (this.maxAllowedTimeForCheckMs <= 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.timeout - " + this.maxAllowedTimeForCheckMs + " (should be > 0)");
        }
        this.timer = timer;
        this.maxVolumeFailuresTolerated = conf.getInt("dfs.datanode.failed.volumes.tolerated", 0);
        this.minDiskCheckGapMs = conf.getTimeDuration("dfs.datanode.disk.check.min.gap", "15m", TimeUnit.MILLISECONDS);
        if (this.minDiskCheckGapMs < 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.min.gap - " + this.minDiskCheckGapMs + " (should be >= 0)");
        }
        this.lastAllVolumesCheck = timer.monotonicNow() - this.minDiskCheckGapMs;
        if (this.maxVolumeFailuresTolerated < 0) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.failed.volumes.tolerated - " + this.maxVolumeFailuresTolerated + " (should be non-negative)");
        }
        this.delegateChecker = new ThrottledAsyncChecker<FsVolumeSpi.VolumeCheckContext, VolumeCheckResult>(timer, this.minDiskCheckGapMs, Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("DataNode DiskChecker thread %d").setDaemon(true).build()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<FsVolumeSpi> checkAllVolumes(FsDatasetSpi<? extends FsVolumeSpi> dataset) throws InterruptedException {
        long gap = this.timer.monotonicNow() - this.lastAllVolumesCheck;
        if (gap < this.minDiskCheckGapMs) {
            this.numSkippedChecks.incrementAndGet();
            LOG.trace("Skipped checking all volumes, time since last check {} is less than the minimum gap between checks ({} ms).", (Object)gap, (Object)this.minDiskCheckGapMs);
            return Collections.emptySet();
        }
        FsDatasetSpi.FsVolumeReferences references = dataset.getFsVolumeReferences();
        if (references.size() == 0) {
            LOG.warn("checkAllVolumesAsync - no volumes can be referenced");
            return Collections.emptySet();
        }
        this.lastAllVolumesCheck = this.timer.monotonicNow();
        HashSet<FsVolumeSpi> healthyVolumes = new HashSet<FsVolumeSpi>();
        HashSet<FsVolumeSpi> failedVolumes = new HashSet<FsVolumeSpi>();
        HashSet<FsVolumeSpi> allVolumes = new HashSet<FsVolumeSpi>();
        AtomicLong numVolumes = new AtomicLong(references.size());
        final CountDownLatch latch = new CountDownLatch(1);
        for (int i = 0; i < references.size(); ++i) {
            FsVolumeReference reference = references.getReference(i);
            Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(reference.getVolume(), IGNORED_CONTEXT);
            LOG.info("Scheduled health check for volume {}", (Object)reference.getVolume());
            if (olf.isPresent()) {
                allVolumes.add(reference.getVolume());
                Futures.addCallback((ListenableFuture)((ListenableFuture)olf.get()), (FutureCallback)new ResultHandler(reference, healthyVolumes, failedVolumes, numVolumes, new Callback(){

                    @Override
                    public void call(Set<FsVolumeSpi> ignored1, Set<FsVolumeSpi> ignored2) {
                        latch.countDown();
                    }
                }));
                continue;
            }
            IOUtils.cleanup(null, (Closeable[])new Closeable[]{reference});
            if (numVolumes.decrementAndGet() != 0L) continue;
            latch.countDown();
        }
        if (!latch.await(this.maxAllowedTimeForCheckMs, TimeUnit.MILLISECONDS)) {
            LOG.warn("checkAllVolumes timed out after {} ms" + this.maxAllowedTimeForCheckMs);
        }
        this.numSyncDatasetChecks.incrementAndGet();
        DatasetVolumeChecker datasetVolumeChecker = this;
        synchronized (datasetVolumeChecker) {
            return new HashSet<FsVolumeSpi>((Collection<FsVolumeSpi>)Sets.difference(allVolumes, healthyVolumes));
        }
    }

    public boolean checkVolume(FsVolumeSpi volume, Callback callback) {
        FsVolumeReference volumeReference;
        if (volume == null) {
            LOG.debug("Cannot schedule check on null volume");
            return false;
        }
        try {
            volumeReference = volume.obtainReference();
        }
        catch (ClosedChannelException e) {
            return false;
        }
        Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(volume, IGNORED_CONTEXT);
        if (olf.isPresent()) {
            this.numVolumeChecks.incrementAndGet();
            Futures.addCallback((ListenableFuture)((ListenableFuture)olf.get()), (FutureCallback)new ResultHandler(volumeReference, new HashSet<FsVolumeSpi>(), new HashSet<FsVolumeSpi>(), new AtomicLong(1L), callback));
            return true;
        }
        IOUtils.cleanup(null, (Closeable[])new Closeable[]{volumeReference});
        return false;
    }

    public void shutdownAndWait(int gracePeriod, TimeUnit timeUnit) {
        try {
            this.delegateChecker.shutdownAndWait(gracePeriod, timeUnit);
        }
        catch (InterruptedException e) {
            LOG.warn("DatasetVolumeChecker interrupted during shutdown.");
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    void setDelegateChecker(AsyncChecker<FsVolumeSpi.VolumeCheckContext, VolumeCheckResult> testDelegate) {
        this.delegateChecker = testDelegate;
    }

    public long getNumVolumeChecks() {
        return this.numVolumeChecks.get();
    }

    public long getNumSyncDatasetChecks() {
        return this.numSyncDatasetChecks.get();
    }

    public long getNumSkippedChecks() {
        return this.numSkippedChecks.get();
    }

    private class ResultHandler
    implements FutureCallback<VolumeCheckResult> {
        private final FsVolumeReference reference;
        private final Set<FsVolumeSpi> failedVolumes;
        private final Set<FsVolumeSpi> healthyVolumes;
        private final AtomicLong volumeCounter;
        @Nullable
        private final Callback callback;

        ResultHandler(FsVolumeReference reference, Set<FsVolumeSpi> healthyVolumes, Set<FsVolumeSpi> failedVolumes, @Nullable AtomicLong volumeCounter, Callback callback) {
            Preconditions.checkState((reference != null ? 1 : 0) != 0);
            this.reference = reference;
            this.healthyVolumes = healthyVolumes;
            this.failedVolumes = failedVolumes;
            this.volumeCounter = volumeCounter;
            this.callback = callback;
        }

        public void onSuccess(@Nonnull VolumeCheckResult result) {
            switch (result) {
                case HEALTHY: 
                case DEGRADED: {
                    LOG.debug("Volume {} is {}.", (Object)this.reference.getVolume(), (Object)result);
                    this.markHealthy();
                    break;
                }
                case FAILED: {
                    LOG.warn("Volume {} detected as being unhealthy", (Object)this.reference.getVolume());
                    this.markFailed();
                    break;
                }
                default: {
                    LOG.error("Unexpected health check result {} for volume {}", (Object)result, (Object)this.reference.getVolume());
                    this.markHealthy();
                }
            }
            this.cleanup();
        }

        public void onFailure(@Nonnull Throwable t) {
            Throwable exception = t instanceof ExecutionException ? t.getCause() : t;
            LOG.warn("Exception running disk checks against volume " + this.reference.getVolume(), exception);
            this.markFailed();
            this.cleanup();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void markHealthy() {
            DatasetVolumeChecker datasetVolumeChecker = DatasetVolumeChecker.this;
            synchronized (datasetVolumeChecker) {
                this.healthyVolumes.add(this.reference.getVolume());
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void markFailed() {
            DatasetVolumeChecker datasetVolumeChecker = DatasetVolumeChecker.this;
            synchronized (datasetVolumeChecker) {
                this.failedVolumes.add(this.reference.getVolume());
            }
        }

        private void cleanup() {
            IOUtils.cleanup(null, (Closeable[])new Closeable[]{this.reference});
            this.invokeCallback();
        }

        private void invokeCallback() {
            try {
                long remaining = this.volumeCounter.decrementAndGet();
                if (this.callback != null && remaining == 0L) {
                    this.callback.call(this.healthyVolumes, this.failedVolumes);
                }
            }
            catch (Exception e) {
                LOG.warn("Unexpected exception", (Throwable)e);
            }
        }
    }

    public static interface Callback {
        public void call(Set<FsVolumeSpi> var1, Set<FsVolumeSpi> var2);
    }
}

