/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.protocol.datatransfer.PacketHeader;
import org.apache.hadoop.hdfs.util.DirectBufferPool;
import org.apache.hadoop.io.IOUtils;

@InterfaceAudience.Private
public class PacketReceiver
implements Closeable {
    private static final int MAX_PACKET_SIZE = 0x1000000;
    static Log LOG = LogFactory.getLog(PacketReceiver.class);
    private static final DirectBufferPool bufferPool = new DirectBufferPool();
    private final boolean useDirectBuffers;
    private final ByteBuffer lengthPrefixBuf = ByteBuffer.allocate(6);
    private ByteBuffer curPacketBuf = null;
    private ByteBuffer curChecksumSlice = null;
    private ByteBuffer curDataSlice = null;
    private PacketHeader curHeader;

    public PacketReceiver(boolean useDirectBuffers) {
        this.useDirectBuffers = useDirectBuffers;
    }

    public PacketHeader getHeader() {
        return this.curHeader;
    }

    public ByteBuffer getDataSlice() {
        return this.curDataSlice;
    }

    public ByteBuffer getChecksumSlice() {
        return this.curChecksumSlice;
    }

    public void receiveNextPacket(ReadableByteChannel in) throws IOException {
        this.doRead(in, null);
    }

    public void receiveNextPacket(InputStream in) throws IOException {
        this.doRead(null, in);
    }

    private void doRead(ReadableByteChannel ch, InputStream in) throws IOException {
        int totalLen;
        Preconditions.checkState((this.curHeader == null || !this.curHeader.isLastPacketInBlock() ? 1 : 0) != 0);
        this.lengthPrefixBuf.clear();
        PacketReceiver.doReadFully(ch, in, this.lengthPrefixBuf);
        this.lengthPrefixBuf.flip();
        int payloadLen = this.lengthPrefixBuf.getInt();
        if (payloadLen < 4) {
            throw new IOException("Invalid payload length " + payloadLen);
        }
        int dataPlusChecksumLen = payloadLen - 4;
        short headerLen = this.lengthPrefixBuf.getShort();
        if (headerLen < 0) {
            throw new IOException("Invalid header length " + headerLen);
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("readNextPacket: dataPlusChecksumLen = " + dataPlusChecksumLen + " headerLen = " + headerLen));
        }
        if ((totalLen = payloadLen + headerLen) < 0 || totalLen > 0x1000000) {
            throw new IOException("Incorrect value for packet payload size: " + payloadLen);
        }
        this.reallocPacketBuf(dataPlusChecksumLen + headerLen);
        this.curPacketBuf.clear();
        this.curPacketBuf.limit(dataPlusChecksumLen + headerLen);
        PacketReceiver.doReadFully(ch, in, this.curPacketBuf);
        this.curPacketBuf.flip();
        byte[] headerBuf = new byte[headerLen];
        this.curPacketBuf.get(headerBuf);
        if (this.curHeader == null) {
            this.curHeader = new PacketHeader();
        }
        this.curHeader.setFieldsFromData(dataPlusChecksumLen, headerBuf);
        int checksumLen = dataPlusChecksumLen - this.curHeader.getDataLen();
        if (checksumLen < 0) {
            throw new IOException("Invalid packet: data length in packet header exceeds data length received. dataPlusChecksumLen=" + dataPlusChecksumLen + " header: " + this.curHeader);
        }
        this.reslicePacket(headerLen, checksumLen, this.curHeader.getDataLen());
    }

    public void mirrorPacketTo(DataOutputStream mirrorOut) throws IOException {
        Preconditions.checkState((!this.useDirectBuffers ? 1 : 0) != 0, (Object)"Currently only supported for non-direct buffers");
        assert (this.lengthPrefixBuf.capacity() == 6);
        mirrorOut.write(this.lengthPrefixBuf.array(), this.lengthPrefixBuf.arrayOffset(), this.lengthPrefixBuf.capacity());
        mirrorOut.write(this.curPacketBuf.array(), this.curPacketBuf.arrayOffset(), this.curPacketBuf.remaining());
    }

    private static void doReadFully(ReadableByteChannel ch, InputStream in, ByteBuffer buf) throws IOException {
        if (ch != null) {
            PacketReceiver.readChannelFully(ch, buf);
        } else {
            Preconditions.checkState((!buf.isDirect() ? 1 : 0) != 0, (Object)"Must not use direct buffers with InputStream API");
            IOUtils.readFully((InputStream)in, (byte[])buf.array(), (int)(buf.arrayOffset() + buf.position()), (int)buf.remaining());
            buf.position(buf.position() + buf.remaining());
        }
    }

    private void reslicePacket(int headerLen, int checksumsLen, int dataLen) {
        assert (dataLen >= 0) : "invalid datalen: " + dataLen;
        assert (this.curPacketBuf.position() == headerLen);
        assert (checksumsLen + dataLen == this.curPacketBuf.remaining()) : "headerLen= " + headerLen + " clen=" + checksumsLen + " dlen=" + dataLen + " rem=" + this.curPacketBuf.remaining();
        this.curPacketBuf.position(headerLen);
        this.curPacketBuf.limit(headerLen + checksumsLen);
        this.curChecksumSlice = this.curPacketBuf.slice();
        this.curPacketBuf.position(headerLen + checksumsLen);
        this.curPacketBuf.limit(headerLen + checksumsLen + dataLen);
        this.curDataSlice = this.curPacketBuf.slice();
        this.curPacketBuf.position(0);
        this.curPacketBuf.limit(headerLen + checksumsLen + dataLen);
    }

    private static void readChannelFully(ReadableByteChannel ch, ByteBuffer buf) throws IOException {
        while (buf.remaining() > 0) {
            int n = ch.read(buf);
            if (n >= 0) continue;
            throw new IOException("Premature EOF reading from " + ch);
        }
    }

    private void reallocPacketBuf(int atLeastCapacity) {
        if (this.curPacketBuf == null || this.curPacketBuf.capacity() < atLeastCapacity) {
            this.returnPacketBufToPool();
            this.curPacketBuf = this.useDirectBuffers ? bufferPool.getBuffer(atLeastCapacity) : ByteBuffer.allocate(atLeastCapacity);
        }
    }

    private void returnPacketBufToPool() {
        if (this.curPacketBuf != null && this.curPacketBuf.isDirect()) {
            bufferPool.returnBuffer(this.curPacketBuf);
            this.curPacketBuf = null;
        }
    }

    @Override
    public void close() {
        this.returnPacketBufToPool();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void finalize() throws Throwable {
        try {
            this.returnPacketBufToPool();
        }
        finally {
            super.finalize();
        }
    }
}

