/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.VersionInfo;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.node.CommandQueue;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStateManager;
import org.apache.hadoop.hdds.scm.node.SCMNodeMetrics;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.ozone.protocol.VersionResponse;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.RegisteredCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeManager
implements NodeManager {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(SCMNodeManager.class);
    private final NodeStateManager nodeStateManager;
    private final String clusterID;
    private final VersionInfo version;
    private final CommandQueue commandQueue;
    private final SCMNodeMetrics metrics = SCMNodeMetrics.create();
    private ObjectName nmInfoBean;
    private final StorageContainerManager scmManager;

    public SCMNodeManager(OzoneConfiguration conf, String clusterID, StorageContainerManager scmManager, EventPublisher eventPublisher) throws IOException {
        this.nodeStateManager = new NodeStateManager((Configuration)conf, eventPublisher);
        this.clusterID = clusterID;
        this.version = VersionInfo.getLatestVersion();
        this.commandQueue = new CommandQueue();
        this.scmManager = scmManager;
        LOG.info("Entering startup chill mode.");
        this.registerMXBean();
    }

    private void registerMXBean() {
        this.nmInfoBean = MBeans.register((String)"SCMNodeManager", (String)"SCMNodeManagerInfo", (Object)this);
    }

    private void unregisterMXBean() {
        if (this.nmInfoBean != null) {
            MBeans.unregister((ObjectName)this.nmInfoBean);
            this.nmInfoBean = null;
        }
    }

    @Override
    public List<DatanodeDetails> getNodes(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodes(nodestate).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getAllNodes() {
        return this.nodeStateManager.getAllNodes().stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public int getNodeCount(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodeCount(nodestate);
    }

    @Override
    public HddsProtos.NodeState getNodeState(DatanodeDetails datanodeDetails) {
        try {
            return this.nodeStateManager.getNodeState(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            return null;
        }
    }

    @Override
    public void close() throws IOException {
        this.unregisterMXBean();
        this.metrics.unRegister();
    }

    public VersionResponse getVersion(StorageContainerDatanodeProtocolProtos.SCMVersionRequestProto versionRequest) {
        return VersionResponse.newBuilder().setVersion(this.version.getVersion()).addValue("scmUuid", this.scmManager.getScmStorageConfig().getScmId()).addValue("clusterID", this.scmManager.getScmStorageConfig().getClusterID()).build();
    }

    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto) {
        InetAddress dnAddress = Server.getRemoteIp();
        if (dnAddress != null) {
            datanodeDetails.setHostName(dnAddress.getHostName());
            datanodeDetails.setIpAddress(dnAddress.getHostAddress());
        }
        try {
            this.nodeStateManager.addNode(datanodeDetails);
            this.processNodeReport(datanodeDetails, nodeReport);
            LOG.info("Registered Data node : {}", (Object)datanodeDetails);
        }
        catch (NodeAlreadyExistsException e) {
            LOG.trace("Datanode is already registered. Datanode: {}", (Object)datanodeDetails.toString());
        }
        return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.success).setDatanodeUUID(datanodeDetails.getUuidString()).setClusterID(this.clusterID).setHostname(datanodeDetails.getHostName()).setIpAddress(datanodeDetails.getIpAddress()).build();
    }

    public List<SCMCommand> processHeartbeat(DatanodeDetails datanodeDetails) {
        Preconditions.checkNotNull((Object)datanodeDetails, (Object)"Heartbeat is missing DatanodeDetails.");
        try {
            this.nodeStateManager.updateLastHeartbeatTime(datanodeDetails);
            this.metrics.incNumHBProcessed();
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumHBProcessingFailed();
            LOG.error("SCM trying to process heartbeat from an unregistered node {}. Ignoring the heartbeat.", (Object)datanodeDetails);
        }
        return this.commandQueue.getCommand(datanodeDetails.getUuid());
    }

    public Boolean isNodeRegistered(DatanodeDetails datanodeDetails) {
        try {
            this.nodeStateManager.getNode(datanodeDetails);
            return true;
        }
        catch (NodeNotFoundException e) {
            return false;
        }
    }

    @Override
    public void processNodeReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            if (nodeReport != null) {
                datanodeInfo.updateStorageReports(nodeReport.getStorageReportList());
                this.metrics.incNumNodeReportProcessed();
            }
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumNodeReportProcessingFailed();
            LOG.warn("Got node report from unregistered datanode {}", (Object)datanodeDetails);
        }
    }

    @Override
    public SCMNodeStat getStats() {
        long capacity = 0L;
        long used = 0L;
        long remaining = 0L;
        for (SCMNodeStat stat : this.getNodeStats().values()) {
            capacity += stat.getCapacity().get().longValue();
            used += stat.getScmUsed().get().longValue();
            remaining += stat.getRemaining().get().longValue();
        }
        return new SCMNodeStat(capacity, used, remaining);
    }

    @Override
    public Map<DatanodeDetails, SCMNodeStat> getNodeStats() {
        HashMap<DatanodeDetails, SCMNodeStat> nodeStats = new HashMap<DatanodeDetails, SCMNodeStat>();
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.STALE);
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            SCMNodeStat nodeStat = this.getNodeStatInternal(dnInfo);
            if (nodeStat == null) continue;
            nodeStats.put(dnInfo, nodeStat);
        }
        return nodeStats;
    }

    @Override
    public SCMNodeMetric getNodeStat(DatanodeDetails datanodeDetails) {
        SCMNodeStat nodeStat = this.getNodeStatInternal(datanodeDetails);
        return nodeStat != null ? new SCMNodeMetric(nodeStat) : null;
    }

    private SCMNodeStat getNodeStatInternal(DatanodeDetails datanodeDetails) {
        try {
            long capacity = 0L;
            long used = 0L;
            long remaining = 0L;
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = datanodeInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                capacity += reportProto.getCapacity();
                used += reportProto.getScmUsed();
                remaining += reportProto.getRemaining();
            }
            return new SCMNodeStat(capacity, used, remaining);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)datanodeDetails.getUuid());
            return null;
        }
    }

    @Override
    public Map<String, Integer> getNodeCount() {
        HashMap<String, Integer> nodeCountMap = new HashMap<String, Integer>();
        for (HddsProtos.NodeState state : HddsProtos.NodeState.values()) {
            nodeCountMap.put(state.toString(), this.getNodeCount(state));
        }
        return nodeCountMap;
    }

    @Override
    public Map<String, Long> getNodeInfo() {
        long diskCapacity = 0L;
        long diskUsed = 0L;
        long diskRemaning = 0L;
        long ssdCapacity = 0L;
        long ssdUsed = 0L;
        long ssdRemaining = 0L;
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.STALE);
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = dnInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                if (reportProto.getStorageType() == StorageContainerDatanodeProtocolProtos.StorageTypeProto.DISK) {
                    diskCapacity += reportProto.getCapacity();
                    diskRemaning += reportProto.getRemaining();
                    diskUsed += reportProto.getScmUsed();
                    continue;
                }
                if (reportProto.getStorageType() != StorageContainerDatanodeProtocolProtos.StorageTypeProto.SSD) continue;
                ssdCapacity += reportProto.getCapacity();
                ssdRemaining += reportProto.getRemaining();
                ssdUsed += reportProto.getScmUsed();
            }
        }
        HashMap<String, Long> nodeInfo = new HashMap<String, Long>();
        nodeInfo.put("DISKCapacity", diskCapacity);
        nodeInfo.put("DISKUsed", diskUsed);
        nodeInfo.put("DISKRemaining", diskRemaning);
        nodeInfo.put("SSDCapacity", ssdCapacity);
        nodeInfo.put("SSDUsed", ssdUsed);
        nodeInfo.put("SSDRemaining", ssdRemaining);
        return nodeInfo;
    }

    @Override
    public Set<PipelineID> getPipelines(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelineByDnID(datanodeDetails.getUuid());
    }

    @Override
    public void addPipeline(Pipeline pipeline) {
        this.nodeStateManager.addPipeline(pipeline);
    }

    @Override
    public void removePipeline(Pipeline pipeline) {
        this.nodeStateManager.removePipeline(pipeline);
    }

    @Override
    public void setContainers(DatanodeDetails datanodeDetails, Set<ContainerID> containerIds) throws NodeNotFoundException {
        this.nodeStateManager.setContainers(datanodeDetails.getUuid(), containerIds);
    }

    @Override
    public Set<ContainerID> getContainers(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getContainers(datanodeDetails.getUuid());
    }

    @Override
    public void addDatanodeCommand(UUID dnId, SCMCommand command) {
        this.commandQueue.addCommand(dnId, command);
    }

    public void onMessage(CommandForDatanode commandForDatanode, EventPublisher ignored) {
        this.addDatanodeCommand(commandForDatanode.getDatanodeId(), commandForDatanode.getCommand());
    }

    @Override
    public List<SCMCommand> getCommandQueue(UUID dnID) {
        return this.commandQueue.getCommand(dnID);
    }
}

