/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificates.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.exceptions.CertificateException;
import org.apache.hadoop.util.Time;
import org.apache.logging.log4j.util.Strings;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DERUTF8String;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

public final class SelfSignedCertificate {
    private static final String NAME_FORMAT = "CN=%s,OU=%s,O=%s";
    private String subject;
    private String clusterID;
    private String scmID;
    private LocalDate beginDate;
    private LocalDate endDate;
    private KeyPair key;
    private SecurityConfig config;
    private List<GeneralName> altNames;

    private SelfSignedCertificate(Builder builder) {
        this.subject = builder.subject;
        this.clusterID = builder.clusterID;
        this.scmID = builder.scmID;
        this.beginDate = builder.beginDate;
        this.endDate = builder.endDate;
        this.config = builder.config;
        this.key = builder.key;
        this.altNames = builder.altNames;
    }

    @VisibleForTesting
    public static String getNameFormat() {
        return NAME_FORMAT;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private X509CertificateHolder generateCertificate(boolean isCA) throws OperatorCreationException, IOException {
        String dnName = String.format(SelfSignedCertificate.getNameFormat(), this.subject, this.scmID, this.clusterID);
        X500Name name = new X500Name(dnName);
        byte[] encoded = this.key.getPublic().getEncoded();
        SubjectPublicKeyInfo publicKeyInfo = SubjectPublicKeyInfo.getInstance((Object)encoded);
        ContentSigner contentSigner = new JcaContentSignerBuilder(this.config.getSignatureAlgo()).setProvider(this.config.getProvider()).build(this.key.getPrivate());
        BigInteger serial = BigInteger.ONE;
        if (!isCA) {
            serial = new BigInteger(Long.toString(Time.monotonicNow()));
        }
        ZoneOffset zoneOffset = this.beginDate.atStartOfDay(ZoneOffset.systemDefault()).getOffset();
        Date validFrom = Date.from(this.beginDate.atTime(LocalTime.MIN).toInstant(zoneOffset));
        Date validTill = Date.from(this.endDate.atTime(LocalTime.MAX).toInstant(zoneOffset));
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(name, serial, validFrom, validTill, name, publicKeyInfo);
        if (isCA) {
            builder.addExtension(Extension.basicConstraints, true, (ASN1Encodable)new BasicConstraints(true));
            int keyUsageFlag = 6;
            KeyUsage keyUsage = new KeyUsage(keyUsageFlag);
            builder.addExtension(Extension.keyUsage, true, (ASN1Encodable)keyUsage);
            if (this.altNames != null && this.altNames.size() >= 1) {
                builder.addExtension(new Extension(Extension.subjectAlternativeName, false, new GeneralNames(this.altNames.toArray(new GeneralName[this.altNames.size()])).getEncoded()));
            }
        }
        return builder.build(contentSigner);
    }

    public static class Builder {
        private String subject;
        private String clusterID;
        private String scmID;
        private LocalDate beginDate;
        private LocalDate endDate;
        private KeyPair key;
        private SecurityConfig config;
        private boolean isCA;
        private List<GeneralName> altNames;

        public Builder setConfiguration(ConfigurationSource configuration) {
            this.config = new SecurityConfig(configuration);
            return this;
        }

        public Builder setKey(KeyPair keyPair) {
            this.key = keyPair;
            return this;
        }

        public Builder setSubject(String subjectString) {
            this.subject = subjectString;
            return this;
        }

        public Builder setClusterID(String s) {
            this.clusterID = s;
            return this;
        }

        public Builder setScmID(String s) {
            this.scmID = s;
            return this;
        }

        public Builder setBeginDate(LocalDate date) {
            this.beginDate = date;
            return this;
        }

        public Builder setEndDate(LocalDate date) {
            this.endDate = date;
            return this;
        }

        public Builder makeCA() {
            this.isCA = true;
            return this;
        }

        public Builder addDnsName(String dnsName) {
            Preconditions.checkNotNull((Object)dnsName, (Object)"dnsName cannot be null");
            this.addAltName(2, dnsName);
            return this;
        }

        public Builder addIpAddress(String ip) {
            Preconditions.checkNotNull((Object)ip, (Object)"Ip address cannot be null");
            this.addAltName(7, ip);
            return this;
        }

        public Builder addServiceName(String serviceName) {
            Preconditions.checkNotNull((Object)serviceName, (Object)"Service Name cannot be null");
            this.addAltName(0, serviceName);
            return this;
        }

        private Builder addAltName(int tag, String name) {
            if (this.altNames == null) {
                this.altNames = new ArrayList<GeneralName>();
            }
            if (tag == 0) {
                ASN1Object ono = this.addOtherNameAsn1Object(name);
                this.altNames.add(new GeneralName(tag, (ASN1Encodable)ono));
            } else {
                this.altNames.add(new GeneralName(tag, name));
            }
            return this;
        }

        private ASN1Object addOtherNameAsn1Object(String name) {
            String otherNameOID = "2.16.840.1.113730.3.1.34";
            ASN1EncodableVector otherName = new ASN1EncodableVector();
            otherName.add((ASN1Encodable)new ASN1ObjectIdentifier("2.16.840.1.113730.3.1.34"));
            otherName.add((ASN1Encodable)new DERTaggedObject(true, 0, (ASN1Encodable)new DERUTF8String(name)));
            return new DERTaggedObject(false, 0, (ASN1Encodable)new DERSequence(otherName));
        }

        public X509CertificateHolder build() throws SCMSecurityException, IOException {
            Preconditions.checkNotNull((Object)this.key, (Object)"Key cannot be null");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.subject), (Object)"Subject cannot be blank");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.clusterID), (Object)"Cluster ID cannot be blank");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.scmID), (Object)"SCM ID cannot be blank");
            Preconditions.checkArgument((boolean)this.beginDate.isBefore(this.endDate), (Object)"Certificate begin date should be before end date");
            Duration certDuration = Duration.between(this.beginDate.atStartOfDay(), this.endDate.atStartOfDay());
            Duration maxDuration = this.config.getMaxCertificateDuration();
            if (certDuration.compareTo(maxDuration) > 0) {
                throw new SCMSecurityException("The cert duration violates the maximum configured value. Please check the hdds.x509.max.duration config key. Current Value: " + certDuration + " config: " + maxDuration);
            }
            SelfSignedCertificate rootCertificate = new SelfSignedCertificate(this);
            try {
                return rootCertificate.generateCertificate(this.isCA);
            }
            catch (CertIOException | OperatorCreationException e) {
                throw new CertificateException("Unable to create root certificate.", e.getCause());
            }
        }
    }
}

