/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.common;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Longs;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.OzoneChecksumException;
import org.apache.hadoop.util.PureJavaCrc32;
import org.apache.hadoop.util.PureJavaCrc32C;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Checksum {
    public static final Logger LOG = LoggerFactory.getLogger(Checksum.class);
    private final ContainerProtos.ChecksumType checksumType;
    private final int bytesPerChecksum;
    private PureJavaCrc32 crc32Checksum;
    private PureJavaCrc32C crc32cChecksum;
    private MessageDigest sha;

    public Checksum(ContainerProtos.ChecksumType type, int bytesPerChecksum) {
        this.checksumType = type;
        this.bytesPerChecksum = bytesPerChecksum;
    }

    @VisibleForTesting
    public Checksum() {
        this.checksumType = ContainerProtos.ChecksumType.valueOf("CRC32");
        this.bytesPerChecksum = 0x100000;
    }

    public ChecksumData computeChecksum(ByteBuffer byteBuffer) throws OzoneChecksumException {
        return this.computeChecksum(byteBuffer.array(), byteBuffer.position(), byteBuffer.limit());
    }

    public ChecksumData computeChecksum(byte[] data) throws OzoneChecksumException {
        return this.computeChecksum(data, 0, data.length);
    }

    public ChecksumData computeChecksum(byte[] data, int offset, int len) throws OzoneChecksumException {
        ChecksumData checksumData = new ChecksumData(this.checksumType, this.bytesPerChecksum);
        if (this.checksumType == ContainerProtos.ChecksumType.NONE) {
            return checksumData;
        }
        switch (this.checksumType) {
            case CRC32: {
                this.crc32Checksum = new PureJavaCrc32();
                break;
            }
            case CRC32C: {
                this.crc32cChecksum = new PureJavaCrc32C();
                break;
            }
            case SHA256: {
                try {
                    this.sha = MessageDigest.getInstance("SHA-256");
                    break;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new OzoneChecksumException("SHA-256", e);
                }
            }
            case MD5: {
                break;
            }
            default: {
                throw new OzoneChecksumException(this.checksumType);
            }
        }
        int dataSize = len - offset;
        int numChecksums = (dataSize + this.bytesPerChecksum - 1) / this.bytesPerChecksum;
        ArrayList<ByteString> checksumList = new ArrayList<ByteString>(numChecksums);
        for (int index = 0; index < numChecksums; ++index) {
            checksumList.add(this.computeChecksumAtIndex(data, index, offset, len));
        }
        checksumData.setChecksums(checksumList);
        return checksumData;
    }

    private ByteString computeChecksumAtIndex(byte[] data, int index, int start, int length) throws OzoneChecksumException {
        int offset = start + index * this.bytesPerChecksum;
        int len = this.bytesPerChecksum;
        int dataLength = length - start;
        if (offset + len > dataLength) {
            len = dataLength - offset;
        }
        byte[] checksumBytes = null;
        switch (this.checksumType) {
            case CRC32: {
                checksumBytes = this.computeCRC32Checksum(data, offset, len);
                break;
            }
            case CRC32C: {
                checksumBytes = this.computeCRC32CChecksum(data, offset, len);
                break;
            }
            case SHA256: {
                checksumBytes = this.computeSHA256Checksum(data, offset, len);
                break;
            }
            case MD5: {
                checksumBytes = this.computeMD5Checksum(data, offset, len);
                break;
            }
            default: {
                throw new OzoneChecksumException(this.checksumType);
            }
        }
        return ByteString.copyFrom((byte[])checksumBytes);
    }

    private byte[] computeCRC32Checksum(byte[] data, int offset, int len) {
        this.crc32Checksum.reset();
        this.crc32Checksum.update(data, offset, len);
        return Longs.toByteArray((long)this.crc32Checksum.getValue());
    }

    private byte[] computeCRC32CChecksum(byte[] data, int offset, int len) {
        this.crc32cChecksum.reset();
        this.crc32cChecksum.update(data, offset, len);
        return Longs.toByteArray((long)this.crc32cChecksum.getValue());
    }

    private byte[] computeSHA256Checksum(byte[] data, int offset, int len) {
        this.sha.reset();
        this.sha.update(data, offset, len);
        return this.sha.digest();
    }

    private byte[] computeMD5Checksum(byte[] data, int offset, int len) {
        MD5Hash md5out = MD5Hash.digest((byte[])data, (int)offset, (int)len);
        return md5out.getDigest();
    }

    public static boolean verifyChecksum(ByteString byteString, ChecksumData checksumData, int startIndex) throws OzoneChecksumException {
        return Checksum.verifyChecksum(byteString.toByteArray(), checksumData, startIndex);
    }

    public static boolean verifyChecksum(byte[] data, ChecksumData checksumData) throws OzoneChecksumException {
        return Checksum.verifyChecksum(data, checksumData, 0);
    }

    public static boolean verifyChecksum(byte[] data, ChecksumData checksumData, int startIndex) throws OzoneChecksumException {
        ContainerProtos.ChecksumType checksumType = checksumData.getChecksumType();
        if (checksumType == ContainerProtos.ChecksumType.NONE) {
            return true;
        }
        int bytesPerChecksum = checksumData.getBytesPerChecksum();
        Checksum checksum = new Checksum(checksumType, bytesPerChecksum);
        ChecksumData computedChecksumData = checksum.computeChecksum(data, 0, data.length);
        return checksumData.verifyChecksumDataMatches(computedChecksumData, startIndex);
    }

    @VisibleForTesting
    public static ContainerProtos.ChecksumData getNoChecksumDataProto() {
        return new ChecksumData(ContainerProtos.ChecksumType.NONE, 0).getProtoBufMessage();
    }
}

