/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificates.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.Date;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.exceptions.CertificateException;
import org.apache.hadoop.util.Time;
import org.apache.logging.log4j.util.Strings;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

public final class SelfSignedCertificate {
    private static final String NAME_FORMAT = "CN=%s,OU=%s,O=%s";
    private String subject;
    private String clusterID;
    private String scmID;
    private LocalDate beginDate;
    private LocalDate endDate;
    private KeyPair key;
    private SecurityConfig config;

    private SelfSignedCertificate(String subject, String scmID, String clusterID, LocalDate beginDate, LocalDate endDate, SecurityConfig configuration, KeyPair keyPair) {
        this.subject = subject;
        this.clusterID = clusterID;
        this.scmID = scmID;
        this.beginDate = beginDate;
        this.endDate = endDate;
        this.config = configuration;
        this.key = keyPair;
    }

    @VisibleForTesting
    public static String getNameFormat() {
        return NAME_FORMAT;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private X509CertificateHolder generateCertificate(boolean isCA) throws OperatorCreationException, IOException {
        String dnName = String.format(SelfSignedCertificate.getNameFormat(), this.subject, this.scmID, this.clusterID);
        X500Name name = new X500Name(dnName);
        byte[] encoded = this.key.getPublic().getEncoded();
        SubjectPublicKeyInfo publicKeyInfo = SubjectPublicKeyInfo.getInstance((Object)encoded);
        ContentSigner contentSigner = new JcaContentSignerBuilder(this.config.getSignatureAlgo()).setProvider(this.config.getProvider()).build(this.key.getPrivate());
        BigInteger serial = BigInteger.ONE;
        if (!isCA) {
            serial = new BigInteger(Long.toString(Time.monotonicNow()));
        }
        ZoneOffset zoneOffset = this.beginDate.atStartOfDay(ZoneOffset.systemDefault()).getOffset();
        Date validFrom = Date.from(this.beginDate.atTime(LocalTime.MIN).toInstant(zoneOffset));
        Date validTill = Date.from(this.endDate.atTime(LocalTime.MAX).toInstant(zoneOffset));
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(name, serial, validFrom, validTill, name, publicKeyInfo);
        if (isCA) {
            builder.addExtension(Extension.basicConstraints, true, (ASN1Encodable)new BasicConstraints(true));
            int keyUsageFlag = 6;
            KeyUsage keyUsage = new KeyUsage(keyUsageFlag);
            builder.addExtension(Extension.keyUsage, false, (ASN1Encodable)new DEROctetString((ASN1Encodable)keyUsage));
        }
        return builder.build(contentSigner);
    }

    public static class Builder {
        private String subject;
        private String clusterID;
        private String scmID;
        private LocalDate beginDate;
        private LocalDate endDate;
        private KeyPair key;
        private SecurityConfig config;
        private boolean isCA;

        public Builder setConfiguration(Configuration configuration) {
            this.config = new SecurityConfig(configuration);
            return this;
        }

        public Builder setKey(KeyPair keyPair) {
            this.key = keyPair;
            return this;
        }

        public Builder setSubject(String subjectString) {
            this.subject = subjectString;
            return this;
        }

        public Builder setClusterID(String s) {
            this.clusterID = s;
            return this;
        }

        public Builder setScmID(String s) {
            this.scmID = s;
            return this;
        }

        public Builder setBeginDate(LocalDate date) {
            this.beginDate = date;
            return this;
        }

        public Builder setEndDate(LocalDate date) {
            this.endDate = date;
            return this;
        }

        public Builder makeCA() {
            this.isCA = true;
            return this;
        }

        public X509CertificateHolder build() throws SCMSecurityException, IOException {
            Preconditions.checkNotNull((Object)this.key, (Object)"Key cannot be null");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.subject), (Object)"Subject cannot be blank");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.clusterID), (Object)"Cluster ID cannot be blank");
            Preconditions.checkArgument((boolean)Strings.isNotBlank((String)this.scmID), (Object)"SCM ID cannot be blank");
            Preconditions.checkArgument((boolean)this.beginDate.isBefore(this.endDate), (Object)"Certificate begin date should be before end date");
            Duration certDuration = Duration.between(this.beginDate.atStartOfDay(), this.endDate.atStartOfDay());
            Duration maxDuration = this.config.getMaxCertificateDuration();
            if (certDuration.compareTo(maxDuration) > 0) {
                throw new SCMSecurityException("The cert duration violates the maximum configured value. Please check the hdds.x509.max.duration config key. Current Value: " + certDuration + " config: " + maxDuration);
            }
            SelfSignedCertificate rootCertificate = new SelfSignedCertificate(this.subject, this.scmID, this.clusterID, this.beginDate, this.endDate, this.config, this.key);
            try {
                return rootCertificate.generateCertificate(this.isCA);
            }
            catch (CertIOException | OperatorCreationException e) {
                throw new CertificateException("Unable to create root certificate.", e.getCause());
            }
        }
    }
}

