/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkItem;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkStatus;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.diskbalancer.DiskBalancerException;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.NodePlan;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.Step;
import org.apache.hadoop.hdfs.web.JsonUtil;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.shaded.org.apache.commons.codec.digest.DigestUtils;
import org.apache.hadoop.util.AutoCloseableLock;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DiskBalancer {
    @VisibleForTesting
    public static final Logger LOG = LoggerFactory.getLogger(DiskBalancer.class);
    private final FsDatasetSpi<?> dataset;
    private final String dataNodeUUID;
    private final BlockMover blockMover;
    private final ReentrantLock lock;
    private final ConcurrentHashMap<VolumePair, DiskBalancerWorkItem> workMap;
    private boolean isDiskBalancerEnabled = false;
    private ExecutorService scheduler;
    private Future future;
    private String planID;
    private String planFile;
    private DiskBalancerWorkStatus.Result currentResult;
    private long bandwidth;
    private long planValidityInterval;
    private final Configuration config;

    public DiskBalancer(String dataNodeUUID, Configuration conf, BlockMover blockMover) {
        this.config = conf;
        this.currentResult = DiskBalancerWorkStatus.Result.NO_PLAN;
        this.blockMover = blockMover;
        this.dataset = this.blockMover.getDataset();
        this.dataNodeUUID = dataNodeUUID;
        this.scheduler = Executors.newSingleThreadExecutor();
        this.lock = new ReentrantLock();
        this.workMap = new ConcurrentHashMap();
        this.planID = "";
        this.planFile = "";
        this.isDiskBalancerEnabled = conf.getBoolean("dfs.disk.balancer.enabled", true);
        this.bandwidth = conf.getInt("dfs.disk.balancer.max.disk.throughputInMBperSec", 10);
        this.planValidityInterval = conf.getTimeDuration("dfs.disk.balancer.plan.valid.interval", "1d", TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        this.lock.lock();
        boolean needShutdown = false;
        try {
            this.isDiskBalancerEnabled = false;
            this.currentResult = DiskBalancerWorkStatus.Result.NO_PLAN;
            if (this.future != null && !this.future.isDone()) {
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_CANCELLED;
                this.blockMover.setExitFlag();
                this.scheduler.shutdown();
                needShutdown = true;
            }
        }
        finally {
            this.lock.unlock();
        }
        if (needShutdown) {
            this.shutdownExecutor();
        }
    }

    private void shutdownExecutor() {
        int secondsTowait = 10;
        try {
            if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
                if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                    LOG.error("Disk Balancer : Scheduler did not terminate.");
                }
            }
        }
        catch (InterruptedException ex) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void submitPlan(String planId, long planVersion, String planFileName, String planData, boolean force) throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            if (this.future != null && !this.future.isDone()) {
                LOG.error("Disk Balancer - Executing another plan, submitPlan failed.");
                throw new DiskBalancerException("Executing another plan", DiskBalancerException.Result.PLAN_ALREADY_IN_PROGRESS);
            }
            NodePlan nodePlan = this.verifyPlan(planId, planVersion, planData, force);
            this.createWorkPlan(nodePlan);
            this.planID = planId;
            this.planFile = planFileName;
            this.currentResult = DiskBalancerWorkStatus.Result.PLAN_UNDER_PROGRESS;
            this.executePlan();
        }
        finally {
            this.lock.unlock();
        }
    }

    private static FsVolumeSpi getFsVolume(FsDatasetSpi<?> fsDataset, String volUuid) {
        FsVolumeSpi fsVolume = null;
        try (FsDatasetSpi.FsVolumeReferences volumeReferences = fsDataset.getFsVolumeReferences();){
            for (int i = 0; i < volumeReferences.size(); ++i) {
                if (!volumeReferences.get(i).getStorageID().equals(volUuid)) continue;
                fsVolume = volumeReferences.get(i);
                break;
            }
        }
        catch (IOException e) {
            LOG.warn("Disk Balancer - Error when closing volume references: ", (Throwable)e);
        }
        return fsVolume;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DiskBalancerWorkStatus queryWorkStatus() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            if (this.currentResult == DiskBalancerWorkStatus.Result.PLAN_UNDER_PROGRESS && this.future != null && this.future.isDone()) {
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_DONE;
            }
            DiskBalancerWorkStatus status = new DiskBalancerWorkStatus(this.currentResult, this.planID, this.planFile);
            for (Map.Entry<VolumePair, DiskBalancerWorkItem> entry : this.workMap.entrySet()) {
                DiskBalancerWorkStatus.DiskBalancerWorkEntry workEntry = new DiskBalancerWorkStatus.DiskBalancerWorkEntry(entry.getKey().getSourceVolBasePath(), entry.getKey().getDestVolBasePath(), entry.getValue());
                status.addWorkEntry(workEntry);
            }
            DiskBalancerWorkStatus diskBalancerWorkStatus = status;
            return diskBalancerWorkStatus;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void cancelPlan(String planID) throws DiskBalancerException {
        this.lock.lock();
        boolean needShutdown = false;
        try {
            this.checkDiskBalancerEnabled();
            if (this.planID == null || !this.planID.equals(planID) || this.planID.isEmpty()) {
                LOG.error("Disk Balancer - No such plan. Cancel plan failed. PlanID: " + planID);
                throw new DiskBalancerException("No such plan.", DiskBalancerException.Result.NO_SUCH_PLAN);
            }
            if (!this.future.isDone()) {
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_CANCELLED;
                this.blockMover.setExitFlag();
                this.scheduler.shutdown();
                needShutdown = true;
            }
        }
        finally {
            this.lock.unlock();
        }
        if (needShutdown) {
            this.shutdownExecutor();
        }
    }

    public String getVolumeNames() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            String string = JsonUtil.toJsonString(this.getStorageIDToVolumeBasePathMap());
            return string;
        }
        catch (DiskBalancerException ex) {
            throw ex;
        }
        catch (IOException e) {
            throw new DiskBalancerException("Internal error, Unable to create JSON string.", e, DiskBalancerException.Result.INTERNAL_ERROR);
        }
        finally {
            this.lock.unlock();
        }
    }

    public long getBandwidth() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            long l = this.bandwidth;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void checkDiskBalancerEnabled() throws DiskBalancerException {
        if (!this.isDiskBalancerEnabled) {
            throw new DiskBalancerException("Disk Balancer is not enabled.", DiskBalancerException.Result.DISK_BALANCER_NOT_ENABLED);
        }
    }

    private NodePlan verifyPlan(String planID, long planVersion, String plan, boolean force) throws DiskBalancerException {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.verifyPlanVersion(planVersion);
        NodePlan nodePlan = this.verifyPlanHash(planID, plan);
        if (!force) {
            this.verifyTimeStamp(nodePlan);
        }
        this.verifyNodeUUID(nodePlan);
        return nodePlan;
    }

    private void verifyPlanVersion(long planVersion) throws DiskBalancerException {
        if (planVersion < 1L || planVersion > 1L) {
            LOG.error("Disk Balancer - Invalid plan version.");
            throw new DiskBalancerException("Invalid plan version.", DiskBalancerException.Result.INVALID_PLAN_VERSION);
        }
    }

    private NodePlan verifyPlanHash(String planID, String plan) throws DiskBalancerException {
        long sha1Length = 40L;
        if (plan == null || plan.length() == 0) {
            LOG.error("Disk Balancer -  Invalid plan.");
            throw new DiskBalancerException("Invalid plan.", DiskBalancerException.Result.INVALID_PLAN);
        }
        if (planID == null || (long)planID.length() != 40L || !DigestUtils.shaHex((byte[])plan.getBytes(Charset.forName("UTF-8"))).equalsIgnoreCase(planID)) {
            LOG.error("Disk Balancer - Invalid plan hash.");
            throw new DiskBalancerException("Invalid or mis-matched hash.", DiskBalancerException.Result.INVALID_PLAN_HASH);
        }
        try {
            return NodePlan.parseJson(plan);
        }
        catch (IOException ex) {
            throw new DiskBalancerException("Parsing plan failed.", ex, DiskBalancerException.Result.MALFORMED_PLAN);
        }
    }

    private void verifyTimeStamp(NodePlan plan) throws DiskBalancerException {
        long now = Time.now();
        long planTime = plan.getTimeStamp();
        if (planTime + this.planValidityInterval < now) {
            String planValidity = this.config.get("dfs.disk.balancer.plan.valid.interval", "1d");
            if (planValidity.matches("[0-9]$")) {
                planValidity = planValidity + "ms";
            }
            String errorString = "Plan was generated more than " + planValidity + " ago";
            LOG.error("Disk Balancer - " + errorString);
            throw new DiskBalancerException(errorString, DiskBalancerException.Result.OLD_PLAN_SUBMITTED);
        }
    }

    private void verifyNodeUUID(NodePlan plan) throws DiskBalancerException {
        if (plan.getNodeUUID() == null || !plan.getNodeUUID().equals(this.dataNodeUUID)) {
            LOG.error("Disk Balancer - Plan was generated for another node.");
            throw new DiskBalancerException("Plan was generated for another node.", DiskBalancerException.Result.DATANODE_ID_MISMATCH);
        }
    }

    private void createWorkPlan(NodePlan plan) throws DiskBalancerException {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.workMap.clear();
        Map<String, String> storageIDToVolBasePathMap = this.getStorageIDToVolumeBasePathMap();
        for (Step step : plan.getVolumeSetPlans()) {
            String sourceVolUuid = step.getSourceVolume().getUuid();
            String destVolUuid = step.getDestinationVolume().getUuid();
            String sourceVolBasePath = storageIDToVolBasePathMap.get(sourceVolUuid);
            if (sourceVolBasePath == null) {
                String errMsg = "Disk Balancer - Unable to find volume: " + step.getSourceVolume().getPath() + ". SubmitPlan failed.";
                LOG.error(errMsg);
                throw new DiskBalancerException(errMsg, DiskBalancerException.Result.INVALID_VOLUME);
            }
            String destVolBasePath = storageIDToVolBasePathMap.get(destVolUuid);
            if (destVolBasePath == null) {
                String errMsg = "Disk Balancer - Unable to find volume: " + step.getDestinationVolume().getPath() + ". SubmitPlan failed.";
                LOG.error(errMsg);
                throw new DiskBalancerException(errMsg, DiskBalancerException.Result.INVALID_VOLUME);
            }
            VolumePair volumePair = new VolumePair(sourceVolUuid, sourceVolBasePath, destVolUuid, destVolBasePath);
            this.createWorkPlan(volumePair, step);
        }
    }

    private Map<String, String> getStorageIDToVolumeBasePathMap() throws DiskBalancerException {
        HashMap<String, String> storageIDToVolBasePathMap = new HashMap<String, String>();
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            FsDatasetSpi.FsVolumeReferences references = this.dataset.getFsVolumeReferences();
            for (int ndx = 0; ndx < references.size(); ++ndx) {
                FsVolumeSpi vol = references.get(ndx);
                storageIDToVolBasePathMap.put(vol.getStorageID(), vol.getBaseURI().getPath());
            }
            references.close();
        }
        catch (IOException ex) {
            LOG.error("Disk Balancer - Internal Error.", (Throwable)ex);
            throw new DiskBalancerException("Internal error", ex, DiskBalancerException.Result.INTERNAL_ERROR);
        }
        return storageIDToVolBasePathMap;
    }

    private void executePlan() {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.blockMover.setRunnable();
        if (this.scheduler.isShutdown()) {
            this.scheduler = Executors.newSingleThreadExecutor();
        }
        this.future = this.scheduler.submit(new Runnable(){

            @Override
            public void run() {
                Thread.currentThread().setName("DiskBalancerThread");
                LOG.info("Executing Disk balancer plan. Plan File: {}, Plan ID: {}", (Object)DiskBalancer.this.planFile, (Object)DiskBalancer.this.planID);
                for (Map.Entry entry : DiskBalancer.this.workMap.entrySet()) {
                    DiskBalancer.this.blockMover.setRunnable();
                    DiskBalancer.this.blockMover.copyBlocks((VolumePair)entry.getKey(), (DiskBalancerWorkItem)entry.getValue());
                }
            }
        });
    }

    private void createWorkPlan(VolumePair volumePair, Step step) throws DiskBalancerException {
        if (volumePair.getSourceVolUuid().equals(volumePair.getDestVolUuid())) {
            String errMsg = "Disk Balancer - Source and destination volumes are same: " + volumePair.getSourceVolUuid();
            LOG.warn(errMsg);
            throw new DiskBalancerException(errMsg, DiskBalancerException.Result.INVALID_MOVE);
        }
        long bytesToMove = step.getBytesToMove();
        if (this.workMap.containsKey(volumePair)) {
            bytesToMove += this.workMap.get(volumePair).getBytesToCopy();
        }
        DiskBalancerWorkItem work = new DiskBalancerWorkItem(bytesToMove, 0L);
        work.setBandwidth(step.getBandwidth());
        work.setTolerancePercent(step.getTolerancePercent());
        work.setMaxDiskErrors(step.getMaxDiskErrors());
        this.workMap.put(volumePair, work);
    }

    public static class DiskBalancerMover
    implements BlockMover {
        private final FsDatasetSpi dataset;
        private long diskBandwidth;
        private long blockTolerance;
        private long maxDiskErrors;
        private int poolIndex;
        private AtomicBoolean shouldRun;
        private long startTime;
        private long secondsElapsed;

        public DiskBalancerMover(FsDatasetSpi dataset, Configuration conf) {
            this.dataset = dataset;
            this.shouldRun = new AtomicBoolean(false);
            this.diskBandwidth = conf.getLong("dfs.disk.balancer.max.disk.throughputInMBperSec", 10L);
            this.blockTolerance = conf.getLong("dfs.disk.balancer.block.tolerance.percent", 10L);
            this.maxDiskErrors = conf.getLong("dfs.disk.balancer.max.disk.errors", 5L);
            if (this.diskBandwidth <= 0L) {
                LOG.debug("Found 0 or less as max disk throughput, ignoring config value. value : " + this.diskBandwidth);
                this.diskBandwidth = 10L;
            }
            if (this.blockTolerance <= 0L) {
                LOG.debug("Found 0 or less for block tolerance value, ignoring configvalue. value : " + this.blockTolerance);
                this.blockTolerance = 10L;
            }
            if (this.maxDiskErrors < 0L) {
                LOG.debug("Found  less than 0 for maxDiskErrors value, ignoring config value. value : " + this.maxDiskErrors);
                this.maxDiskErrors = 5L;
            }
        }

        @Override
        public void setRunnable() {
            this.shouldRun.set(true);
        }

        @Override
        public void setExitFlag() {
            this.shouldRun.set(false);
        }

        public boolean shouldRun() {
            return this.shouldRun.get();
        }

        private boolean isLessThanNeeded(long blockSize, DiskBalancerWorkItem item) {
            long bytesToCopy = item.getBytesToCopy() - item.getBytesCopied();
            return blockSize <= (bytesToCopy += bytesToCopy * this.getBlockTolerancePercentage(item) / 100L);
        }

        private long getBlockTolerancePercentage(DiskBalancerWorkItem item) {
            return item.getTolerancePercent() <= 0L ? this.blockTolerance : item.getTolerancePercent();
        }

        private boolean isCloseEnough(DiskBalancerWorkItem item) {
            long temp = item.getBytesCopied() + item.getBytesCopied() * this.getBlockTolerancePercentage(item) / 100L;
            return item.getBytesToCopy() < temp;
        }

        private long getDiskBandwidth(DiskBalancerWorkItem item) {
            return item.getBandwidth() <= 0L ? this.diskBandwidth : item.getBandwidth();
        }

        @VisibleForTesting
        public long computeDelay(long bytesCopied, long timeUsed, DiskBalancerWorkItem item) {
            if (timeUsed == 0L) {
                return 0L;
            }
            int megaByte = 0x100000;
            if (bytesCopied < 0x100000L) {
                return 0L;
            }
            long bytesInMB = bytesCopied / 0x100000L;
            float bandwidth = (float)this.getDiskBandwidth(item) / 1000.0f;
            float delay = (long)((float)bytesInMB / bandwidth) - timeUsed;
            return delay <= 0.0f ? 0L : (long)delay;
        }

        private long getMaxError(DiskBalancerWorkItem item) {
            return item.getMaxDiskErrors() <= 0L ? this.maxDiskErrors : item.getMaxDiskErrors();
        }

        private ExtendedBlock getBlockToCopy(FsVolumeSpi.BlockIterator iter, DiskBalancerWorkItem item) {
            while (!iter.atEnd() && item.getErrorCount() < this.getMaxError(item)) {
                try {
                    ExtendedBlock block = iter.nextBlock();
                    if (null == block) {
                        LOG.info("NextBlock call returned null. No valid block to copy. {}", (Object)item.toJson());
                        return null;
                    }
                    if (!this.dataset.isValidBlock(block) || !this.isLessThanNeeded(block.getNumBytes(), item)) continue;
                    return block;
                }
                catch (IOException e) {
                    item.incErrorCount();
                }
            }
            if (item.getErrorCount() >= this.getMaxError(item)) {
                item.setErrMsg("Error count exceeded.");
                LOG.info("Maximum error count exceeded. Error count: {} Max error:{} ", (Object)item.getErrorCount(), (Object)item.getMaxDiskErrors());
            }
            return null;
        }

        private void openPoolIters(FsVolumeSpi source, List<FsVolumeSpi.BlockIterator> poolIters) {
            Preconditions.checkNotNull((Object)source);
            Preconditions.checkNotNull(poolIters);
            for (String blockPoolID : source.getBlockPoolList()) {
                poolIters.add(source.newBlockIterator(blockPoolID, "DiskBalancerSource"));
            }
        }

        ExtendedBlock getNextBlock(List<FsVolumeSpi.BlockIterator> poolIters, DiskBalancerWorkItem item) {
            Preconditions.checkNotNull(poolIters);
            ExtendedBlock block = null;
            for (int currentCount = 0; block == null && currentCount < poolIters.size(); ++currentCount) {
                int index = this.poolIndex++ % poolIters.size();
                FsVolumeSpi.BlockIterator currentPoolIter = poolIters.get(index);
                block = this.getBlockToCopy(currentPoolIter, item);
            }
            if (block == null) {
                try {
                    item.setErrMsg("No source blocks found to move.");
                    LOG.error("No movable source blocks found. {}", (Object)item.toJson());
                }
                catch (IOException e) {
                    LOG.error("Unable to get json from Item.");
                }
            }
            return block;
        }

        private void closePoolIters(List<FsVolumeSpi.BlockIterator> poolIters) {
            Preconditions.checkNotNull(poolIters);
            for (FsVolumeSpi.BlockIterator iter : poolIters) {
                try {
                    iter.close();
                }
                catch (IOException ex) {
                    LOG.error("Error closing a block pool iter. ex: {}", (Throwable)ex);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void copyBlocks(VolumePair pair, DiskBalancerWorkItem item) {
            String sourceVolUuid = pair.getSourceVolUuid();
            String destVolUuuid = pair.getDestVolUuid();
            FsVolumeSpi source = DiskBalancer.getFsVolume(this.dataset, sourceVolUuid);
            if (source == null) {
                String errMsg = "Disk Balancer - Unable to find source volume: " + pair.getDestVolBasePath();
                LOG.error(errMsg);
                item.setErrMsg(errMsg);
                return;
            }
            FsVolumeSpi dest = DiskBalancer.getFsVolume(this.dataset, destVolUuuid);
            if (dest == null) {
                String errMsg = "Disk Balancer - Unable to find dest volume: " + pair.getDestVolBasePath();
                LOG.error(errMsg);
                item.setErrMsg(errMsg);
                return;
            }
            if (source.isTransientStorage() || dest.isTransientStorage()) {
                String errMsg = "Disk Balancer - Unable to support transient storage type.";
                LOG.error("Disk Balancer - Unable to support transient storage type.");
                item.setErrMsg("Disk Balancer - Unable to support transient storage type.");
                return;
            }
            LinkedList<FsVolumeSpi.BlockIterator> poolIters = new LinkedList<FsVolumeSpi.BlockIterator>();
            this.startTime = Time.now();
            item.setStartTime(this.startTime);
            this.secondsElapsed = 0L;
            try {
                this.openPoolIters(source, poolIters);
                if (poolIters.size() == 0) {
                    LOG.error("No block pools found on volume. volume : {}. Exiting.", (Object)source.getBaseURI());
                    return;
                }
                while (this.shouldRun()) {
                    try {
                        if (item.getErrorCount() > this.getMaxError(item)) {
                            LOG.error("Exceeded the max error count. source {}, dest: {} error count: {}", new Object[]{source.getBaseURI(), dest.getBaseURI(), item.getErrorCount()});
                            break;
                        }
                        if (this.isCloseEnough(item)) {
                            LOG.info("Copy from {} to {} done. copied {} bytes and {} blocks.", new Object[]{source.getBaseURI(), dest.getBaseURI(), item.getBytesCopied(), item.getBlocksCopied()});
                            this.setExitFlag();
                            continue;
                        }
                        ExtendedBlock block = this.getNextBlock(poolIters, item);
                        if (block == null) {
                            LOG.error("No source blocks, exiting the copy. Source: {}, Dest:{}", (Object)source.getBaseURI(), (Object)dest.getBaseURI());
                            this.setExitFlag();
                            continue;
                        }
                        if (!this.shouldRun()) continue;
                        if (dest.getAvailable() <= item.getBytesToCopy()) {
                            LOG.error("Destination volume: {} does not have enough space to accommodate a block. Block Size: {} Exiting from copyBlocks.", (Object)dest.getBaseURI(), (Object)block.getNumBytes());
                            break;
                        }
                        long begin = System.nanoTime();
                        this.dataset.moveBlockAcrossVolumes(block, dest);
                        long now = System.nanoTime();
                        long timeUsed = now - begin > 0L ? now - begin : 0L;
                        LOG.debug("Moved block with size {} from  {} to {}", new Object[]{block.getNumBytes(), source.getBaseURI(), dest.getBaseURI()});
                        Thread.sleep(this.computeDelay(block.getNumBytes(), TimeUnit.NANOSECONDS.toMillis(timeUsed), item));
                        item.incCopiedSoFar(block.getNumBytes());
                        item.incBlocksCopied();
                        this.secondsElapsed = TimeUnit.MILLISECONDS.toSeconds(Time.now() - this.startTime);
                        item.setSecondsElapsed(this.secondsElapsed);
                    }
                    catch (IOException ex) {
                        LOG.error("Exception while trying to copy blocks. error: {}", (Throwable)ex);
                        item.incErrorCount();
                    }
                    catch (InterruptedException e) {
                        LOG.error("Copy Block Thread interrupted, exiting the copy.");
                        Thread.currentThread().interrupt();
                        item.incErrorCount();
                        this.setExitFlag();
                    }
                    catch (RuntimeException ex) {
                        LOG.error("Got an unexpected Runtime Exception {}", (Throwable)ex);
                        item.incErrorCount();
                        this.setExitFlag();
                    }
                }
            }
            finally {
                this.closePoolIters(poolIters);
            }
        }

        @Override
        public FsDatasetSpi getDataset() {
            return this.dataset;
        }

        @Override
        public long getStartTime() {
            return this.startTime;
        }

        @Override
        public long getElapsedSeconds() {
            return this.secondsElapsed;
        }
    }

    public static class VolumePair {
        private final String sourceVolUuid;
        private final String destVolUuid;
        private final String sourceVolBasePath;
        private final String destVolBasePath;

        public VolumePair(String sourceVolUuid, String sourceVolBasePath, String destVolUuid, String destVolBasePath) {
            this.sourceVolUuid = sourceVolUuid;
            this.sourceVolBasePath = sourceVolBasePath;
            this.destVolUuid = destVolUuid;
            this.destVolBasePath = destVolBasePath;
        }

        public String getSourceVolUuid() {
            return this.sourceVolUuid;
        }

        public String getSourceVolBasePath() {
            return this.sourceVolBasePath;
        }

        public String getDestVolUuid() {
            return this.destVolUuid;
        }

        public String getDestVolBasePath() {
            return this.destVolBasePath;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            VolumePair that = (VolumePair)o;
            return this.sourceVolUuid.equals(that.sourceVolUuid) && this.sourceVolBasePath.equals(that.sourceVolBasePath) && this.destVolUuid.equals(that.destVolUuid) && this.destVolBasePath.equals(that.destVolBasePath);
        }

        public int hashCode() {
            int primeNum = 31;
            List<String> volumeStrList = Arrays.asList(this.sourceVolUuid, this.sourceVolBasePath, this.destVolUuid, this.destVolBasePath);
            int result = 1;
            for (String str : volumeStrList) {
                result = result * 31 + str.hashCode();
            }
            return result;
        }
    }

    public static interface BlockMover {
        public void copyBlocks(VolumePair var1, DiskBalancerWorkItem var2);

        public void setRunnable();

        public void setExitFlag();

        public FsDatasetSpi getDataset();

        public long getStartTime();

        public long getElapsedSeconds();
    }
}

