/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.util.HashMap;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.StripedBlockInfo;
import org.apache.hadoop.hdfs.protocol.datatransfer.DataTransferProtoUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.Op;
import org.apache.hadoop.hdfs.protocol.datatransfer.Sender;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedBlockChecksumReconstructor;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedReconstructionInfo;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.LengthInputStream;
import org.apache.hadoop.hdfs.util.StripedBlockUtil;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.DataChecksum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
final class BlockChecksumHelper {
    static final Logger LOG = LoggerFactory.getLogger(BlockChecksumHelper.class);

    private BlockChecksumHelper() {
    }

    static class BlockGroupNonStripedChecksumComputer
    extends AbstractBlockChecksumComputer {
        private final ExtendedBlock blockGroup;
        private final ErasureCodingPolicy ecPolicy;
        private final DatanodeInfo[] datanodes;
        private final Token<BlockTokenIdentifier>[] blockTokens;
        private final byte[] blockIndices;
        private final long requestedNumBytes;
        private final DataOutputBuffer md5writer = new DataOutputBuffer();

        BlockGroupNonStripedChecksumComputer(DataNode datanode, StripedBlockInfo stripedBlockInfo, long requestedNumBytes) throws IOException {
            super(datanode);
            this.blockGroup = stripedBlockInfo.getBlock();
            this.ecPolicy = stripedBlockInfo.getErasureCodingPolicy();
            this.datanodes = stripedBlockInfo.getDatanodes();
            this.blockTokens = stripedBlockInfo.getBlockTokens();
            this.blockIndices = stripedBlockInfo.getBlockIndices();
            this.requestedNumBytes = requestedNumBytes;
        }

        @Override
        void compute() throws IOException {
            assert (this.datanodes.length == this.blockIndices.length);
            HashMap<Byte, LiveBlockInfo> liveDns = new HashMap<Byte, LiveBlockInfo>(this.datanodes.length);
            int blkIndxLen = this.blockIndices.length;
            int numDataUnits = this.ecPolicy.getNumDataUnits();
            for (int idx = 0; idx < blkIndxLen; ++idx) {
                liveDns.put(this.blockIndices[idx], new LiveBlockInfo(this.datanodes[idx], this.blockTokens[idx]));
            }
            long checksumLen = 0L;
            for (int idx = 0; idx < numDataUnits && idx < blkIndxLen; ++idx) {
                try {
                    ExtendedBlock block = this.getInternalBlock(numDataUnits, idx);
                    LiveBlockInfo liveBlkInfo = (LiveBlockInfo)liveDns.get((byte)idx);
                    if (liveBlkInfo == null) {
                        this.recalculateChecksum(idx, block.getNumBytes());
                    } else {
                        try {
                            this.checksumBlock(block, idx, liveBlkInfo.getToken(), liveBlkInfo.getDn());
                        }
                        catch (IOException ioe) {
                            LOG.warn("Exception while reading checksum", (Throwable)ioe);
                            this.recalculateChecksum(idx, block.getNumBytes());
                        }
                    }
                    if ((checksumLen += block.getNumBytes()) < this.requestedNumBytes) continue;
                    break;
                }
                catch (IOException e) {
                    LOG.warn("Failed to get the checksum", (Throwable)e);
                }
            }
            MD5Hash md5out = MD5Hash.digest((byte[])this.md5writer.getData());
            this.setOutBytes(md5out.getDigest());
        }

        private ExtendedBlock getInternalBlock(int numDataUnits, int idx) {
            long actualNumBytes = this.blockGroup.getNumBytes();
            this.blockGroup.setNumBytes(this.requestedNumBytes);
            ExtendedBlock block = StripedBlockUtil.constructInternalBlock((ExtendedBlock)this.blockGroup, (int)this.ecPolicy.getCellSize(), (int)numDataUnits, (int)idx);
            this.blockGroup.setNumBytes(actualNumBytes);
            return block;
        }

        private void checksumBlock(ExtendedBlock block, int blockIdx, Token<BlockTokenIdentifier> blockToken, DatanodeInfo targetDatanode) throws IOException {
            int timeout = 3000;
            try (IOStreamPair pair = this.getDatanode().connectToDN(targetDatanode, timeout, block, blockToken);){
                DataChecksum.Type ct;
                LOG.debug("write to {}: {}, block={}", new Object[]{this.getDatanode(), Op.BLOCK_CHECKSUM, block});
                this.createSender(pair).blockChecksum(block, blockToken);
                DataTransferProtos.BlockOpResponseProto reply = DataTransferProtos.BlockOpResponseProto.parseFrom((InputStream)PBHelperClient.vintPrefixed((InputStream)pair.in));
                String logInfo = "for block " + block + " from datanode " + targetDatanode;
                DataTransferProtoUtil.checkBlockOpStatus((DataTransferProtos.BlockOpResponseProto)reply, (String)logInfo);
                DataTransferProtos.OpBlockChecksumResponseProto checksumData = reply.getChecksumResponse();
                if (checksumData.hasCrcType()) {
                    ct = PBHelperClient.convert((HdfsProtos.ChecksumTypeProto)checksumData.getCrcType());
                } else {
                    LOG.debug("Retrieving checksum from an earlier-version DataNode: inferring checksum by reading first byte");
                    ct = DataChecksum.Type.DEFAULT;
                }
                this.setOrVerifyChecksumProperties(blockIdx, checksumData.getBytesPerCrc(), checksumData.getCrcPerBlock(), ct);
                MD5Hash md5 = new MD5Hash(checksumData.getMd5().toByteArray());
                md5.write((DataOutput)this.md5writer);
                LOG.debug("got reply from datanode:{}, md5={}", (Object)targetDatanode, (Object)md5);
            }
        }

        private void recalculateChecksum(int errBlkIndex, long blockLength) throws IOException {
            LOG.debug("Recalculate checksum for the missing/failed block index {}", (Object)errBlkIndex);
            byte[] errIndices = new byte[]{(byte)errBlkIndex};
            StripedReconstructionInfo stripedReconInfo = new StripedReconstructionInfo(this.blockGroup, this.ecPolicy, this.blockIndices, this.datanodes, errIndices);
            StripedBlockChecksumReconstructor checksumRecon = new StripedBlockChecksumReconstructor(this.getDatanode().getErasureCodingWorker(), stripedReconInfo, this.md5writer, blockLength);
            checksumRecon.reconstruct();
            DataChecksum checksum = checksumRecon.getChecksum();
            long crcPerBlock = checksum.getChecksumSize() <= 0 ? 0L : checksumRecon.getChecksumDataLen() / (long)checksum.getChecksumSize();
            this.setOrVerifyChecksumProperties(errBlkIndex, checksum.getBytesPerChecksum(), crcPerBlock, checksum.getChecksumType());
            LOG.debug("Recalculated checksum for the block index:{}, md5={}", (Object)errBlkIndex, (Object)checksumRecon.getMD5());
        }

        private void setOrVerifyChecksumProperties(int blockIdx, int bpc, long cpb, DataChecksum.Type ct) throws IOException {
            if (blockIdx == 0) {
                this.setBytesPerCRC(bpc);
            } else if (bpc != this.getBytesPerCRC()) {
                throw new IOException("Byte-per-checksum not matched: bpc=" + bpc + " but bytesPerCRC=" + this.getBytesPerCRC());
            }
            if (blockIdx == 0) {
                this.setCrcPerBlock(cpb);
            }
            if (blockIdx == 0) {
                this.setCrcType(ct);
            } else if (this.getCrcType() != DataChecksum.Type.MIXED && this.getCrcType() != ct) {
                this.setCrcType(DataChecksum.Type.MIXED);
            }
            if (blockIdx == 0) {
                LOG.debug("set bytesPerCRC={}, crcPerBlock={}", (Object)this.getBytesPerCRC(), (Object)this.getCrcPerBlock());
            }
        }

        private static class LiveBlockInfo {
            private final DatanodeInfo dn;
            private final Token<BlockTokenIdentifier> token;

            LiveBlockInfo(DatanodeInfo dn, Token<BlockTokenIdentifier> token) {
                this.dn = dn;
                this.token = token;
            }

            DatanodeInfo getDn() {
                return this.dn;
            }

            Token<BlockTokenIdentifier> getToken() {
                return this.token;
            }
        }
    }

    static class ReplicatedBlockChecksumComputer
    extends BlockChecksumComputer {
        ReplicatedBlockChecksumComputer(DataNode datanode, ExtendedBlock block) throws IOException {
            super(datanode, block);
        }

        @Override
        void compute() throws IOException {
            try {
                this.readHeader();
                MD5Hash md5out = this.isPartialBlk() && this.getCrcPerBlock() > 0L ? this.checksumPartialBlock() : this.checksumWholeBlock();
                this.setOutBytes(md5out.getDigest());
                LOG.debug("block={}, bytesPerCRC={}, crcPerBlock={}, md5out={}", new Object[]{this.getBlock(), this.getBytesPerCRC(), this.getCrcPerBlock(), md5out});
            }
            finally {
                IOUtils.closeStream((Closeable)this.getChecksumIn());
                IOUtils.closeStream((Closeable)this.getMetadataIn());
            }
        }

        private MD5Hash checksumWholeBlock() throws IOException {
            MD5Hash md5out = MD5Hash.digest((InputStream)this.getChecksumIn());
            return md5out;
        }

        private MD5Hash checksumPartialBlock() throws IOException {
            byte[] buffer = new byte[4096];
            MessageDigest digester = MD5Hash.getDigester();
            int toDigest = 0;
            for (long remaining = this.getRequestLength() / (long)this.getBytesPerCRC() * (long)this.getChecksumSize(); remaining > 0L && (toDigest = this.getChecksumIn().read(buffer, 0, (int)Math.min(remaining, (long)buffer.length))) >= 0; remaining -= (long)toDigest) {
                digester.update(buffer, 0, toDigest);
            }
            byte[] partialCrc = this.crcPartialBlock();
            if (partialCrc != null) {
                digester.update(partialCrc);
            }
            return new MD5Hash(digester.digest());
        }
    }

    static abstract class BlockChecksumComputer
    extends AbstractBlockChecksumComputer {
        private final ExtendedBlock block;
        private final long requestLength;
        private final LengthInputStream metadataIn;
        private final DataInputStream checksumIn;
        private final long visibleLength;
        private final boolean partialBlk;
        private BlockMetadataHeader header;
        private DataChecksum checksum;

        BlockChecksumComputer(DataNode datanode, ExtendedBlock block) throws IOException {
            super(datanode);
            this.block = block;
            this.requestLength = block.getNumBytes();
            Preconditions.checkArgument((this.requestLength >= 0L ? 1 : 0) != 0);
            this.metadataIn = datanode.data.getMetaDataInputStream(block);
            this.visibleLength = datanode.data.getReplicaVisibleLength(block);
            this.partialBlk = this.requestLength < this.visibleLength;
            int ioFileBufferSize = DFSUtilClient.getIoFileBufferSize((Configuration)datanode.getConf());
            this.checksumIn = new DataInputStream(new BufferedInputStream(this.metadataIn, ioFileBufferSize));
        }

        @Override
        Sender createSender(IOStreamPair pair) {
            DataOutputStream out = (DataOutputStream)pair.out;
            return new Sender(out);
        }

        ExtendedBlock getBlock() {
            return this.block;
        }

        long getRequestLength() {
            return this.requestLength;
        }

        LengthInputStream getMetadataIn() {
            return this.metadataIn;
        }

        DataInputStream getChecksumIn() {
            return this.checksumIn;
        }

        long getVisibleLength() {
            return this.visibleLength;
        }

        boolean isPartialBlk() {
            return this.partialBlk;
        }

        BlockMetadataHeader getHeader() {
            return this.header;
        }

        DataChecksum getChecksum() {
            return this.checksum;
        }

        @Override
        abstract void compute() throws IOException;

        void readHeader() throws IOException {
            this.header = BlockMetadataHeader.readHeader((DataInputStream)this.checksumIn);
            this.checksum = this.header.getChecksum();
            this.setChecksumSize(this.checksum.getChecksumSize());
            this.setBytesPerCRC(this.checksum.getBytesPerChecksum());
            long crcPerBlock = this.checksum.getChecksumSize() <= 0 ? 0L : (this.metadataIn.getLength() - (long)BlockMetadataHeader.getHeaderSize()) / (long)this.checksum.getChecksumSize();
            this.setCrcPerBlock(crcPerBlock);
            this.setCrcType(this.checksum.getChecksumType());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        byte[] crcPartialBlock() throws IOException {
            int partialLength = (int)(this.requestLength % (long)this.getBytesPerCRC());
            if (partialLength > 0) {
                byte[] buf = new byte[partialLength];
                InputStream blockIn = this.getBlockInputStream(this.block, this.requestLength - (long)partialLength);
                try {
                    IOUtils.readFully((InputStream)blockIn, (byte[])buf, (int)0, (int)partialLength);
                }
                finally {
                    IOUtils.closeStream((Closeable)blockIn);
                }
                this.checksum.update(buf, 0, partialLength);
                byte[] partialCrc = new byte[this.getChecksumSize()];
                this.checksum.writeValue(partialCrc, 0, true);
                return partialCrc;
            }
            return null;
        }
    }

    static abstract class AbstractBlockChecksumComputer {
        private final DataNode datanode;
        private byte[] outBytes;
        private int bytesPerCRC = -1;
        private DataChecksum.Type crcType = null;
        private long crcPerBlock = -1L;
        private int checksumSize = -1;

        AbstractBlockChecksumComputer(DataNode datanode) throws IOException {
            this.datanode = datanode;
        }

        abstract void compute() throws IOException;

        Sender createSender(IOStreamPair pair) {
            DataOutputStream out = (DataOutputStream)pair.out;
            return new Sender(out);
        }

        DataNode getDatanode() {
            return this.datanode;
        }

        InputStream getBlockInputStream(ExtendedBlock block, long seekOffset) throws IOException {
            return this.datanode.data.getBlockInputStream(block, seekOffset);
        }

        void setOutBytes(byte[] bytes) {
            this.outBytes = bytes;
        }

        byte[] getOutBytes() {
            return this.outBytes;
        }

        int getBytesPerCRC() {
            return this.bytesPerCRC;
        }

        public void setBytesPerCRC(int bytesPerCRC) {
            this.bytesPerCRC = bytesPerCRC;
        }

        public void setCrcType(DataChecksum.Type crcType) {
            this.crcType = crcType;
        }

        public void setCrcPerBlock(long crcPerBlock) {
            this.crcPerBlock = crcPerBlock;
        }

        public void setChecksumSize(int checksumSize) {
            this.checksumSize = checksumSize;
        }

        DataChecksum.Type getCrcType() {
            return this.crcType;
        }

        long getCrcPerBlock() {
            return this.crcPerBlock;
        }

        int getChecksumSize() {
            return this.checksumSize;
        }
    }
}

