/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.resource;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.CoreOptions;
import org.apache.flink.configuration.ReadableConfig;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.testutils.CommonTestUtils;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.resource.ResourceManager;
import org.apache.flink.table.resource.ResourceType;
import org.apache.flink.table.resource.ResourceUri;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.UserClassLoaderJarTestUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.CsvSource;
import org.junit.jupiter.params.provider.MethodSource;

public class ResourceManagerTest {
    @TempDir
    private static File tempFolder;
    private static File udfJar;
    private static File file;
    private ResourceManager resourceManager;

    @BeforeAll
    public static void prepare() throws Exception {
        udfJar = UserClassLoaderJarTestUtils.createJarFile((File)tempFolder, (String)"test-classloader-udf.jar", (String)"LowerUDF", (String)String.format("public class %s extends org.apache.flink.table.functions.ScalarFunction {\n  public String eval(String str) {\n    return str.toLowerCase();\n  }\n}\n", "LowerUDF"));
        file = File.createTempFile("ResourceManagerTest", ".txt", tempFolder);
        FileUtils.writeFileUtf8((File)file, (String)"Hello World");
    }

    @BeforeEach
    public void before() {
        this.resourceManager = ResourceManager.createResourceManager((URL[])new URL[0], (ClassLoader)this.getClass().getClassLoader(), (ReadableConfig)new Configuration());
    }

    @AfterEach
    public void after() throws Exception {
        this.resourceManager.close();
        FileSystem.initialize((Configuration)new Configuration(), null);
    }

    @Test
    public void testRegisterJarResource() throws Exception {
        URLClassLoader userClassLoader = this.resourceManager.getUserClassLoader();
        CommonTestUtils.assertThrows((String)"LowerUDF", ClassNotFoundException.class, () -> Class.forName("LowerUDF", false, userClassLoader));
        ResourceUri resourceUri = new ResourceUri(ResourceType.JAR, udfJar.getPath());
        this.resourceManager.registerJarResources(Arrays.asList(resourceUri, resourceUri));
        Map<ResourceUri, URL> expected = Collections.singletonMap(resourceUri, this.resourceManager.getURLFromPath(new Path(udfJar.getPath())));
        org.junit.jupiter.api.Assertions.assertEquals(expected, (Object)this.resourceManager.getResources());
        Class<?> clazz1 = Class.forName("LowerUDF", false, userClassLoader);
        Class<?> clazz2 = Class.forName("LowerUDF", false, userClassLoader);
        org.junit.jupiter.api.Assertions.assertEquals(clazz1, clazz2);
    }

    @Test
    public void testRegisterFileResource() throws Exception {
        ResourceUri normalizedResource = new ResourceUri(ResourceType.FILE, this.resourceManager.getURLFromPath(new Path(file.getPath())).getPath());
        String localFilePath = this.resourceManager.registerFileResource(new ResourceUri(ResourceType.FILE, "file://" + file.getPath()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)file.getPath(), (Object)localFilePath);
        Map<ResourceUri, URL> actualResource = Collections.singletonMap(normalizedResource, this.resourceManager.getURLFromPath(new Path(localFilePath)));
        Assertions.assertThat((Map)this.resourceManager.getResources()).containsExactlyEntriesOf(actualResource);
        Assertions.assertThat((String)this.resourceManager.registerFileResource(new ResourceUri(ResourceType.FILE, file.getPath()))).isEqualTo(localFilePath);
        Assertions.assertThat((Map)this.resourceManager.getResources()).containsExactlyEntriesOf(actualResource);
        Assertions.assertThat((String)this.resourceManager.registerFileResource(new ResourceUri(ResourceType.FILE, new File(".").getCanonicalFile().toPath().relativize(file.toPath()).toString()))).isEqualTo(localFilePath);
        Assertions.assertThat((Map)this.resourceManager.getResources()).containsExactlyEntriesOf(actualResource);
    }

    @Test
    public void testRegisterResourceWithRelativePath() throws Exception {
        URLClassLoader userClassLoader = this.resourceManager.getUserClassLoader();
        CommonTestUtils.assertThrows((String)"LowerUDF", ClassNotFoundException.class, () -> Class.forName("LowerUDF", false, userClassLoader));
        ResourceUri resourceUri = new ResourceUri(ResourceType.JAR, new File(".").getCanonicalFile().toPath().relativize(udfJar.toPath()).toString());
        this.resourceManager.registerJarResources(Collections.singletonList(resourceUri));
        Map<ResourceUri, URL> expected = Collections.singletonMap(new ResourceUri(ResourceType.JAR, udfJar.getPath()), this.resourceManager.getURLFromPath(new Path(udfJar.getPath())));
        org.junit.jupiter.api.Assertions.assertEquals(expected, (Object)this.resourceManager.getResources());
        Class<?> clazz1 = Class.forName("LowerUDF", false, userClassLoader);
        Class<?> clazz2 = Class.forName("LowerUDF", false, userClassLoader);
        org.junit.jupiter.api.Assertions.assertEquals(clazz1, clazz2);
    }

    @Test
    public void testRegisterInvalidJarResource() throws Exception {
        String fileUri = file.getPath();
        CommonTestUtils.assertThrows((String)String.format("Expect the resource type to be jar, but encounter a resource [%s] with type %s.", fileUri, ResourceType.FILE.name().toLowerCase()), ValidationException.class, () -> {
            this.resourceManager.registerJarResources(Collections.singletonList(new ResourceUri(ResourceType.FILE, fileUri)));
            return null;
        });
        CommonTestUtils.assertThrows((String)String.format("The registering or unregistering jar resource [%s] must ends with '.jar' suffix.", fileUri), ValidationException.class, () -> {
            this.resourceManager.registerJarResources(Collections.singletonList(new ResourceUri(ResourceType.JAR, fileUri)));
            return null;
        });
        String jarDir = tempFolder.getPath();
        CommonTestUtils.assertThrows((String)String.format("The registering or unregistering jar resource [%s] is a directory that is not allowed.", jarDir), ValidationException.class, () -> {
            this.resourceManager.registerJarResources(Collections.singletonList(new ResourceUri(ResourceType.JAR, jarDir)));
            return null;
        });
        String jarPath = Files.createDirectory(Paths.get(tempFolder.getPath(), "test-jar.jar"), new FileAttribute[0]).toString();
        CommonTestUtils.assertThrows((String)String.format("The registering or unregistering jar resource [%s] is a directory that is not allowed.", jarPath), ValidationException.class, () -> {
            this.resourceManager.registerJarResources(Collections.singletonList(new ResourceUri(ResourceType.JAR, jarPath)));
            return null;
        });
    }

    @MethodSource(value={"provideResource"})
    @ParameterizedTest
    public void testDownloadResource(String pathString, boolean executable) throws Exception {
        Path srcPath = new Path(pathString);
        URL localUrl = this.resourceManager.downloadResource(srcPath, executable);
        byte[] expected = FileUtils.readAllBytes((java.nio.file.Path)Paths.get(pathString, new String[0]));
        byte[] actual = FileUtils.readAllBytes((java.nio.file.Path)Paths.get(localUrl.toURI()));
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])expected, (byte[])actual);
    }

    @CsvSource(value={"file://path/to/file,hdfs://foo/bar:9000/,false", "/path/to/file,file://foo/bar/,false", "../path/to/file,file://foo/bar/,false", "/path/to/file,hdfs://foo/bar:9000/,true", "../path/to/file,hdfs://foo/bar:9000/,true", "hdfs://path/to/file,file://foo/bar/,true"})
    @ParameterizedTest
    public void testIsRemotePath(String pathString, String defaultFsScheme, boolean remote) {
        Configuration conf = new Configuration();
        conf.set(CoreOptions.DEFAULT_FILESYSTEM_SCHEME, (Object)defaultFsScheme);
        FileSystem.initialize((Configuration)conf, null);
        Assertions.assertThat((boolean)this.resourceManager.isRemotePath(new Path(pathString))).isEqualTo(remote);
    }

    @Test
    public void testSyncFileResource() throws Exception {
        String targetUri = tempFolder.getAbsolutePath() + "foo/bar.txt";
        ResourceUri resource = new ResourceUri(ResourceType.FILE, targetUri);
        this.resourceManager.syncFileResource(resource, path -> {
            try {
                FileUtils.copy((Path)new Path(file.getPath()), (Path)new Path(path), (boolean)false);
            }
            catch (IOException e) {
                org.junit.jupiter.api.Assertions.fail((String)"Test failed.", (Throwable)e);
            }
        });
        Assertions.assertThat((String)FileUtils.readFileUtf8((File)new File(targetUri))).isEqualTo("Hello World");
        this.resourceManager.syncFileResource(resource, path -> {
            try {
                Files.write(new File(targetUri).toPath(), "Bye Bye".getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);
            }
            catch (IOException e) {
                org.junit.jupiter.api.Assertions.fail((String)"Test failed.", (Throwable)e);
            }
        });
        Assertions.assertThat((String)FileUtils.readFileUtf8((File)new File(targetUri))).isEqualTo("Bye Bye");
    }

    @Test
    public void testCloseResourceManagerCleanDownloadedResources() throws Exception {
        this.resourceManager.close();
        FileSystem fileSystem = FileSystem.getLocalFileSystem();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)fileSystem.exists(this.resourceManager.getLocalResourceDir()));
    }

    public static Stream<Arguments> provideResource() {
        return Stream.of(Arguments.of((Object[])new Object[]{udfJar.getPath(), true}), Arguments.of((Object[])new Object[]{file.getPath(), false}));
    }
}

