/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.flink.annotation.Internal;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.type.TypeReference;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.dataformat.yaml.YAMLMapper;
import org.apache.flink.table.catalog.AbstractCatalogStore;
import org.apache.flink.table.catalog.CatalogDescriptor;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class FileCatalogStore
extends AbstractCatalogStore {
    private static final Logger LOG = LoggerFactory.getLogger(FileCatalogStore.class);
    static final String FILE_EXTENSION = ".yaml";
    private static final YAMLMapper YAML_MAPPER = new YAMLMapper().disable(YAMLGenerator.Feature.WRITE_DOC_START_MARKER);
    private final Path catalogStorePath;

    public FileCatalogStore(String catalogStorePath) {
        this.catalogStorePath = new Path(catalogStorePath);
    }

    @Override
    public void open() throws CatalogException {
        try {
            FileSystem fs = this.catalogStorePath.getFileSystem();
            if (!fs.exists(this.catalogStorePath)) {
                fs.mkdirs(this.catalogStorePath);
            }
            if (!fs.getFileStatus(this.catalogStorePath).isDir()) {
                throw new CatalogException(String.format("Failed to open catalog store. The given catalog store path %s is not a directory.", this.catalogStorePath));
            }
        }
        catch (CatalogException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to open file catalog store directory %s.", this.catalogStorePath), e);
        }
        super.open();
    }

    @Override
    public void storeCatalog(String catalogName, CatalogDescriptor catalog) throws CatalogException {
        this.checkOpenState();
        Path catalogPath = this.getCatalogPath(catalogName);
        try {
            FileSystem fs = catalogPath.getFileSystem();
            if (fs.exists(catalogPath)) {
                throw new CatalogException(String.format("Catalog %s's store file %s is already exist.", catalogName, catalogPath));
            }
            try (FSDataOutputStream os = fs.create(catalogPath, FileSystem.WriteMode.NO_OVERWRITE);){
                YAML_MAPPER.writeValue(os, catalog.getConfiguration().toFileWritableMap());
            }
            LOG.info("Catalog {}'s configuration saved to file {}", (Object)catalogName, (Object)catalogPath);
        }
        catch (CatalogException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to store catalog %s's configuration to file %s.", catalogName, catalogPath), e);
        }
    }

    @Override
    public void removeCatalog(String catalogName, boolean ignoreIfNotExists) throws CatalogException {
        this.checkOpenState();
        Path catalogPath = this.getCatalogPath(catalogName);
        try {
            FileSystem fs = catalogPath.getFileSystem();
            if (fs.exists(catalogPath)) {
                fs.delete(catalogPath, false);
            } else if (!ignoreIfNotExists) {
                throw new CatalogException(String.format("Catalog %s's store file %s does not exist.", catalogName, catalogPath));
            }
        }
        catch (CatalogException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to remove catalog %s's store file.", catalogName), e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Optional<CatalogDescriptor> getCatalog(String catalogName) throws CatalogException {
        this.checkOpenState();
        Path catalogPath = this.getCatalogPath(catalogName);
        try {
            FileSystem fs = catalogPath.getFileSystem();
            if (!fs.exists(catalogPath)) {
                return Optional.empty();
            }
            try (FSDataInputStream is = fs.open(catalogPath);){
                Map<String, String> configMap = YAML_MAPPER.readValue((InputStream)is, new TypeReference<Map<String, String>>(){});
                CatalogDescriptor catalog = CatalogDescriptor.of(catalogName, Configuration.fromMap(configMap));
                Optional<CatalogDescriptor> optional = Optional.of(catalog);
                return optional;
            }
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to load catalog %s's configuration from file.", catalogName), e);
        }
    }

    @Override
    public Set<String> listCatalogs() throws CatalogException {
        this.checkOpenState();
        try {
            FileStatus[] statusArr = this.catalogStorePath.getFileSystem().listStatus(this.catalogStorePath);
            return Arrays.stream(statusArr).filter(status -> !status.isDir()).map(FileStatus::getPath).map(Path::getName).map(filename -> filename.replace(FILE_EXTENSION, "")).collect(Collectors.toSet());
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to list file catalog store directory %s.", this.catalogStorePath), e);
        }
    }

    @Override
    public boolean contains(String catalogName) throws CatalogException {
        this.checkOpenState();
        Path catalogPath = this.getCatalogPath(catalogName);
        try {
            return catalogPath.getFileSystem().exists(catalogPath);
        }
        catch (Exception e) {
            throw new CatalogException(String.format("Failed to check if catalog %s exists in the catalog store.", catalogName), e);
        }
    }

    private Path getCatalogPath(String catalogName) {
        return new Path(this.catalogStorePath, catalogName + FILE_EXTENSION);
    }
}

