/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.tools;

import java.io.IOException;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.lang.WordUtils;
import org.apache.flink.hadoop.shaded.com.google.common.base.Joiner;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.CacheFlag;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveEntry;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveStats;
import org.apache.hadoop.hdfs.protocol.CachePoolEntry;
import org.apache.hadoop.hdfs.protocol.CachePoolInfo;
import org.apache.hadoop.hdfs.protocol.CachePoolStats;
import org.apache.hadoop.hdfs.tools.TableListing;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;

@InterfaceAudience.Private
public class CacheAdmin
extends Configured
implements Tool {
    private static final int MAX_LINE_WIDTH = 80;
    private static Command[] COMMANDS = new Command[]{new AddCacheDirectiveInfoCommand(), new ModifyCacheDirectiveInfoCommand(), new ListCacheDirectiveInfoCommand(), new RemoveCacheDirectiveInfoCommand(), new RemoveCacheDirectiveInfosCommand(), new AddCachePoolCommand(), new ModifyCachePoolCommand(), new RemoveCachePoolCommand(), new ListCachePoolsCommand(), new HelpCommand()};

    public CacheAdmin() {
        this(null);
    }

    public CacheAdmin(Configuration conf) {
        super(conf);
    }

    @Override
    public int run(String[] args) throws IOException {
        if (args.length == 0) {
            CacheAdmin.printUsage(false);
            return 1;
        }
        Command command = CacheAdmin.determineCommand(args[0]);
        if (command == null) {
            System.err.println("Can't understand command '" + args[0] + "'");
            if (!args[0].startsWith("-")) {
                System.err.println("Command names must start with dashes.");
            }
            CacheAdmin.printUsage(false);
            return 1;
        }
        LinkedList<String> argsList = new LinkedList<String>();
        for (int j = 1; j < args.length; ++j) {
            argsList.add(args[j]);
        }
        try {
            return command.run(this.getConf(), argsList);
        }
        catch (IllegalArgumentException e) {
            System.err.println(CacheAdmin.prettifyException(e));
            return -1;
        }
    }

    public static void main(String[] argsArray) throws IOException {
        CacheAdmin cacheAdmin = new CacheAdmin(new Configuration());
        System.exit(cacheAdmin.run(argsArray));
    }

    private static DistributedFileSystem getDFS(Configuration conf) throws IOException {
        FileSystem fs = FileSystem.get(conf);
        if (!(fs instanceof DistributedFileSystem)) {
            throw new IllegalArgumentException("FileSystem " + fs.getUri() + " is not an HDFS file system");
        }
        return (DistributedFileSystem)fs;
    }

    private static String prettifyException(Exception e) {
        return e.getClass().getSimpleName() + ": " + e.getLocalizedMessage().split("\n")[0];
    }

    private static TableListing getOptionDescriptionListing() {
        TableListing listing = new TableListing.Builder().addField("").addField("", true).wrapWidth(80).hideHeaders().build();
        return listing;
    }

    private static Long parseTtlString(String maxTtlString) throws IOException {
        Long maxTtl = null;
        if (maxTtlString != null) {
            maxTtl = maxTtlString.equalsIgnoreCase("never") ? Long.valueOf(0x1FFFFFFFFFFFFFFFL) : Long.valueOf(DFSUtil.parseRelativeTime(maxTtlString));
        }
        return maxTtl;
    }

    private static CacheDirectiveInfo.Expiration parseExpirationString(String ttlString) throws IOException {
        CacheDirectiveInfo.Expiration ex = null;
        if (ttlString != null) {
            if (ttlString.equalsIgnoreCase("never")) {
                ex = CacheDirectiveInfo.Expiration.NEVER;
            } else {
                long ttl = DFSUtil.parseRelativeTime(ttlString);
                ex = CacheDirectiveInfo.Expiration.newRelative(ttl);
            }
        }
        return ex;
    }

    private static void printUsage(boolean longUsage) {
        System.err.println("Usage: bin/hdfs cacheadmin [COMMAND]");
        for (Command command : COMMANDS) {
            if (longUsage) {
                System.err.print(command.getLongUsage());
                continue;
            }
            System.err.print("          " + command.getShortUsage());
        }
        System.err.println();
    }

    private static Command determineCommand(String commandName) {
        for (int i = 0; i < COMMANDS.length; ++i) {
            if (!COMMANDS[i].getName().equals(commandName)) continue;
            return COMMANDS[i];
        }
        return null;
    }

    private static class HelpCommand
    implements Command {
        private HelpCommand() {
        }

        @Override
        public String getName() {
            return "-help";
        }

        @Override
        public String getShortUsage() {
            return "[-help <command-name>]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<command-name>", "The command for which to get detailed help. If no command is specified, print detailed help for all commands");
            return this.getShortUsage() + "\n" + "Get detailed help about a command.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            if (args.size() == 0) {
                for (Command command : COMMANDS) {
                    System.err.println(command.getLongUsage());
                }
                return 0;
            }
            if (args.size() != 1) {
                System.out.println("You must give exactly one argument to -help.");
                return 0;
            }
            String commandName = args.get(0);
            Command command = CacheAdmin.determineCommand("-" + commandName);
            if (command == null) {
                System.err.print("Sorry, I don't know the command '" + commandName + "'.\n");
                System.err.print("Valid help command names are:\n");
                String separator = "";
                for (Command c : COMMANDS) {
                    System.err.print(separator + c.getName().substring(1));
                    separator = ", ";
                }
                System.err.print("\n");
                return 1;
            }
            System.err.print(command.getLongUsage());
            return 0;
        }
    }

    private static class ListCachePoolsCommand
    implements Command {
        private ListCachePoolsCommand() {
        }

        @Override
        public String getName() {
            return "-listPools";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " [-stats] [<name>]]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("-stats", "Display additional cache pool statistics.");
            listing.addRow("<name>", "If specified, list only the named cache pool.");
            return this.getShortUsage() + "\n" + WordUtils.wrap((String)"Display information about one or more cache pools, e.g. name, owner, group, permissions, etc.", (int)80) + "\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String name = StringUtils.popFirstNonOption(args);
            boolean printStats = StringUtils.popOption("-stats", args);
            if (!args.isEmpty()) {
                System.err.print("Can't understand arguments: " + Joiner.on(" ").join(args) + "\n");
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            TableListing.Builder builder = new TableListing.Builder().addField("NAME", TableListing.Justification.LEFT).addField("OWNER", TableListing.Justification.LEFT).addField("GROUP", TableListing.Justification.LEFT).addField("MODE", TableListing.Justification.LEFT).addField("LIMIT", TableListing.Justification.RIGHT).addField("MAXTTL", TableListing.Justification.RIGHT);
            if (printStats) {
                builder.addField("BYTES_NEEDED", TableListing.Justification.RIGHT).addField("BYTES_CACHED", TableListing.Justification.RIGHT).addField("BYTES_OVERLIMIT", TableListing.Justification.RIGHT).addField("FILES_NEEDED", TableListing.Justification.RIGHT).addField("FILES_CACHED", TableListing.Justification.RIGHT);
            }
            TableListing listing = builder.build();
            int numResults = 0;
            try {
                RemoteIterator<CachePoolEntry> iter = dfs.listCachePools();
                while (iter.hasNext()) {
                    CachePoolEntry entry = iter.next();
                    CachePoolInfo info = entry.getInfo();
                    LinkedList<String> row = new LinkedList<String>();
                    if (name != null && !info.getPoolName().equals(name)) continue;
                    row.add(info.getPoolName());
                    row.add(info.getOwnerName());
                    row.add(info.getGroupName());
                    row.add(info.getMode() != null ? info.getMode().toString() : null);
                    Long limit = info.getLimit();
                    String limitString = limit != null && limit.equals(Long.MAX_VALUE) ? "unlimited" : "" + limit;
                    row.add(limitString);
                    Long maxTtl = info.getMaxRelativeExpiryMs();
                    String maxTtlString = null;
                    if (maxTtl != null) {
                        maxTtlString = maxTtl == 0x1FFFFFFFFFFFFFFFL ? "never" : DFSUtil.durationToString(maxTtl);
                    }
                    row.add(maxTtlString);
                    if (printStats) {
                        CachePoolStats stats = entry.getStats();
                        row.add(Long.toString(stats.getBytesNeeded()));
                        row.add(Long.toString(stats.getBytesCached()));
                        row.add(Long.toString(stats.getBytesOverlimit()));
                        row.add(Long.toString(stats.getFilesNeeded()));
                        row.add(Long.toString(stats.getFilesCached()));
                    }
                    listing.addRow(row.toArray(new String[0]));
                    ++numResults;
                    if (name == null) continue;
                    break;
                }
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            System.out.print(String.format("Found %d result%s.\n", numResults, numResults == 1 ? "" : "s"));
            if (numResults > 0) {
                System.out.print(listing);
            }
            return numResults == 0 ? 1 : 0;
        }
    }

    private static class RemoveCachePoolCommand
    implements Command {
        private RemoveCachePoolCommand() {
        }

        @Override
        public String getName() {
            return "-removePool";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " <name>]\n";
        }

        @Override
        public String getLongUsage() {
            return this.getShortUsage() + "\n" + WordUtils.wrap((String)"Remove a cache pool. This also uncaches paths associated with the pool.\n\n", (int)80) + "<name>  Name of the cache pool to remove.\n";
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String name = StringUtils.popFirstNonOption(args);
            if (name == null) {
                System.err.println("You must specify a name when deleting a cache pool.");
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.print("Can't understand arguments: " + Joiner.on(" ").join(args) + "\n");
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            try {
                dfs.removeCachePool(name);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            System.out.println("Successfully removed cache pool " + name + ".");
            return 0;
        }
    }

    private static class ModifyCachePoolCommand
    implements Command {
        private ModifyCachePoolCommand() {
        }

        @Override
        public String getName() {
            return "-modifyPool";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " <name> [-owner <owner>] " + "[-group <group>] [-mode <mode>] [-limit <limit>] " + "[-maxTtl <maxTtl>]]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<name>", "Name of the pool to modify.");
            listing.addRow("<owner>", "Username of the owner of the pool");
            listing.addRow("<group>", "Groupname of the group of the pool.");
            listing.addRow("<mode>", "Unix-style permissions of the pool in octal.");
            listing.addRow("<limit>", "Maximum number of bytes that can be cached by this pool.");
            listing.addRow("<maxTtl>", "The maximum allowed time-to-live for directives being added to the pool.");
            return this.getShortUsage() + "\n" + WordUtils.wrap((String)"Modifies the metadata of an existing cache pool. See usage of -addPool for more details.", (int)80) + "\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String owner = StringUtils.popOptionWithArgument("-owner", args);
            String group = StringUtils.popOptionWithArgument("-group", args);
            String modeString = StringUtils.popOptionWithArgument("-mode", args);
            Integer mode = modeString == null ? null : Integer.valueOf(Integer.parseInt(modeString, 8));
            String limitString = StringUtils.popOptionWithArgument("-limit", args);
            Long limit = limitString == null ? null : Long.valueOf(Long.parseLong(limitString));
            String maxTtlString = StringUtils.popOptionWithArgument("-maxTtl", args);
            Long maxTtl = null;
            try {
                maxTtl = CacheAdmin.parseTtlString(maxTtlString);
            }
            catch (IOException e) {
                System.err.println("Error while parsing maxTtl value: " + e.getMessage());
                return 1;
            }
            String name = StringUtils.popFirstNonOption(args);
            if (name == null) {
                System.err.println("You must specify a name when creating a cache pool.");
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.print("Can't understand arguments: " + Joiner.on(" ").join(args) + "\n");
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            boolean changed = false;
            CachePoolInfo info = new CachePoolInfo(name);
            if (owner != null) {
                info.setOwnerName(owner);
                changed = true;
            }
            if (group != null) {
                info.setGroupName(group);
                changed = true;
            }
            if (mode != null) {
                info.setMode(new FsPermission(mode.shortValue()));
                changed = true;
            }
            if (limit != null) {
                info.setLimit(limit);
                changed = true;
            }
            if (maxTtl != null) {
                info.setMaxRelativeExpiryMs(maxTtl);
                changed = true;
            }
            if (!changed) {
                System.err.println("You must specify at least one attribute to change in the cache pool.");
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            try {
                dfs.modifyCachePool(info);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            System.out.print("Successfully modified cache pool " + name);
            String prefix = " to have ";
            if (owner != null) {
                System.out.print(prefix + "owner name " + owner);
                prefix = " and ";
            }
            if (group != null) {
                System.out.print(prefix + "group name " + group);
                prefix = " and ";
            }
            if (mode != null) {
                System.out.print(prefix + "mode " + new FsPermission(mode.shortValue()));
                prefix = " and ";
            }
            if (limit != null) {
                System.out.print(prefix + "limit " + limit);
                prefix = " and ";
            }
            if (maxTtl != null) {
                System.out.print(prefix + "max time-to-live " + maxTtlString);
            }
            System.out.print("\n");
            return 0;
        }
    }

    private static class AddCachePoolCommand
    implements Command {
        private static final String NAME = "-addPool";

        private AddCachePoolCommand() {
        }

        @Override
        public String getName() {
            return NAME;
        }

        @Override
        public String getShortUsage() {
            return "[-addPool <name> [-owner <owner>] [-group <group>] [-mode <mode>] [-limit <limit>] [-maxTtl <maxTtl>]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<name>", "Name of the new pool.");
            listing.addRow("<owner>", "Username of the owner of the pool. Defaults to the current user.");
            listing.addRow("<group>", "Group of the pool. Defaults to the primary group name of the current user.");
            listing.addRow("<mode>", "UNIX-style permissions for the pool. Permissions are specified in octal, e.g. 0755. By default, this is set to " + String.format("0%03o", FsPermission.getCachePoolDefault().toShort()) + ".");
            listing.addRow("<limit>", "The maximum number of bytes that can be cached by directives in this pool, in aggregate. By default, no limit is set.");
            listing.addRow("<maxTtl>", "The maximum allowed time-to-live for directives being added to the pool. This can be specified in seconds, minutes, hours, and days, e.g. 120s, 30m, 4h, 2d. Valid units are [smhd]. By default, no maximum is set. A value of \"never\" specifies that there is no limit.");
            return this.getShortUsage() + "\n" + "Add a new cache pool.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String limitString;
            String modeString;
            String group;
            String name = StringUtils.popFirstNonOption(args);
            if (name == null) {
                System.err.println("You must specify a name when creating a cache pool.");
                return 1;
            }
            CachePoolInfo info = new CachePoolInfo(name);
            String owner = StringUtils.popOptionWithArgument("-owner", args);
            if (owner != null) {
                info.setOwnerName(owner);
            }
            if ((group = StringUtils.popOptionWithArgument("-group", args)) != null) {
                info.setGroupName(group);
            }
            if ((modeString = StringUtils.popOptionWithArgument("-mode", args)) != null) {
                short mode = Short.parseShort(modeString, 8);
                info.setMode(new FsPermission(mode));
            }
            if ((limitString = StringUtils.popOptionWithArgument("-limit", args)) != null) {
                long limit = Long.parseLong(limitString);
                info.setLimit(limit);
            }
            String maxTtlString = StringUtils.popOptionWithArgument("-maxTtl", args);
            try {
                Long maxTtl = CacheAdmin.parseTtlString(maxTtlString);
                if (maxTtl != null) {
                    info.setMaxRelativeExpiryMs(maxTtl);
                }
            }
            catch (IOException e) {
                System.err.println("Error while parsing maxTtl value: " + e.getMessage());
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.print("Can't understand arguments: " + Joiner.on(" ").join(args) + "\n");
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            try {
                dfs.addCachePool(info);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            System.out.println("Successfully added cache pool " + name + ".");
            return 0;
        }
    }

    private static class ListCacheDirectiveInfoCommand
    implements Command {
        private ListCacheDirectiveInfoCommand() {
        }

        @Override
        public String getName() {
            return "-listDirectives";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " [-stats] [-path <path>] [-pool <pool>]]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<path>", "List only cache directives with this path. Note that if there is a cache directive for <path> in a cache pool that we don't have read access for, it will not be listed.");
            listing.addRow("<pool>", "List only path cache directives in that pool.");
            listing.addRow("-stats", "List path-based cache directive statistics.");
            return this.getShortUsage() + "\n" + "List cache directives.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String poolFilter;
            CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder();
            String pathFilter = StringUtils.popOptionWithArgument("-path", args);
            if (pathFilter != null) {
                builder.setPath(new Path(pathFilter));
            }
            if ((poolFilter = StringUtils.popOptionWithArgument("-pool", args)) != null) {
                builder.setPool(poolFilter);
            }
            boolean printStats = StringUtils.popOption("-stats", args);
            if (!args.isEmpty()) {
                System.err.println("Can't understand argument: " + args.get(0));
                return 1;
            }
            TableListing.Builder tableBuilder = new TableListing.Builder().addField("ID", TableListing.Justification.RIGHT).addField("POOL", TableListing.Justification.LEFT).addField("REPL", TableListing.Justification.RIGHT).addField("EXPIRY", TableListing.Justification.LEFT).addField("PATH", TableListing.Justification.LEFT);
            if (printStats) {
                tableBuilder.addField("BYTES_NEEDED", TableListing.Justification.RIGHT).addField("BYTES_CACHED", TableListing.Justification.RIGHT).addField("FILES_NEEDED", TableListing.Justification.RIGHT).addField("FILES_CACHED", TableListing.Justification.RIGHT);
            }
            TableListing tableListing = tableBuilder.build();
            try {
                DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
                RemoteIterator<CacheDirectiveEntry> iter = dfs.listCacheDirectives(builder.build());
                int numEntries = 0;
                while (iter.hasNext()) {
                    CacheDirectiveEntry entry = iter.next();
                    CacheDirectiveInfo directive = entry.getInfo();
                    CacheDirectiveStats stats = entry.getStats();
                    LinkedList<String> row = new LinkedList<String>();
                    row.add("" + directive.getId());
                    row.add(directive.getPool());
                    row.add("" + directive.getReplication());
                    String expiry = directive.getExpiration().getMillis() > 0xFFFFFFFFFFFFFFFL ? "never" : directive.getExpiration().toString();
                    row.add(expiry);
                    row.add(directive.getPath().toUri().getPath());
                    if (printStats) {
                        row.add("" + stats.getBytesNeeded());
                        row.add("" + stats.getBytesCached());
                        row.add("" + stats.getFilesNeeded());
                        row.add("" + stats.getFilesCached());
                    }
                    tableListing.addRow(row.toArray(new String[0]));
                    ++numEntries;
                }
                System.out.print(String.format("Found %d entr%s\n", numEntries, numEntries == 1 ? "y" : "ies"));
                if (numEntries > 0) {
                    System.out.print(tableListing);
                }
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            return 0;
        }
    }

    private static class RemoveCacheDirectiveInfosCommand
    implements Command {
        private RemoveCacheDirectiveInfosCommand() {
        }

        @Override
        public String getName() {
            return "-removeDirectives";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " -path <path>]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("-path <path>", "The path of the cache directives to remove.  You must have write permission on the pool of the directive in order to remove it.  To see a list of cache directives, use the -listDirectives command.");
            return this.getShortUsage() + "\n" + "Remove every cache directive with the specified path.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String path = StringUtils.popOptionWithArgument("-path", args);
            if (path == null) {
                System.err.println("You must specify a path with -path.");
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.println("Can't understand argument: " + args.get(0));
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            int exitCode = 0;
            try {
                DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
                RemoteIterator<CacheDirectiveEntry> iter = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPath(new Path(path)).build());
                while (iter.hasNext()) {
                    CacheDirectiveEntry entry = iter.next();
                    try {
                        dfs.removeCacheDirective(entry.getInfo().getId());
                        System.out.println("Removed cache directive " + entry.getInfo().getId());
                    }
                    catch (IOException e) {
                        System.err.println(CacheAdmin.prettifyException(e));
                        exitCode = 2;
                    }
                }
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                exitCode = 2;
            }
            if (exitCode == 0) {
                System.out.println("Removed every cache directive with path " + path);
            }
            return exitCode;
        }
    }

    private static class ModifyCacheDirectiveInfoCommand
    implements Command {
        private ModifyCacheDirectiveInfoCommand() {
        }

        @Override
        public String getName() {
            return "-modifyDirective";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " -id <id> [-path <path>] [-force] [-replication <replication>] " + "[-pool <pool-name>] [-ttl <time-to-live>]]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<id>", "The ID of the directive to modify (required)");
            listing.addRow("<path>", "A path to cache. The path can be a directory or a file. (optional)");
            listing.addRow("-force", "Skips checking of cache pool resource limits.");
            listing.addRow("<replication>", "The cache replication factor to use. (optional)");
            listing.addRow("<pool-name>", "The pool to which the directive will be added. You must have write permission on the cache pool in order to move a directive into it. (optional)");
            listing.addRow("<time-to-live>", "How long the directive is valid. Can be specified in minutes, hours, and days, e.g. 30m, 4h, 2d. Valid units are [smhd]. \"never\" indicates a directive that never expires.");
            return this.getShortUsage() + "\n" + "Modify a cache directive.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            String poolName;
            CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder();
            boolean modified = false;
            String idString = StringUtils.popOptionWithArgument("-id", args);
            if (idString == null) {
                System.err.println("You must specify a directive ID with -id.");
                return 1;
            }
            builder.setId(Long.parseLong(idString));
            String path = StringUtils.popOptionWithArgument("-path", args);
            if (path != null) {
                builder.setPath(new Path(path));
                modified = true;
            }
            boolean force = StringUtils.popOption("-force", args);
            String replicationString = StringUtils.popOptionWithArgument("-replication", args);
            if (replicationString != null) {
                builder.setReplication(Short.parseShort(replicationString));
                modified = true;
            }
            if ((poolName = StringUtils.popOptionWithArgument("-pool", args)) != null) {
                builder.setPool(poolName);
                modified = true;
            }
            String ttlString = StringUtils.popOptionWithArgument("-ttl", args);
            try {
                CacheDirectiveInfo.Expiration ex = CacheAdmin.parseExpirationString(ttlString);
                if (ex != null) {
                    builder.setExpiration(ex);
                    modified = true;
                }
            }
            catch (IOException e) {
                System.err.println("Error while parsing ttl value: " + e.getMessage());
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.println("Can't understand argument: " + args.get(0));
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            if (!modified) {
                System.err.println("No modifications were specified.");
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            EnumSet<CacheFlag> flags = EnumSet.noneOf(CacheFlag.class);
            if (force) {
                flags.add(CacheFlag.FORCE);
            }
            try {
                dfs.modifyCacheDirective(builder.build(), flags);
                System.out.println("Modified cache directive " + idString);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            return 0;
        }
    }

    private static class RemoveCacheDirectiveInfoCommand
    implements Command {
        private RemoveCacheDirectiveInfoCommand() {
        }

        @Override
        public String getName() {
            return "-removeDirective";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " <id>]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<id>", "The id of the cache directive to remove.  You must have write permission on the pool of the directive in order to remove it.  To see a list of cache directive IDs, use the -listDirectives command.");
            return this.getShortUsage() + "\n" + "Remove a cache directive.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            long id;
            String idString = StringUtils.popFirstNonOption(args);
            if (idString == null) {
                System.err.println("You must specify a directive ID to remove.");
                return 1;
            }
            try {
                id = Long.valueOf(idString);
            }
            catch (NumberFormatException e) {
                System.err.println("Invalid directive ID " + idString + ": expected " + "a numeric value.");
                return 1;
            }
            if (id <= 0L) {
                System.err.println("Invalid directive ID " + id + ": ids must " + "be greater than 0.");
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.println("Can't understand argument: " + args.get(0));
                System.err.println("Usage is " + this.getShortUsage());
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            try {
                dfs.getClient().removeCacheDirective(id);
                System.out.println("Removed cached directive " + id);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            return 0;
        }
    }

    private static class AddCacheDirectiveInfoCommand
    implements Command {
        private AddCacheDirectiveInfoCommand() {
        }

        @Override
        public String getName() {
            return "-addDirective";
        }

        @Override
        public String getShortUsage() {
            return "[" + this.getName() + " -path <path> -pool <pool-name> " + "[-force] " + "[-replication <replication>] [-ttl <time-to-live>]]\n";
        }

        @Override
        public String getLongUsage() {
            TableListing listing = CacheAdmin.getOptionDescriptionListing();
            listing.addRow("<path>", "A path to cache. The path can be a directory or a file.");
            listing.addRow("<pool-name>", "The pool to which the directive will be added. You must have write permission on the cache pool in order to add new directives.");
            listing.addRow("-force", "Skips checking of cache pool resource limits.");
            listing.addRow("<replication>", "The cache replication factor to use. Defaults to 1.");
            listing.addRow("<time-to-live>", "How long the directive is valid. Can be specified in minutes, hours, and days, e.g. 30m, 4h, 2d. Valid units are [smhd]. \"never\" indicates a directive that never expires. If unspecified, the directive never expires.");
            return this.getShortUsage() + "\n" + "Add a new cache directive.\n\n" + listing.toString();
        }

        @Override
        public int run(Configuration conf, List<String> args) throws IOException {
            CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder();
            String path = StringUtils.popOptionWithArgument("-path", args);
            if (path == null) {
                System.err.println("You must specify a path with -path.");
                return 1;
            }
            builder.setPath(new Path(path));
            String poolName = StringUtils.popOptionWithArgument("-pool", args);
            if (poolName == null) {
                System.err.println("You must specify a pool name with -pool.");
                return 1;
            }
            builder.setPool(poolName);
            boolean force = StringUtils.popOption("-force", args);
            String replicationString = StringUtils.popOptionWithArgument("-replication", args);
            if (replicationString != null) {
                Short replication = Short.parseShort(replicationString);
                builder.setReplication(replication);
            }
            String ttlString = StringUtils.popOptionWithArgument("-ttl", args);
            try {
                CacheDirectiveInfo.Expiration ex = CacheAdmin.parseExpirationString(ttlString);
                if (ex != null) {
                    builder.setExpiration(ex);
                }
            }
            catch (IOException e) {
                System.err.println("Error while parsing ttl value: " + e.getMessage());
                return 1;
            }
            if (!args.isEmpty()) {
                System.err.println("Can't understand argument: " + args.get(0));
                return 1;
            }
            DistributedFileSystem dfs = CacheAdmin.getDFS(conf);
            CacheDirectiveInfo directive = builder.build();
            EnumSet<CacheFlag> flags = EnumSet.noneOf(CacheFlag.class);
            if (force) {
                flags.add(CacheFlag.FORCE);
            }
            try {
                long id = dfs.addCacheDirective(directive, flags);
                System.out.println("Added cache directive " + id);
            }
            catch (IOException e) {
                System.err.println(CacheAdmin.prettifyException(e));
                return 2;
            }
            return 0;
        }
    }

    static interface Command {
        public String getName();

        public String getShortUsage();

        public String getLongUsage();

        public int run(Configuration var1, List<String> var2) throws IOException;
    }
}

