/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.shaded.hadoop2.com.google.common.annotations.VisibleForTesting;
import org.apache.flink.shaded.hadoop2.com.google.common.base.Preconditions;
import org.apache.flink.shaded.hadoop2.com.google.common.collect.Lists;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.hdfs.protocol.OpenFileEntry;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private long lastHolderUpdateTime;
    private String internalLeaseHolder;
    private final SortedMap<String, Lease> leases = new TreeMap<String, Lease>();
    private final NavigableSet<Lease> sortedLeases = new TreeSet<Lease>(new Comparator<Lease>(){

        @Override
        public int compare(Lease o1, Lease o2) {
            if (o1.getLastUpdate() != o2.getLastUpdate()) {
                return Long.signum(o1.getLastUpdate() - o2.getLastUpdate());
            }
            return o1.holder.compareTo(o2.holder);
        }
    });
    private final TreeMap<Long, Lease> leasesById = new TreeMap();
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
        this.updateInternalLeaseHolder();
    }

    private void updateInternalLeaseHolder() {
        this.lastHolderUpdateTime = Time.monotonicNow();
        this.internalLeaseHolder = "HDFS_NameNode-" + Time.formatTime(Time.now());
    }

    String getInternalLeaseHolder() {
        long elapsed = Time.monotonicNow() - this.lastHolderUpdateTime;
        if (elapsed > this.hardLimit) {
            this.updateInternalLeaseHolder();
        }
        return this.internalLeaseHolder;
    }

    Lease getLease(String holder) {
        return (Lease)this.leases.get(holder);
    }

    synchronized long getNumUnderConstructionBlocks() {
        assert (this.fsnamesystem.hasReadLock()) : "The FSNamesystem read lock wasn'tacquired before counting under construction blocks";
        long numUCBlocks = 0L;
        for (Long id : this.getINodeIdWithLeases()) {
            INodeFile cons = this.fsnamesystem.getFSDirectory().getInode(id).asFile();
            if (!cons.isUnderConstruction()) {
                LOG.warn((Object)("The file " + cons.getFullPathName() + " is not under construction but has lease."));
                continue;
            }
            BlockInfo[] blocks = cons.getBlocks();
            if (blocks == null) continue;
            for (BlockInfo b : blocks) {
                if (b.isComplete()) continue;
                ++numUCBlocks;
            }
        }
        LOG.info((Object)("Number of blocks under construction: " + numUCBlocks));
        return numUCBlocks;
    }

    Collection<Long> getINodeIdWithLeases() {
        return this.leasesById.keySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BatchedRemoteIterator.BatchedListEntries<OpenFileEntry> getUnderConstructionFiles(long prevId) throws IOException {
        NavigableMap<Long, Lease> remainingLeases;
        assert (this.fsnamesystem.hasReadLock());
        LeaseManager leaseManager = this;
        synchronized (leaseManager) {
            remainingLeases = this.leasesById.tailMap(prevId, false);
        }
        Set inodeIds = remainingLeases.keySet();
        int numResponses = Math.min(this.fsnamesystem.getMaxListOpenFilesResponses(), inodeIds.size());
        ArrayList<OpenFileEntry> openFileEntries = Lists.newArrayListWithExpectedSize(numResponses);
        int count = 0;
        for (Long inodeId : inodeIds) {
            INodeFile inodeFile = this.fsnamesystem.getFSDirectory().getInode(inodeId).asFile();
            if (!inodeFile.isUnderConstruction()) {
                LOG.warn((Object)("The file " + inodeFile.getFullPathName() + " is not under construction but has lease."));
                continue;
            }
            openFileEntries.add(new OpenFileEntry(inodeFile.getId(), inodeFile.getFullPathName(), inodeFile.getFileUnderConstructionFeature().getClientName(), inodeFile.getFileUnderConstructionFeature().getClientMachine()));
            if (++count < numResponses) continue;
            break;
        }
        boolean hasMore = numResponses < remainingLeases.size();
        return new BatchedRemoteIterator.BatchedListEntries<OpenFileEntry>(openFileEntries, hasMore);
    }

    public synchronized Lease getLease(INodeFile src) {
        return this.leasesById.get(src.getId());
    }

    @VisibleForTesting
    public synchronized int countLease() {
        return this.sortedLeases.size();
    }

    synchronized long countPath() {
        return this.leasesById.size();
    }

    synchronized Lease addLease(String holder, long inodeId) {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
            this.sortedLeases.add(lease);
        } else {
            this.renewLease(lease);
        }
        this.leasesById.put(inodeId, lease);
        lease.files.add(inodeId);
        return lease;
    }

    synchronized void removeLease(long inodeId) {
        Lease lease = this.leasesById.get(inodeId);
        if (lease != null) {
            this.removeLease(lease, inodeId);
        }
    }

    private synchronized void removeLease(Lease lease, long inodeId) {
        this.leasesById.remove(inodeId);
        if (!lease.removeFile(inodeId) && LOG.isDebugEnabled()) {
            LOG.debug((Object)("inode " + inodeId + " not found in lease.files (=" + lease + ")"));
        }
        if (!lease.hasFiles()) {
            this.leases.remove(lease.holder);
            if (!this.sortedLeases.remove(lease)) {
                LOG.error((Object)(lease + " not found in sortedLeases"));
            }
        }
    }

    synchronized void removeLease(String holder, INodeFile src) {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, src.getId());
        } else {
            LOG.warn((Object)("Removing non-existent lease! holder=" + holder + " src=" + src.getFullPathName()));
        }
    }

    synchronized void removeAllLeases() {
        this.sortedLeases.clear();
        this.leasesById.clear();
        this.leases.clear();
    }

    synchronized Lease reassignLease(Lease lease, INodeFile src, String newHolder) {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            this.removeLease(lease, src.getId());
        }
        return this.addLease(newHolder, src.getId());
    }

    synchronized void renewLease(String holder) {
        this.renewLease(this.getLease(holder));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            this.sortedLeases.remove(lease);
            lease.renew();
            this.sortedLeases.add(lease);
        }
    }

    synchronized void renewAllLeases() {
        for (Lease l : this.leases.values()) {
            this.renewLease(l);
        }
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    @VisibleForTesting
    synchronized boolean checkLeases() {
        boolean needSync = false;
        assert (this.fsnamesystem.hasWriteLock());
        long start = Time.monotonicNow();
        while (!this.sortedLeases.isEmpty() && ((Lease)this.sortedLeases.first()).expiredHardLimit() && !this.isMaxLockHoldToReleaseLease(start)) {
            Lease leaseToCheck = (Lease)this.sortedLeases.first();
            LOG.info((Object)(leaseToCheck + " has expired hard limit"));
            ArrayList<Long> removing = new ArrayList<Long>();
            Collection files = leaseToCheck.getFiles();
            Long[] leaseINodeIds = files.toArray(new Long[files.size()]);
            FSDirectory fsd = this.fsnamesystem.getFSDirectory();
            String p = null;
            String newHolder = this.getInternalLeaseHolder();
            for (Long id : leaseINodeIds) {
                try {
                    INodesInPath iip = INodesInPath.fromINode(fsd.getInode(id));
                    p = iip.getPath();
                    if (!p.startsWith("/")) {
                        throw new IOException("Invalid path in the lease " + p);
                    }
                    INodeFile lastINode = iip.getLastINode().asFile();
                    if (this.fsnamesystem.isFileDeleted(lastINode)) {
                        this.removeLease(lastINode.getId());
                        continue;
                    }
                    boolean completed = false;
                    try {
                        completed = this.fsnamesystem.internalReleaseLease(leaseToCheck, p, iip, newHolder);
                    }
                    catch (IOException e) {
                        LOG.warn((Object)("Cannot release the path " + p + " in the lease " + leaseToCheck + ". It will be retried."), (Throwable)e);
                        continue;
                    }
                    if (LOG.isDebugEnabled()) {
                        if (completed) {
                            LOG.debug((Object)("Lease recovery for inode " + id + " is complete. " + "File closed."));
                        } else {
                            LOG.debug((Object)("Started block recovery " + p + " lease " + leaseToCheck));
                        }
                    }
                    if (!needSync && !completed) {
                        needSync = true;
                    }
                }
                catch (IOException e) {
                    LOG.warn((Object)("Removing lease with an invalid path: " + p + "," + leaseToCheck), (Throwable)e);
                    removing.add(id);
                }
                if (!this.isMaxLockHoldToReleaseLease(start)) continue;
                LOG.debug((Object)("Breaking out of checkLeases after " + this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs() + "ms."));
                break;
            }
            for (Long id : removing) {
                this.removeLease(leaseToCheck, id);
            }
        }
        return needSync;
    }

    private boolean isMaxLockHoldToReleaseLease(long start) {
        return Time.monotonicNow() - start > this.fsnamesystem.getMaxLockHoldToReleaseLeaseMs();
    }

    public synchronized String toString() {
        return this.getClass().getSimpleName() + "= {" + "\n leases=" + this.leases + "\n sortedLeases=" + this.sortedLeases + "\n leasesById=" + this.leasesById + "\n}";
    }

    void startMonitor() {
        Preconditions.checkState(this.lmthread == null, "Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon(new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn((Object)"Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    public void triggerMonitorCheckNow() {
        Preconditions.checkState(this.lmthread != null, "Lease monitor is not running");
        this.lmthread.interrupt();
    }

    @VisibleForTesting
    public void runLeaseChecks() {
        this.checkLeases();
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                boolean needSync = false;
                try {
                    LeaseManager.this.fsnamesystem.writeLockInterruptibly();
                    try {
                        if (!LeaseManager.this.fsnamesystem.isInSafeMode()) {
                            needSync = LeaseManager.this.checkLeases();
                        }
                    }
                    finally {
                        LeaseManager.this.fsnamesystem.writeUnlock("leaseManager");
                        if (needSync) {
                            LeaseManager.this.fsnamesystem.getEditLog().logSync();
                        }
                    }
                    Thread.sleep(LeaseManager.this.fsnamesystem.getLeaseRecheckIntervalMs());
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(this.name + " is interrupted"), (Throwable)ie);
                }
                catch (Throwable e) {
                    LOG.warn((Object)"Unexpected throwable: ", e);
                }
            }
        }
    }

    class Lease {
        private final String holder;
        private long lastUpdate;
        private final HashSet<Long> files = new HashSet();

        private Lease(String holder) {
            this.holder = holder;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = Time.monotonicNow();
        }

        public boolean expiredHardLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        boolean hasFiles() {
            return !this.files.isEmpty();
        }

        boolean removeFile(long inodeId) {
            return this.files.remove(inodeId);
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pending creates: " + this.files.size() + "]";
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        private Collection<Long> getFiles() {
            return Collections.unmodifiableCollection(this.files);
        }

        String getHolder() {
            return this.holder;
        }

        @VisibleForTesting
        long getLastUpdate() {
            return this.lastUpdate;
        }
    }
}

