/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.flink.shaded.hadoop2.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.HardLink;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.InconsistentFSStateException;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.StorageInfo;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeLayoutVersion;
import org.apache.hadoop.hdfs.server.datanode.DataStorage;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.util.Daemon;

@InterfaceAudience.Private
public class BlockPoolSliceStorage
extends Storage {
    static final String TRASH_ROOT_DIR = "trash";
    private static final String BLOCK_POOL_ID_PATTERN_BASE = Pattern.quote(File.separator) + "BP-\\d+-\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}-\\d+" + Pattern.quote(File.separator);
    private static final Pattern BLOCK_POOL_PATH_PATTERN = Pattern.compile("^(.*)(" + BLOCK_POOL_ID_PATTERN_BASE + ")(.*)$");
    private static final Pattern BLOCK_POOL_CURRENT_PATH_PATTERN = Pattern.compile("^(.*)(" + BLOCK_POOL_ID_PATTERN_BASE + ")(" + "current" + ")(.*)$");
    private static final Pattern BLOCK_POOL_TRASH_PATH_PATTERN = Pattern.compile("^(.*)(" + BLOCK_POOL_ID_PATTERN_BASE + ")(" + "trash" + ")(.*)$");
    private String blockpoolID = "";

    public BlockPoolSliceStorage(StorageInfo storageInfo, String bpid) {
        super(storageInfo);
        this.blockpoolID = bpid;
    }

    BlockPoolSliceStorage(int namespaceID, String bpID, long cTime, String clusterId) {
        super(HdfsServerConstants.NodeType.DATA_NODE);
        this.namespaceID = namespaceID;
        this.blockpoolID = bpID;
        this.cTime = cTime;
        this.clusterID = clusterId;
    }

    private BlockPoolSliceStorage() {
        super(HdfsServerConstants.NodeType.DATA_NODE);
    }

    void recoverTransitionRead(DataNode datanode, NamespaceInfo nsInfo, Collection<File> dataDirs, HdfsServerConstants.StartupOption startOpt) throws IOException {
        LOG.info((Object)("Analyzing storage directories for bpid " + nsInfo.getBlockPoolID()));
        this.storageDirs = new ArrayList(dataDirs.size());
        ArrayList<Storage.StorageState> dataDirStates = new ArrayList<Storage.StorageState>(dataDirs.size());
        Iterator<File> it = dataDirs.iterator();
        block7: while (it.hasNext()) {
            Storage.StorageState curState;
            File dataDir = it.next();
            Storage.StorageDirectory sd = new Storage.StorageDirectory(dataDir, null, true);
            try {
                curState = sd.analyzeStorage(startOpt, this);
                switch (curState) {
                    case NORMAL: {
                        break;
                    }
                    case NON_EXISTENT: {
                        LOG.info((Object)("Storage directory " + dataDir + " does not exist."));
                        it.remove();
                        continue block7;
                    }
                    case NOT_FORMATTED: {
                        LOG.info((Object)("Storage directory " + dataDir + " is not formatted."));
                        LOG.info((Object)"Formatting ...");
                        this.format(sd, nsInfo);
                        break;
                    }
                    default: {
                        sd.doRecover(curState);
                        break;
                    }
                }
            }
            catch (IOException ioe) {
                sd.unlock();
                throw ioe;
            }
            this.addStorageDir(sd);
            dataDirStates.add(curState);
        }
        if (dataDirs.size() == 0) {
            throw new IOException("All specified directories are not accessible or do not exist.");
        }
        for (int idx = 0; idx < this.getNumStorageDirs(); ++idx) {
            this.doTransition(this.getStorageDir(idx), nsInfo, startOpt);
            assert (this.getCTime() == nsInfo.getCTime()) : "Data-node and name-node CTimes must be the same.";
        }
        this.writeAll();
    }

    void format(File dnCurDir, NamespaceInfo nsInfo) throws IOException {
        File curBpDir = BlockPoolSliceStorage.getBpRoot(nsInfo.getBlockPoolID(), dnCurDir);
        Storage.StorageDirectory bpSdir = new Storage.StorageDirectory(curBpDir);
        this.format(bpSdir, nsInfo);
    }

    private void format(Storage.StorageDirectory bpSdir, NamespaceInfo nsInfo) throws IOException {
        LOG.info((Object)("Formatting block pool " + this.blockpoolID + " directory " + bpSdir.getCurrentDir()));
        bpSdir.clearDirectory();
        this.layoutVersion = HdfsConstants.DATANODE_LAYOUT_VERSION;
        this.cTime = nsInfo.getCTime();
        this.namespaceID = nsInfo.getNamespaceID();
        this.blockpoolID = nsInfo.getBlockPoolID();
        this.writeProperties(bpSdir);
    }

    @Override
    protected void setPropertiesFromFields(Properties props, Storage.StorageDirectory sd) throws IOException {
        props.setProperty("layoutVersion", String.valueOf(this.layoutVersion));
        props.setProperty("namespaceID", String.valueOf(this.namespaceID));
        props.setProperty("blockpoolID", this.blockpoolID);
        props.setProperty("cTime", String.valueOf(this.cTime));
    }

    private void setBlockPoolID(File storage, String bpid) throws InconsistentFSStateException {
        if (bpid == null || bpid.equals("")) {
            throw new InconsistentFSStateException(storage, "file VERSION is invalid.");
        }
        if (!this.blockpoolID.equals("") && !this.blockpoolID.equals(bpid)) {
            throw new InconsistentFSStateException(storage, "Unexpected blockpoolID " + bpid + ". Expected " + this.blockpoolID);
        }
        this.blockpoolID = bpid;
    }

    @Override
    protected void setFieldsFromProperties(Properties props, Storage.StorageDirectory sd) throws IOException {
        this.setLayoutVersion(props, sd);
        this.setNamespaceID(props, sd);
        this.setcTime(props, sd);
        String sbpid = props.getProperty("blockpoolID");
        this.setBlockPoolID(sd.getRoot(), sbpid);
    }

    private void doTransition(Storage.StorageDirectory sd, NamespaceInfo nsInfo, HdfsServerConstants.StartupOption startOpt) throws IOException {
        if (startOpt == HdfsServerConstants.StartupOption.ROLLBACK) {
            this.doRollback(sd, nsInfo);
        } else {
            int restored = this.restoreBlockFilesFromTrash(this.getTrashRootDir(sd));
            LOG.info((Object)("Restored " + restored + " block files from trash."));
        }
        this.readProperties(sd);
        BlockPoolSliceStorage.checkVersionUpgradable(this.layoutVersion);
        assert (this.layoutVersion >= HdfsConstants.DATANODE_LAYOUT_VERSION) : "Future version is not allowed";
        if (this.getNamespaceID() != nsInfo.getNamespaceID()) {
            throw new IOException("Incompatible namespaceIDs in " + sd.getRoot().getCanonicalPath() + ": namenode namespaceID = " + nsInfo.getNamespaceID() + "; datanode namespaceID = " + this.getNamespaceID());
        }
        if (!this.blockpoolID.equals(nsInfo.getBlockPoolID())) {
            throw new IOException("Incompatible blockpoolIDs in " + sd.getRoot().getCanonicalPath() + ": namenode blockpoolID = " + nsInfo.getBlockPoolID() + "; datanode blockpoolID = " + this.blockpoolID);
        }
        if (this.layoutVersion == HdfsConstants.DATANODE_LAYOUT_VERSION && this.cTime == nsInfo.getCTime()) {
            return;
        }
        if (this.layoutVersion > HdfsConstants.DATANODE_LAYOUT_VERSION || this.cTime < nsInfo.getCTime()) {
            this.doUpgrade(sd, nsInfo);
            return;
        }
        throw new IOException("Datanode state: LV = " + this.getLayoutVersion() + " CTime = " + this.getCTime() + " is newer than the namespace state: LV = " + nsInfo.getLayoutVersion() + " CTime = " + nsInfo.getCTime());
    }

    void doUpgrade(Storage.StorageDirectory bpSd, NamespaceInfo nsInfo) throws IOException {
        if (!DataNodeLayoutVersion.supports(LayoutVersion.Feature.FEDERATION, this.layoutVersion)) {
            return;
        }
        LOG.info((Object)("Upgrading block pool storage directory " + bpSd.getRoot() + ".\n   old LV = " + this.getLayoutVersion() + "; old CTime = " + this.getCTime() + ".\n   new LV = " + nsInfo.getLayoutVersion() + "; new CTime = " + nsInfo.getCTime()));
        String dnRoot = BlockPoolSliceStorage.getDataNodeStorageRoot(bpSd.getRoot().getCanonicalPath());
        Storage.StorageDirectory dnSdStorage = new Storage.StorageDirectory(new File(dnRoot));
        File dnPrevDir = dnSdStorage.getPreviousDir();
        if (dnPrevDir.exists()) {
            BlockPoolSliceStorage.deleteDir(dnPrevDir);
        }
        File bpCurDir = bpSd.getCurrentDir();
        File bpPrevDir = bpSd.getPreviousDir();
        assert (bpCurDir.exists()) : "BP level current directory must exist.";
        this.cleanupDetachDir(new File(bpCurDir, "detach"));
        if (bpPrevDir.exists()) {
            BlockPoolSliceStorage.deleteDir(bpPrevDir);
        }
        File bpTmpDir = bpSd.getPreviousTmp();
        assert (!bpTmpDir.exists()) : "previous.tmp directory must not exist.";
        BlockPoolSliceStorage.rename(bpCurDir, bpTmpDir);
        this.linkAllBlocks(bpTmpDir, bpCurDir);
        this.layoutVersion = HdfsConstants.DATANODE_LAYOUT_VERSION;
        assert (this.namespaceID == nsInfo.getNamespaceID()) : "Data-node and name-node layout versions must be the same.";
        this.cTime = nsInfo.getCTime();
        this.writeProperties(bpSd);
        BlockPoolSliceStorage.rename(bpTmpDir, bpPrevDir);
        LOG.info((Object)("Upgrade of block pool " + this.blockpoolID + " at " + bpSd.getRoot() + " is complete"));
    }

    private void cleanupDetachDir(File detachDir) throws IOException {
        if (!DataNodeLayoutVersion.supports(LayoutVersion.Feature.APPEND_RBW_DIR, this.layoutVersion) && detachDir.exists() && detachDir.isDirectory()) {
            if (FileUtil.list(detachDir).length != 0) {
                throw new IOException("Detached directory " + detachDir + " is not empty. Please manually move each file under this " + "directory to the finalized directory if the finalized " + "directory tree does not have the file.");
            }
            if (!detachDir.delete()) {
                throw new IOException("Cannot remove directory " + detachDir);
            }
        }
    }

    private int restoreBlockFilesFromTrash(File trashRoot) throws IOException {
        File[] children;
        int filesRestored = 0;
        File[] fileArray = children = trashRoot.exists() ? trashRoot.listFiles() : null;
        if (children == null) {
            return 0;
        }
        File restoreDirectory = null;
        for (File child : children) {
            if (child.isDirectory()) {
                filesRestored += this.restoreBlockFilesFromTrash(child);
                continue;
            }
            if (restoreDirectory == null && !(restoreDirectory = new File(this.getRestoreDirectory(child))).exists() && !restoreDirectory.mkdirs()) {
                throw new IOException("Failed to create directory " + restoreDirectory);
            }
            File newChild = new File(restoreDirectory, child.getName());
            if (!child.renameTo(newChild)) {
                throw new IOException("Failed to rename " + child + " to " + newChild);
            }
            ++filesRestored;
        }
        FileUtil.fullyDelete(trashRoot);
        return filesRestored;
    }

    void doRollback(Storage.StorageDirectory bpSd, NamespaceInfo nsInfo) throws IOException {
        File prevDir = bpSd.getPreviousDir();
        if (!prevDir.exists()) {
            return;
        }
        BlockPoolSliceStorage prevInfo = new BlockPoolSliceStorage();
        prevInfo.readPreviousVersionProperties(bpSd);
        if (prevInfo.getLayoutVersion() < HdfsConstants.DATANODE_LAYOUT_VERSION || prevInfo.getCTime() > nsInfo.getCTime()) {
            throw new InconsistentFSStateException(bpSd.getRoot(), "Cannot rollback to a newer state.\nDatanode previous state: LV = " + prevInfo.getLayoutVersion() + " CTime = " + prevInfo.getCTime() + " is newer than the namespace state: LV = " + HdfsConstants.DATANODE_LAYOUT_VERSION + " CTime = " + nsInfo.getCTime());
        }
        LOG.info((Object)("Rolling back storage directory " + bpSd.getRoot() + ".\n   target LV = " + nsInfo.getLayoutVersion() + "; target CTime = " + nsInfo.getCTime()));
        File tmpDir = bpSd.getRemovedTmp();
        assert (!tmpDir.exists()) : "removed.tmp directory must not exist.";
        File curDir = bpSd.getCurrentDir();
        assert (curDir.exists()) : "Current directory must exist.";
        BlockPoolSliceStorage.rename(curDir, tmpDir);
        BlockPoolSliceStorage.rename(prevDir, curDir);
        BlockPoolSliceStorage.deleteDir(tmpDir);
        LOG.info((Object)("Rollback of " + bpSd.getRoot() + " is complete"));
    }

    void doFinalize(File dnCurDir) throws IOException {
        File bpRoot = BlockPoolSliceStorage.getBpRoot(this.blockpoolID, dnCurDir);
        Storage.StorageDirectory bpSd = new Storage.StorageDirectory(bpRoot);
        File prevDir = bpSd.getPreviousDir();
        if (!prevDir.exists()) {
            return;
        }
        final String dataDirPath = bpSd.getRoot().getCanonicalPath();
        LOG.info((Object)("Finalizing upgrade for storage directory " + dataDirPath + ".\n   cur LV = " + this.getLayoutVersion() + "; cur CTime = " + this.getCTime()));
        assert (bpSd.getCurrentDir().exists()) : "Current directory must exist.";
        final File tmpDir = bpSd.getFinalizedTmp();
        BlockPoolSliceStorage.rename(prevDir, tmpDir);
        new Daemon(new Runnable(){

            @Override
            public void run() {
                try {
                    Storage.deleteDir(tmpDir);
                }
                catch (IOException ex) {
                    Storage.LOG.error((Object)("Finalize upgrade for " + dataDirPath + " failed."), (Throwable)ex);
                }
                Storage.LOG.info((Object)("Finalize upgrade for " + dataDirPath + " is complete."));
            }

            public String toString() {
                return "Finalize " + dataDirPath;
            }
        }).start();
    }

    private void linkAllBlocks(File fromDir, File toDir) throws IOException {
        int diskLayoutVersion = this.getLayoutVersion();
        HardLink hardLink = new HardLink();
        DataStorage.linkBlocks(new File(fromDir, "finalized"), new File(toDir, "finalized"), diskLayoutVersion, hardLink);
        DataStorage.linkBlocks(new File(fromDir, "rbw"), new File(toDir, "rbw"), diskLayoutVersion, hardLink);
        LOG.info((Object)hardLink.linkStats.report());
    }

    private static String getDataNodeStorageRoot(String bpRoot) {
        Matcher matcher = BLOCK_POOL_PATH_PATTERN.matcher(bpRoot);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return bpRoot;
    }

    @Override
    public String toString() {
        return super.toString() + ";bpid=" + this.blockpoolID;
    }

    public static File getBpRoot(String bpID, File dnCurDir) {
        return new File(dnCurDir, bpID);
    }

    @Override
    public boolean isPreUpgradableLayout(Storage.StorageDirectory sd) throws IOException {
        return false;
    }

    private File getTrashRootDir(Storage.StorageDirectory sd) {
        return new File(sd.getRoot(), TRASH_ROOT_DIR);
    }

    public String getTrashDirectory(File blockFile) {
        Matcher matcher = BLOCK_POOL_CURRENT_PATH_PATTERN.matcher(blockFile.getParent());
        String trashDirectory = matcher.replaceFirst("$1$2trash$4");
        return trashDirectory;
    }

    @VisibleForTesting
    String getRestoreDirectory(File blockFile) {
        Matcher matcher = BLOCK_POOL_TRASH_PATH_PATTERN.matcher(blockFile.getParent());
        String restoreDirectory = matcher.replaceFirst("$1$2current$4");
        LOG.info((Object)("Restoring " + blockFile + " to " + restoreDirectory));
        return restoreDirectory;
    }

    public void restoreTrash() {
        for (Storage.StorageDirectory sd : this.storageDirs) {
            File trashRoot = this.getTrashRootDir(sd);
            try {
                this.restoreBlockFilesFromTrash(trashRoot);
                FileUtil.fullyDelete(this.getTrashRootDir(sd));
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Restoring trash failed for storage directory " + sd));
            }
        }
    }

    @VisibleForTesting
    public boolean trashEnabled() {
        for (Storage.StorageDirectory sd : this.storageDirs) {
            if (!this.getTrashRootDir(sd).exists()) continue;
            return true;
        }
        return false;
    }
}

