/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.NavigableMap;
import java.util.TreeMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.shaded.hadoop2.com.google.common.annotations.VisibleForTesting;
import org.apache.flink.shaded.hadoop2.com.google.common.base.Preconditions;
import org.apache.flink.shaded.hadoop2.com.google.common.net.InetAddresses;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.UnregisteredNodeException;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStatistics;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.DecommissionManager;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.Host2NodesMap;
import org.apache.hadoop.hdfs.server.blockmanagement.UnresolvedTopologyException;
import org.apache.hadoop.hdfs.server.namenode.CachedBlock;
import org.apache.hadoop.hdfs.server.namenode.HostFileManager;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.server.protocol.BalancerBandwidthCommand;
import org.apache.hadoop.hdfs.server.protocol.BlockCommand;
import org.apache.hadoop.hdfs.server.protocol.BlockIdCommand;
import org.apache.hadoop.hdfs.server.protocol.BlockRecoveryCommand;
import org.apache.hadoop.hdfs.server.protocol.DatanodeCommand;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.DisallowedDatanodeException;
import org.apache.hadoop.hdfs.server.protocol.RegisterCommand;
import org.apache.hadoop.hdfs.server.protocol.StorageReport;
import org.apache.hadoop.hdfs.util.CyclicIteration;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.net.CachedDNSToSwitchMapping;
import org.apache.hadoop.net.DNSToSwitchMapping;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.net.NetworkTopology;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.net.NodeBase;
import org.apache.hadoop.net.ScriptBasedMapping;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class DatanodeManager {
    static final Log LOG = LogFactory.getLog(DatanodeManager.class);
    private final Namesystem namesystem;
    private final BlockManager blockManager;
    private final HeartbeatManager heartbeatManager;
    private Daemon decommissionthread = null;
    private final NavigableMap<String, DatanodeDescriptor> datanodeMap = new TreeMap<String, DatanodeDescriptor>();
    private final NetworkTopology networktopology;
    private final Host2NodesMap host2DatanodeMap = new Host2NodesMap();
    private final DNSToSwitchMapping dnsToSwitchMapping;
    private final boolean rejectUnresolvedTopologyDN;
    private final int defaultXferPort;
    private final int defaultInfoPort;
    private final int defaultInfoSecurePort;
    private final int defaultIpcPort;
    private final HostFileManager hostFileManager = new HostFileManager();
    private final long heartbeatExpireInterval;
    final int blockInvalidateLimit;
    private final long staleInterval;
    private final boolean avoidStaleDataNodesForRead;
    private final boolean avoidStaleDataNodesForWrite;
    private final float ratioUseStaleDataNodesForWrite;
    private volatile int numStaleNodes;
    private boolean hasClusterEverBeenMultiRack = false;
    private final boolean checkIpHostnameInRegistration;
    private boolean shouldSendCachingCommands = false;
    private HashMap<String, Integer> datanodesSoftwareVersions = new HashMap(4, 0.75f);
    private final long timeBetweenResendingCachingDirectivesMs;

    DatanodeManager(BlockManager blockManager, Namesystem namesystem, Configuration conf) throws IOException {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.heartbeatManager = new HeartbeatManager(namesystem, blockManager, conf);
        this.networktopology = NetworkTopology.getInstance(conf);
        this.defaultXferPort = NetUtils.createSocketAddr(conf.get("dfs.datanode.address", "0.0.0.0:50010")).getPort();
        this.defaultInfoPort = NetUtils.createSocketAddr(conf.get("dfs.datanode.http.address", "0.0.0.0:50075")).getPort();
        this.defaultInfoSecurePort = NetUtils.createSocketAddr(conf.get("dfs.datanode.https.address", "0.0.0.0:50475")).getPort();
        this.defaultIpcPort = NetUtils.createSocketAddr(conf.get("dfs.datanode.ipc.address", "0.0.0.0:50020")).getPort();
        try {
            this.hostFileManager.refresh(conf.get("dfs.hosts", ""), conf.get("dfs.hosts.exclude", ""));
        }
        catch (IOException e) {
            LOG.error("error reading hosts files: ", e);
        }
        this.dnsToSwitchMapping = ReflectionUtils.newInstance(conf.getClass("net.topology.node.switch.mapping.impl", ScriptBasedMapping.class, DNSToSwitchMapping.class), conf);
        this.rejectUnresolvedTopologyDN = conf.getBoolean("dfs.namenode.reject-unresolved-dn-topology-mapping", false);
        if (this.dnsToSwitchMapping instanceof CachedDNSToSwitchMapping) {
            ArrayList<String> locations = new ArrayList<String>();
            for (HostFileManager.Entry entry : this.hostFileManager.getIncludes()) {
                if (entry.getIpAddress().isEmpty()) continue;
                locations.add(entry.getIpAddress());
            }
            this.dnsToSwitchMapping.resolve(locations);
        }
        long heartbeatIntervalSeconds = conf.getLong("dfs.heartbeat.interval", 3L);
        int heartbeatRecheckInterval = conf.getInt("dfs.namenode.heartbeat.recheck-interval", 300000);
        this.heartbeatExpireInterval = (long)(2 * heartbeatRecheckInterval) + 10000L * heartbeatIntervalSeconds;
        int blockInvalidateLimit = Math.max(20 * (int)heartbeatIntervalSeconds, 1000);
        this.blockInvalidateLimit = conf.getInt("dfs.block.invalidate.limit", blockInvalidateLimit);
        LOG.info("dfs.block.invalidate.limit=" + this.blockInvalidateLimit);
        this.checkIpHostnameInRegistration = conf.getBoolean("dfs.namenode.datanode.registration.ip-hostname-check", true);
        LOG.info("dfs.namenode.datanode.registration.ip-hostname-check=" + this.checkIpHostnameInRegistration);
        this.avoidStaleDataNodesForRead = conf.getBoolean("dfs.namenode.avoid.read.stale.datanode", false);
        this.avoidStaleDataNodesForWrite = conf.getBoolean("dfs.namenode.avoid.write.stale.datanode", false);
        this.staleInterval = DatanodeManager.getStaleIntervalFromConf(conf, this.heartbeatExpireInterval);
        this.ratioUseStaleDataNodesForWrite = conf.getFloat("dfs.namenode.write.stale.datanode.ratio", 0.5f);
        Preconditions.checkArgument(this.ratioUseStaleDataNodesForWrite > 0.0f && this.ratioUseStaleDataNodesForWrite <= 1.0f, "dfs.namenode.write.stale.datanode.ratio = '" + this.ratioUseStaleDataNodesForWrite + "' is invalid. " + "It should be a positive non-zero float value, not greater than 1.0f.");
        this.timeBetweenResendingCachingDirectivesMs = conf.getLong("dfs.namenode.path.based.cache.retry.interval.ms", 30000L);
    }

    private static long getStaleIntervalFromConf(Configuration conf, long heartbeatExpireInterval) {
        long staleInterval = conf.getLong("dfs.namenode.stale.datanode.interval", 30000L);
        Preconditions.checkArgument(staleInterval > 0L, "dfs.namenode.stale.datanode.interval = '" + staleInterval + "' is invalid. " + "It should be a positive non-zero value.");
        long heartbeatIntervalSeconds = conf.getLong("dfs.heartbeat.interval", 3L);
        long minStaleInterval = (long)conf.getInt("dfs.namenode.stale.datanode.minimum.interval", 3) * heartbeatIntervalSeconds * 1000L;
        if (staleInterval < minStaleInterval) {
            LOG.warn("The given interval for marking stale datanode = " + staleInterval + ", which is less than " + 3 + " heartbeat intervals. This may cause too frequent changes of " + "stale states of DataNodes since a heartbeat msg may be missing " + "due to temporary short-term failures. Reset stale interval to " + minStaleInterval + ".");
            staleInterval = minStaleInterval;
        }
        if (staleInterval > heartbeatExpireInterval) {
            LOG.warn("The given interval for marking stale datanode = " + staleInterval + ", which is larger than heartbeat expire interval " + heartbeatExpireInterval + ".");
        }
        return staleInterval;
    }

    void activate(Configuration conf) {
        DecommissionManager dm;
        DecommissionManager decommissionManager = dm = new DecommissionManager(this.namesystem, this.blockManager);
        decommissionManager.getClass();
        this.decommissionthread = new Daemon(decommissionManager.new DecommissionManager.Monitor(conf.getInt("dfs.namenode.decommission.interval", 30), conf.getInt("dfs.namenode.decommission.nodes.per.interval", 5)));
        this.decommissionthread.start();
        this.heartbeatManager.activate(conf);
    }

    void close() {
        if (this.decommissionthread != null) {
            this.decommissionthread.interrupt();
            try {
                this.decommissionthread.join(3000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        this.heartbeatManager.close();
    }

    public NetworkTopology getNetworkTopology() {
        return this.networktopology;
    }

    HeartbeatManager getHeartbeatManager() {
        return this.heartbeatManager;
    }

    public DatanodeStatistics getDatanodeStatistics() {
        return this.heartbeatManager;
    }

    public void sortLocatedBlocks(String targethost, List<LocatedBlock> locatedblocks) {
        Node client = this.getDatanodeByHost(targethost);
        if (client == null) {
            ArrayList<String> hosts = new ArrayList<String>(1);
            hosts.add(targethost);
            String rName = this.dnsToSwitchMapping.resolve(hosts).get(0);
            if (rName != null) {
                client = new NodeBase(rName + "/" + targethost);
            }
        }
        DFSUtil.DecomStaleComparator comparator = this.avoidStaleDataNodesForRead ? new DFSUtil.DecomStaleComparator(this.staleInterval) : DFSUtil.DECOM_COMPARATOR;
        for (LocatedBlock b : locatedblocks) {
            this.networktopology.pseudoSortByDistance(client, b.getLocations());
            Arrays.sort(b.getLocations(), comparator);
        }
    }

    CyclicIteration<String, DatanodeDescriptor> getDatanodeCyclicIteration(String firstkey) {
        return new CyclicIteration<String, DatanodeDescriptor>(this.datanodeMap, firstkey);
    }

    public DatanodeDescriptor getDatanodeByHost(String host) {
        return this.host2DatanodeMap.getDatanodeByHost(host);
    }

    public DatanodeDescriptor getDatanodeByXferAddr(String host, int xferPort) {
        return this.host2DatanodeMap.getDatanodeByXferAddr(host, xferPort);
    }

    DatanodeDescriptor getDatanodeDescriptor(String address) {
        int xferPort;
        DatanodeID dnId = this.parseDNFromHostsEntry(address);
        String host = dnId.getIpAddr();
        DatanodeDescriptor node = this.getDatanodeByXferAddr(host, xferPort = dnId.getXferPort());
        if (node == null) {
            node = this.getDatanodeByHost(host);
        }
        if (node == null) {
            String networkLocation = this.resolveNetworkLocationWithFallBackToDefaultLocation(dnId);
            List<Node> rackNodes = this.getNetworkTopology().getDatanodesInRack(networkLocation);
            if (rackNodes != null) {
                for (Node rackNode : rackNodes) {
                    if (!((DatanodeDescriptor)rackNode).getIpAddr().equals(host)) continue;
                    node = (DatanodeDescriptor)rackNode;
                    break;
                }
                if (node == null && !rackNodes.isEmpty()) {
                    node = (DatanodeDescriptor)rackNodes.get(DFSUtil.getRandom().nextInt(rackNodes.size()));
                }
            }
            if (node == null) {
                node = (DatanodeDescriptor)this.getNetworkTopology().chooseRandom("");
            }
        }
        return node;
    }

    DatanodeDescriptor getDatanode(String datanodeUuid) {
        if (datanodeUuid == null) {
            return null;
        }
        return (DatanodeDescriptor)this.datanodeMap.get(datanodeUuid);
    }

    public DatanodeDescriptor getDatanode(DatanodeID nodeID) throws UnregisteredNodeException {
        DatanodeDescriptor node = this.getDatanode(nodeID.getDatanodeUuid());
        if (node == null) {
            return null;
        }
        if (!node.getXferAddr().equals(nodeID.getXferAddr())) {
            UnregisteredNodeException e = new UnregisteredNodeException(nodeID, node);
            NameNode.stateChangeLog.fatal("BLOCK* NameSystem.getDatanode: " + e.getLocalizedMessage());
            throw e;
        }
        return node;
    }

    public DatanodeStorageInfo[] getDatanodeStorageInfos(DatanodeID[] datanodeID, String[] storageIDs) throws UnregisteredNodeException {
        if (datanodeID.length == 0) {
            return null;
        }
        DatanodeStorageInfo[] storages = new DatanodeStorageInfo[datanodeID.length];
        for (int i = 0; i < datanodeID.length; ++i) {
            DatanodeDescriptor dd = this.getDatanode(datanodeID[i]);
            storages[i] = dd.getStorageInfo(storageIDs[i]);
        }
        return storages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void datanodeDump(PrintWriter out) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            out.println("Metasave: Number of datanodes: " + this.datanodeMap.size());
            for (DatanodeDescriptor node : this.datanodeMap.values()) {
                out.println(node.dumpDatanode());
            }
        }
    }

    private void removeDatanode(DatanodeDescriptor nodeInfo) {
        assert (this.namesystem.hasWriteLock());
        this.heartbeatManager.removeDatanode(nodeInfo);
        this.blockManager.removeBlocksAssociatedTo(nodeInfo);
        this.networktopology.remove(nodeInfo);
        this.decrementVersionCount(nodeInfo.getSoftwareVersion());
        if (LOG.isDebugEnabled()) {
            LOG.debug("remove datanode " + nodeInfo);
        }
        this.namesystem.checkSafeMode();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeDatanode(DatanodeID node) throws UnregisteredNodeException {
        this.namesystem.writeLock();
        try {
            DatanodeDescriptor descriptor = this.getDatanode(node);
            if (descriptor != null) {
                this.removeDatanode(descriptor);
            } else {
                NameNode.stateChangeLog.warn("BLOCK* removeDatanode: " + node + " does not exist");
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeDeadDatanode(DatanodeID nodeID) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            DatanodeDescriptor d;
            try {
                d = this.getDatanode(nodeID);
            }
            catch (IOException e) {
                d = null;
            }
            if (d != null && this.isDatanodeDead(d)) {
                NameNode.stateChangeLog.info("BLOCK* removeDeadDatanode: lost heartbeat from " + d);
                this.removeDatanode(d);
            }
        }
    }

    boolean isDatanodeDead(DatanodeDescriptor node) {
        return node.getLastUpdate() < Time.now() - this.heartbeatExpireInterval;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addDatanode(DatanodeDescriptor node) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            this.host2DatanodeMap.remove(this.datanodeMap.put(node.getDatanodeUuid(), node));
        }
        this.networktopology.add(node);
        this.host2DatanodeMap.add(node);
        this.checkIfClusterIsNowMultiRack(node);
        if (LOG.isDebugEnabled()) {
            LOG.debug(this.getClass().getSimpleName() + ".addDatanode: " + "node " + node + " is added to datanodeMap.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void wipeDatanode(DatanodeID node) {
        String key = node.getDatanodeUuid();
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            this.host2DatanodeMap.remove((DatanodeDescriptor)this.datanodeMap.remove(key));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(this.getClass().getSimpleName() + ".wipeDatanode(" + node + "): storage " + key + " is removed from datanodeMap.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void incrementVersionCount(String version) {
        if (version == null) {
            return;
        }
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            Integer count = this.datanodesSoftwareVersions.get(version);
            count = count == null ? 1 : count + 1;
            this.datanodesSoftwareVersions.put(version, count);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void decrementVersionCount(String version) {
        if (version == null) {
            return;
        }
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            Integer count = this.datanodesSoftwareVersions.get(version);
            if (count != null) {
                if (count > 1) {
                    this.datanodesSoftwareVersions.put(version, count - 1);
                } else {
                    this.datanodesSoftwareVersions.remove(version);
                }
            }
        }
    }

    private boolean shouldCountVersion(DatanodeDescriptor node) {
        return node.getSoftwareVersion() != null && node.isAlive && !this.isDatanodeDead(node);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void countSoftwareVersions() {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            HashMap<String, Integer> versionCount = new HashMap<String, Integer>();
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                if (!this.shouldCountVersion(dn)) continue;
                Integer num = (Integer)versionCount.get(dn.getSoftwareVersion());
                num = num == null ? 1 : num + 1;
                versionCount.put(dn.getSoftwareVersion(), num);
            }
            this.datanodesSoftwareVersions = versionCount;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public HashMap<String, Integer> getDatanodesSoftwareVersions() {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            return new HashMap<String, Integer>(this.datanodesSoftwareVersions);
        }
    }

    private String resolveNetworkLocationWithFallBackToDefaultLocation(DatanodeID node) {
        String networkLocation;
        try {
            networkLocation = this.resolveNetworkLocation(node);
        }
        catch (UnresolvedTopologyException e) {
            LOG.error("Unresolved topology mapping. Using /default-rack for host " + node.getHostName());
            networkLocation = "/default-rack";
        }
        return networkLocation;
    }

    private String resolveNetworkLocation(DatanodeID node) throws UnresolvedTopologyException {
        ArrayList<String> names = new ArrayList<String>(1);
        if (this.dnsToSwitchMapping instanceof CachedDNSToSwitchMapping) {
            names.add(node.getIpAddr());
        } else {
            names.add(node.getHostName());
        }
        List<String> rName = this.dnsToSwitchMapping.resolve(names);
        if (rName == null) {
            LOG.error("The resolve call returned null!");
            throw new UnresolvedTopologyException("Unresolved topology mapping for host " + node.getHostName());
        }
        String networkLocation = rName.get(0);
        return networkLocation;
    }

    private void removeDecomNodeFromList(List<DatanodeDescriptor> nodeList) {
        if (!this.hostFileManager.hasIncludes()) {
            return;
        }
        Iterator<DatanodeDescriptor> it = nodeList.iterator();
        while (it.hasNext()) {
            DatanodeDescriptor node = it.next();
            if (this.hostFileManager.isIncluded(node) || this.hostFileManager.isExcluded(node) || !node.isDecommissioned()) continue;
            it.remove();
        }
    }

    private void checkDecommissioning(DatanodeDescriptor nodeReg) {
        if (this.hostFileManager.isExcluded(nodeReg)) {
            this.startDecommission(nodeReg);
        }
    }

    boolean checkDecommissionState(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() && !this.blockManager.isReplicationInProgress(node)) {
            node.setDecommissioned();
            LOG.info("Decommission complete for " + node);
        }
        return node.isDecommissioned();
    }

    private void startDecommission(DatanodeDescriptor node) {
        if (!node.isDecommissionInProgress() && !node.isDecommissioned()) {
            for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                LOG.info("Start Decommissioning " + node + " " + storage + " with " + storage.numBlocks() + " blocks");
            }
            this.heartbeatManager.startDecommission(node);
            node.decommissioningStatus.setStartTime(Time.now());
            this.checkDecommissionState(node);
        }
    }

    void stopDecommission(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            LOG.info("Stop Decommissioning " + node);
            this.heartbeatManager.stopDecommission(node);
            if (node.isAlive) {
                this.blockManager.processOverReplicatedBlocksOnReCommission(node);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerDatanode(DatanodeRegistration nodeReg) throws DisallowedDatanodeException, UnresolvedTopologyException {
        InetAddress dnAddress = Server.getRemoteIp();
        if (dnAddress != null) {
            String hostname = dnAddress.getHostName();
            String ip = dnAddress.getHostAddress();
            if (this.checkIpHostnameInRegistration && !DatanodeManager.isNameResolved(dnAddress)) {
                String message = "hostname cannot be resolved (ip=" + ip + ", hostname=" + hostname + ")";
                LOG.warn("Unresolved datanode registration: " + message);
                throw new DisallowedDatanodeException(nodeReg, message);
            }
            nodeReg.setIpAddr(ip);
            nodeReg.setPeerHostName(hostname);
        }
        try {
            nodeReg.setExportedKeys(this.blockManager.getBlockKeys());
            if (!this.hostFileManager.isIncluded(nodeReg)) {
                throw new DisallowedDatanodeException(nodeReg);
            }
            NameNode.stateChangeLog.info("BLOCK* registerDatanode: from " + nodeReg + " storage " + nodeReg.getDatanodeUuid());
            DatanodeDescriptor nodeS = this.getDatanode(nodeReg.getDatanodeUuid());
            DatanodeDescriptor nodeN = this.host2DatanodeMap.getDatanodeByXferAddr(nodeReg.getIpAddr(), nodeReg.getXferPort());
            if (nodeN != null && nodeN != nodeS) {
                NameNode.LOG.info("BLOCK* registerDatanode: " + nodeN);
                this.removeDatanode(nodeN);
                this.wipeDatanode(nodeN);
                nodeN = null;
            }
            if (nodeS != null) {
                if (nodeN == nodeS) {
                    if (NameNode.stateChangeLog.isDebugEnabled()) {
                        NameNode.stateChangeLog.debug("BLOCK* registerDatanode: node restarted.");
                    }
                } else {
                    NameNode.stateChangeLog.info("BLOCK* registerDatanode: " + nodeS + " is replaced by " + nodeReg + " with the same storageID " + nodeReg.getDatanodeUuid());
                }
                boolean success = false;
                try {
                    this.getNetworkTopology().remove(nodeS);
                    if (this.shouldCountVersion(nodeS)) {
                        this.decrementVersionCount(nodeS.getSoftwareVersion());
                    }
                    nodeS.updateRegInfo(nodeReg);
                    nodeS.setSoftwareVersion(nodeReg.getSoftwareVersion());
                    nodeS.setDisallowed(false);
                    if (this.rejectUnresolvedTopologyDN) {
                        nodeS.setNetworkLocation(this.resolveNetworkLocation(nodeS));
                    } else {
                        nodeS.setNetworkLocation(this.resolveNetworkLocationWithFallBackToDefaultLocation(nodeS));
                    }
                    this.getNetworkTopology().add(nodeS);
                    this.heartbeatManager.register(nodeS);
                    this.incrementVersionCount(nodeS.getSoftwareVersion());
                    this.checkDecommissioning(nodeS);
                    success = true;
                }
                finally {
                    if (!success) {
                        this.removeDatanode(nodeS);
                        this.wipeDatanode(nodeS);
                        this.countSoftwareVersions();
                    }
                }
                return;
            }
            DatanodeDescriptor nodeDescr = new DatanodeDescriptor(nodeReg, "/default-rack");
            boolean success = false;
            try {
                if (this.rejectUnresolvedTopologyDN) {
                    nodeDescr.setNetworkLocation(this.resolveNetworkLocation(nodeDescr));
                } else {
                    nodeDescr.setNetworkLocation(this.resolveNetworkLocationWithFallBackToDefaultLocation(nodeDescr));
                }
                this.networktopology.add(nodeDescr);
                nodeDescr.setSoftwareVersion(nodeReg.getSoftwareVersion());
                this.addDatanode(nodeDescr);
                this.checkDecommissioning(nodeDescr);
                this.heartbeatManager.addDatanode(nodeDescr);
                success = true;
                this.incrementVersionCount(nodeReg.getSoftwareVersion());
            }
            finally {
                if (!success) {
                    this.removeDatanode(nodeDescr);
                    this.wipeDatanode(nodeDescr);
                    this.countSoftwareVersions();
                }
            }
        }
        catch (NetworkTopology.InvalidTopologyException e) {
            ArrayList<String> invalidNodeNames = new ArrayList<String>(3);
            invalidNodeNames.add(nodeReg.getIpAddr());
            invalidNodeNames.add(nodeReg.getHostName());
            invalidNodeNames.add(nodeReg.getPeerHostName());
            this.dnsToSwitchMapping.reloadCachedMappings(invalidNodeNames);
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refreshNodes(Configuration conf) throws IOException {
        this.refreshHostsReader(conf);
        this.namesystem.writeLock();
        try {
            this.refreshDatanodes();
            this.countSoftwareVersions();
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    private void refreshHostsReader(Configuration conf) throws IOException {
        if (conf == null) {
            conf = new HdfsConfiguration();
        }
        this.hostFileManager.refresh(conf.get("dfs.hosts", ""), conf.get("dfs.hosts.exclude", ""));
    }

    private void refreshDatanodes() {
        for (DatanodeDescriptor node : this.datanodeMap.values()) {
            if (!this.hostFileManager.isIncluded(node)) {
                node.setDisallowed(true);
                continue;
            }
            if (this.hostFileManager.isExcluded(node)) {
                this.startDecommission(node);
                continue;
            }
            this.stopDecommission(node);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumLiveDataNodes() {
        int numLive = 0;
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                if (this.isDatanodeDead(dn)) continue;
                ++numLive;
            }
        }
        return numLive;
    }

    public int getNumDeadDataNodes() {
        return this.getDatanodeListForReport(HdfsConstants.DatanodeReportType.DEAD).size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DatanodeDescriptor> getDecommissioningNodes() {
        this.namesystem.readLock();
        try {
            ArrayList<DatanodeDescriptor> decommissioningNodes = new ArrayList<DatanodeDescriptor>();
            List<DatanodeDescriptor> results = this.getDatanodeListForReport(HdfsConstants.DatanodeReportType.LIVE);
            for (DatanodeDescriptor node : results) {
                if (!node.isDecommissionInProgress()) continue;
                decommissioningNodes.add(node);
            }
            ArrayList<DatanodeDescriptor> arrayList = decommissioningNodes;
            return arrayList;
        }
        finally {
            this.namesystem.readUnlock();
        }
    }

    public boolean shouldAvoidStaleDataNodesForWrite() {
        return this.avoidStaleDataNodesForWrite && (float)this.numStaleNodes <= (float)this.heartbeatManager.getLiveDatanodeCount() * this.ratioUseStaleDataNodesForWrite;
    }

    long getStaleInterval() {
        return this.staleInterval;
    }

    void setNumStaleNodes(int numStaleNodes) {
        this.numStaleNodes = numStaleNodes;
    }

    public int getNumStaleNodes() {
        return this.numStaleNodes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void fetchDatanodes(List<DatanodeDescriptor> live, List<DatanodeDescriptor> dead, boolean removeDecommissionNode) {
        if (live == null && dead == null) {
            throw new HadoopIllegalArgumentException("Both live and dead lists are null");
        }
        this.namesystem.readLock();
        try {
            List<DatanodeDescriptor> results = this.getDatanodeListForReport(HdfsConstants.DatanodeReportType.ALL);
            for (DatanodeDescriptor node : results) {
                if (this.isDatanodeDead(node)) {
                    if (dead == null) continue;
                    dead.add(node);
                    continue;
                }
                if (live == null) continue;
                live.add(node);
            }
        }
        finally {
            this.namesystem.readUnlock();
        }
        if (removeDecommissionNode) {
            if (live != null) {
                this.removeDecomNodeFromList(live);
            }
            if (dead != null) {
                this.removeDecomNodeFromList(dead);
            }
        }
    }

    boolean hasClusterEverBeenMultiRack() {
        return this.hasClusterEverBeenMultiRack;
    }

    @VisibleForTesting
    void checkIfClusterIsNowMultiRack(DatanodeDescriptor node) {
        if (!this.hasClusterEverBeenMultiRack && this.networktopology.getNumOfRacks() > 1) {
            String message = "DN " + node + " joining cluster has expanded a formerly " + "single-rack cluster to be multi-rack. ";
            if (this.namesystem.isPopulatingReplQueues()) {
                message = message + "Re-checking all blocks for replication, since they should now be replicated cross-rack";
                LOG.info(message);
            } else {
                message = message + "Not checking for mis-replicated blocks because this NN is not yet processing repl queues.";
                LOG.debug(message);
            }
            this.hasClusterEverBeenMultiRack = true;
            if (this.namesystem.isPopulatingReplQueues()) {
                this.blockManager.processMisReplicatedBlocks();
            }
        }
    }

    private DatanodeID parseDNFromHostsEntry(String hostLine) {
        DatanodeID dnId;
        int port;
        String hostStr;
        int idx = hostLine.indexOf(58);
        if (-1 == idx) {
            hostStr = hostLine;
            port = 50010;
        } else {
            hostStr = hostLine.substring(0, idx);
            port = Integer.valueOf(hostLine.substring(idx + 1));
        }
        if (InetAddresses.isInetAddress(hostStr)) {
            dnId = new DatanodeID(hostStr, "", "", port, 50075, 50475, 50020);
        } else {
            String ipAddr = "";
            try {
                ipAddr = InetAddress.getByName(hostStr).getHostAddress();
            }
            catch (UnknownHostException e) {
                LOG.warn("Invalid hostname " + hostStr + " in hosts file");
            }
            dnId = new DatanodeID(ipAddr, hostStr, "", port, 50075, 50475, 50020);
        }
        return dnId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DatanodeDescriptor> getDatanodeListForReport(HdfsConstants.DatanodeReportType type) {
        boolean listLiveNodes = type == HdfsConstants.DatanodeReportType.ALL || type == HdfsConstants.DatanodeReportType.LIVE;
        boolean listDeadNodes = type == HdfsConstants.DatanodeReportType.ALL || type == HdfsConstants.DatanodeReportType.DEAD;
        ArrayList<DatanodeDescriptor> nodes = null;
        HostFileManager.MutableEntrySet foundNodes = new HostFileManager.MutableEntrySet();
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            nodes = new ArrayList<DatanodeDescriptor>(this.datanodeMap.size());
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                boolean isDead = this.isDatanodeDead(dn);
                if (isDead && listDeadNodes || !isDead && listLiveNodes) {
                    nodes.add(dn);
                }
                foundNodes.add(dn);
            }
        }
        if (listDeadNodes) {
            HostFileManager.EntrySet includedNodes = this.hostFileManager.getIncludes();
            HostFileManager.EntrySet excludedNodes = this.hostFileManager.getExcludes();
            for (HostFileManager.Entry entry : includedNodes) {
                if (foundNodes.find(entry) != null || excludedNodes.find(entry) != null) continue;
                DatanodeDescriptor dn = new DatanodeDescriptor(new DatanodeID(entry.getIpAddress(), entry.getPrefix(), "", entry.getPort() == 0 ? this.defaultXferPort : entry.getPort(), this.defaultInfoPort, this.defaultInfoSecurePort, this.defaultIpcPort));
                dn.setLastUpdate(0L);
                nodes.add(dn);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("getDatanodeListForReport with includedNodes = " + this.hostFileManager.getIncludes() + ", excludedNodes = " + this.hostFileManager.getExcludes() + ", foundNodes = " + foundNodes + ", nodes = " + nodes);
        }
        return nodes;
    }

    private static boolean isNameResolved(InetAddress address) {
        String ip;
        String hostname = address.getHostName();
        return !hostname.equals(ip = address.getHostAddress()) || NetUtils.isLocalAddress(address);
    }

    private void setDatanodeDead(DatanodeDescriptor node) {
        node.setLastUpdate(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DatanodeCommand[] handleHeartbeat(DatanodeRegistration nodeReg, StorageReport[] reports, String blockPoolId, long cacheCapacity, long cacheUsed, int xceiverCount, int maxTransfers, int failedVolumes) throws IOException {
        HeartbeatManager heartbeatManager = this.heartbeatManager;
        synchronized (heartbeatManager) {
            NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
            synchronized (navigableMap) {
                Block[] blks;
                DatanodeDescriptor nodeinfo = null;
                try {
                    nodeinfo = this.getDatanode(nodeReg);
                }
                catch (UnregisteredNodeException e) {
                    return new DatanodeCommand[]{RegisterCommand.REGISTER};
                }
                if (nodeinfo != null && nodeinfo.isDisallowed()) {
                    this.setDatanodeDead(nodeinfo);
                    throw new DisallowedDatanodeException(nodeinfo);
                }
                if (nodeinfo == null || !nodeinfo.isAlive) {
                    return new DatanodeCommand[]{RegisterCommand.REGISTER};
                }
                this.heartbeatManager.updateHeartbeat(nodeinfo, reports, cacheCapacity, cacheUsed, xceiverCount, failedVolumes);
                if (this.namesystem.isInSafeMode()) {
                    return new DatanodeCommand[0];
                }
                BlockInfoUnderConstruction[] blocks = nodeinfo.getLeaseRecoveryCommand(Integer.MAX_VALUE);
                if (blocks != null) {
                    BlockRecoveryCommand brCommand = new BlockRecoveryCommand(blocks.length);
                    for (BlockInfoUnderConstruction b : blocks) {
                        DatanodeStorageInfo[] storages = b.getExpectedStorageLocations();
                        ArrayList<DatanodeStorageInfo> recoveryLocations = new ArrayList<DatanodeStorageInfo>(storages.length);
                        for (int i = 0; i < storages.length; ++i) {
                            if (storages[i].getDatanodeDescriptor().isStale(this.staleInterval)) continue;
                            recoveryLocations.add(storages[i]);
                        }
                        if (recoveryLocations.size() > 1) {
                            if (recoveryLocations.size() != storages.length) {
                                LOG.info("Skipped stale nodes for recovery : " + (storages.length - recoveryLocations.size()));
                            }
                            brCommand.add(new BlockRecoveryCommand.RecoveringBlock(new ExtendedBlock(blockPoolId, b), DatanodeStorageInfo.toDatanodeInfos(recoveryLocations), b.getBlockRecoveryId()));
                            continue;
                        }
                        brCommand.add(new BlockRecoveryCommand.RecoveringBlock(new ExtendedBlock(blockPoolId, b), DatanodeStorageInfo.toDatanodeInfos(storages), b.getBlockRecoveryId()));
                    }
                    return new DatanodeCommand[]{brCommand};
                }
                ArrayList<DatanodeCommand> cmds = new ArrayList<DatanodeCommand>();
                List<DatanodeDescriptor.BlockTargetPair> pendingList = nodeinfo.getReplicationCommand(maxTransfers);
                if (pendingList != null) {
                    cmds.add(new BlockCommand(1, blockPoolId, pendingList));
                }
                if ((blks = nodeinfo.getInvalidateBlocks(this.blockInvalidateLimit)) != null) {
                    cmds.add(new BlockCommand(2, blockPoolId, blks));
                }
                boolean sendingCachingCommands = false;
                long nowMs = Time.monotonicNow();
                if (this.shouldSendCachingCommands && nowMs - nodeinfo.getLastCachingDirectiveSentTimeMs() >= this.timeBetweenResendingCachingDirectivesMs) {
                    DatanodeCommand pendingUncacheCommand;
                    DatanodeCommand pendingCacheCommand = this.getCacheCommand(nodeinfo.getPendingCached(), nodeinfo, 9, blockPoolId);
                    if (pendingCacheCommand != null) {
                        cmds.add(pendingCacheCommand);
                        sendingCachingCommands = true;
                    }
                    if ((pendingUncacheCommand = this.getCacheCommand(nodeinfo.getPendingUncached(), nodeinfo, 10, blockPoolId)) != null) {
                        cmds.add(pendingUncacheCommand);
                        sendingCachingCommands = true;
                    }
                    if (sendingCachingCommands) {
                        nodeinfo.setLastCachingDirectiveSentTimeMs(nowMs);
                    }
                }
                this.blockManager.addKeyUpdateCommand(cmds, nodeinfo);
                if (nodeinfo.getBalancerBandwidth() > 0L) {
                    cmds.add(new BalancerBandwidthCommand(nodeinfo.getBalancerBandwidth()));
                    nodeinfo.setBalancerBandwidth(0L);
                }
                if (!cmds.isEmpty()) {
                    return cmds.toArray(new DatanodeCommand[cmds.size()]);
                }
            }
        }
        return new DatanodeCommand[0];
    }

    private DatanodeCommand getCacheCommand(DatanodeDescriptor.CachedBlocksList list, DatanodeDescriptor datanode, int action, String poolId) {
        int length = list.size();
        if (length == 0) {
            return null;
        }
        long[] blockIds = new long[length];
        int i = 0;
        for (CachedBlock cachedBlock : list) {
            blockIds[i++] = cachedBlock.getBlockId();
        }
        return new BlockIdCommand(action, poolId, blockIds);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setBalancerBandwidth(long bandwidth) throws IOException {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor nodeInfo : this.datanodeMap.values()) {
                nodeInfo.setBalancerBandwidth(bandwidth);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void markAllDatanodesStale() {
        LOG.info("Marking all datandoes as stale");
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                for (DatanodeStorageInfo storage : dn.getStorageInfos()) {
                    storage.markStaleAfterFailover();
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearPendingQueues() {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                dn.clearBlockQueues();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void resetLastCachingDirectiveSentTime() {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                dn.setLastCachingDirectiveSentTimeMs(0L);
            }
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + ": " + this.host2DatanodeMap;
    }

    public void clearPendingCachingCommands() {
        for (DatanodeDescriptor dn : this.datanodeMap.values()) {
            dn.getPendingCached().clear();
            dn.getPendingUncached().clear();
        }
    }

    public void setShouldSendCachingCommands(boolean shouldSendCachingCommands) {
        this.shouldSendCachingCommands = shouldSendCachingCommands;
    }
}

