/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Collection;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.event.AbstractEvent;
import org.apache.flink.runtime.io.network.api.EndOfPartitionEvent;
import org.apache.flink.runtime.io.network.api.serialization.EventSerializer;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferCompressor;
import org.apache.flink.runtime.io.network.buffer.BufferDecompressor;
import org.apache.flink.runtime.io.network.buffer.FreeingBufferRecycler;
import org.apache.flink.runtime.io.network.buffer.NetworkBuffer;
import org.apache.flink.testutils.junit.extensions.parameterized.ParameterizedTestExtension;
import org.apache.flink.testutils.junit.extensions.parameterized.Parameters;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={ParameterizedTestExtension.class})
class BufferCompressionTest {
    private static final int BUFFER_SIZE = 0x400000;
    private static final int NUM_LONGS = 524288;
    private final boolean compressToOriginalBuffer;
    private final boolean decompressToOriginalBuffer;
    private final BufferCompressor compressor;
    private final BufferDecompressor decompressor;
    private final Buffer bufferToCompress;

    @Parameters(name="isDirect = {0}, codec = {1}, compressToOriginal = {2}, decompressToOriginal = {3}")
    public static Collection<Object[]> parameters() {
        return Arrays.asList({true, "LZ4", true, false}, {true, "LZ4", false, true}, {true, "LZ4", false, false}, {false, "LZ4", true, false}, {false, "LZ4", false, true}, {false, "LZ4", false, false}, {true, "ZSTD", true, false}, {true, "ZSTD", false, true}, {true, "ZSTD", false, false}, {false, "ZSTD", true, false}, {false, "ZSTD", false, true}, {false, "ZSTD", false, false}, {true, "LZO", true, false}, {true, "LZO", false, true}, {true, "LZO", false, false}, {false, "LZO", true, false}, {false, "LZO", false, true}, {false, "LZO", false, false});
    }

    public BufferCompressionTest(boolean isDirect, String compressionCodec, boolean compressToOriginalBuffer, boolean decompressToOriginalBuffer) {
        this.compressToOriginalBuffer = compressToOriginalBuffer;
        this.decompressToOriginalBuffer = decompressToOriginalBuffer;
        this.compressor = new BufferCompressor(0x400000, compressionCodec);
        this.decompressor = new BufferDecompressor(0x400000, compressionCodec);
        this.bufferToCompress = BufferCompressionTest.createBufferAndFillWithLongValues(isDirect);
    }

    @TestTemplate
    void testCompressAndDecompressNetWorkBuffer() {
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer);
        Assertions.assertThat((boolean)compressedBuffer.isCompressed()).isTrue();
        Buffer decompressedBuffer = BufferCompressionTest.decompress(this.decompressor, compressedBuffer, this.decompressToOriginalBuffer);
        Assertions.assertThat((boolean)decompressedBuffer.isCompressed()).isFalse();
        BufferCompressionTest.verifyDecompressionResult(decompressedBuffer, 0L, 524288);
    }

    @TestTemplate
    void testCompressAndDecompressReadOnlySlicedNetworkBuffer() {
        int offset = 0x100000;
        int length = 0x200000;
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(offset, length);
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, readOnlySlicedBuffer, this.compressToOriginalBuffer);
        Assertions.assertThat((boolean)compressedBuffer.isCompressed()).isTrue();
        Buffer decompressedBuffer = BufferCompressionTest.decompress(this.decompressor, compressedBuffer, this.decompressToOriginalBuffer);
        Assertions.assertThat((boolean)decompressedBuffer.isCompressed()).isFalse();
        BufferCompressionTest.verifyDecompressionResult(decompressedBuffer, 131072L, 262144);
    }

    @TestTemplate
    void testCompressEmptyBuffer() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.compress(this.compressor, this.bufferToCompress.readOnlySlice(0, 0), this.compressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDecompressEmptyBuffer() {
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(0, 0);
        readOnlySlicedBuffer.setCompressed(true);
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.decompress(this.decompressor, readOnlySlicedBuffer, this.decompressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testCompressBufferWithNonZeroReadOffset() {
        this.bufferToCompress.setReaderIndex(1);
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDecompressBufferWithNonZeroReadOffset() {
        this.bufferToCompress.setReaderIndex(1);
        this.bufferToCompress.setCompressed(true);
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.decompress(this.decompressor, this.bufferToCompress, this.decompressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testCompressNull() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.compress(this.compressor, null, this.compressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDecompressNull() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.decompress(this.decompressor, null, this.decompressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testCompressCompressedBuffer() {
        this.bufferToCompress.setCompressed(true);
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDecompressUncompressedBuffer() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.decompress(this.decompressor, this.bufferToCompress, this.decompressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testCompressEvent() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.compress(this.compressor, EventSerializer.toBuffer((AbstractEvent)EndOfPartitionEvent.INSTANCE, (boolean)false), this.compressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDecompressEvent() {
        Assertions.assertThatThrownBy(() -> BufferCompressionTest.decompress(this.decompressor, EventSerializer.toBuffer((AbstractEvent)EndOfPartitionEvent.INSTANCE, (boolean)false), this.decompressToOriginalBuffer)).isInstanceOf(IllegalArgumentException.class);
    }

    @TestTemplate
    void testDataSizeGrowsAfterCompression() {
        int numBytes = 1;
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(0x200000, numBytes);
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, readOnlySlicedBuffer, this.compressToOriginalBuffer);
        Assertions.assertThat((boolean)compressedBuffer.isCompressed()).isFalse();
        Assertions.assertThat((Object)compressedBuffer).isEqualTo((Object)readOnlySlicedBuffer);
        Assertions.assertThat((int)compressedBuffer.readableBytes()).isEqualTo(numBytes);
    }

    private static Buffer createBufferAndFillWithLongValues(boolean isDirect) {
        MemorySegment segment = isDirect ? MemorySegmentFactory.allocateUnpooledSegment((int)0x400000) : MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)0x400000);
        for (int i = 0; i < 524288; ++i) {
            segment.putLongLittleEndian(8 * i, (long)i);
        }
        NetworkBuffer buffer = new NetworkBuffer(segment, FreeingBufferRecycler.INSTANCE);
        buffer.setSize(0x400000);
        return buffer;
    }

    private static void verifyDecompressionResult(Buffer buffer, long start, int numLongs) {
        ByteBuffer byteBuffer = buffer.getNioBufferReadable().order(ByteOrder.LITTLE_ENDIAN);
        for (int i = 0; i < numLongs; ++i) {
            Assertions.assertThat((long)byteBuffer.getLong()).isEqualTo(start + (long)i);
        }
    }

    private static Buffer compress(BufferCompressor compressor, Buffer buffer, boolean compressToOriginalBuffer) {
        if (compressToOriginalBuffer) {
            return compressor.compressToOriginalBuffer(buffer);
        }
        return compressor.compressToIntermediateBuffer(buffer);
    }

    private static Buffer decompress(BufferDecompressor decompressor, Buffer buffer, boolean decompressToOriginalBuffer) {
        if (decompressToOriginalBuffer) {
            return decompressor.decompressToOriginalBuffer(buffer);
        }
        return decompressor.decompressToIntermediateBuffer(buffer);
    }
}

