/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.net;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketAddress;
import java.util.Enumeration;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.flink.runtime.akka.AkkaUtils;
import org.apache.flink.runtime.leaderretrieval.LeaderRetrievalException;
import org.apache.flink.runtime.leaderretrieval.LeaderRetrievalListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.concurrent.duration.FiniteDuration;

public class ConnectionUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ConnectionUtils.class);
    private static final long MIN_SLEEP_TIME = 50L;
    private static final long MAX_SLEEP_TIME = 20000L;

    public static InetAddress findConnectingAddress(InetSocketAddress targetAddress, long maxWaitMillis, long startLoggingAfter) throws IOException {
        if (targetAddress == null) {
            throw new NullPointerException("targetAddress must not be null");
        }
        if (maxWaitMillis <= 0L) {
            throw new IllegalArgumentException("Max wait time must be positive");
        }
        long startTime = System.currentTimeMillis();
        long currentSleepTime = 50L;
        long elapsedTime = 0L;
        while (elapsedTime < maxWaitMillis) {
            boolean logging;
            AddressDetectionState strategy = AddressDetectionState.LOCAL_HOST;
            boolean bl = logging = elapsedTime >= startLoggingAfter;
            if (logging) {
                LOG.info("Trying to connect to " + targetAddress);
            }
            do {
                InetAddress address;
                if ((address = ConnectionUtils.findAddressUsingStrategy(strategy, targetAddress, logging)) != null) {
                    return address;
                }
                switch (strategy) {
                    case LOCAL_HOST: {
                        strategy = AddressDetectionState.ADDRESS;
                        break;
                    }
                    case ADDRESS: {
                        strategy = AddressDetectionState.FAST_CONNECT;
                        break;
                    }
                    case FAST_CONNECT: {
                        strategy = AddressDetectionState.SLOW_CONNECT;
                        break;
                    }
                    case SLOW_CONNECT: {
                        strategy = null;
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
                    }
                }
            } while (strategy != null);
            elapsedTime = System.currentTimeMillis() - startTime;
            long toWait = Math.min(maxWaitMillis - elapsedTime, currentSleepTime);
            if (toWait > 0L) {
                if (logging) {
                    LOG.info("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                } else {
                    LOG.debug("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                }
                try {
                    Thread.sleep(toWait);
                }
                catch (InterruptedException e) {
                    throw new IOException("Connection attempts have been interrupted.");
                }
            }
            currentSleepTime = Math.min(2L * currentSleepTime, 20000L);
        }
        LOG.warn("Could not connect to {}. Selecting a local address using heuristics.", (Object)targetAddress);
        InetAddress heuristic = ConnectionUtils.findAddressUsingStrategy(AddressDetectionState.HEURISTIC, targetAddress, true);
        if (heuristic != null) {
            return heuristic;
        }
        LOG.warn("Could not find any IPv4 address that is not loopback or link-local. Using localhost address.");
        return InetAddress.getLocalHost();
    }

    private static InetAddress tryLocalHostBeforeReturning(InetAddress preliminaryResult, SocketAddress targetAddress, boolean logging) throws IOException {
        InetAddress localhostName = InetAddress.getLocalHost();
        if (preliminaryResult.equals(localhostName)) {
            return preliminaryResult;
        }
        if (ConnectionUtils.tryToConnect(localhostName, targetAddress, AddressDetectionState.SLOW_CONNECT.getTimeout(), logging)) {
            LOG.debug("Preferring {} (InetAddress.getLocalHost()) for local bind point over previous candidate {}", (Object)localhostName, (Object)preliminaryResult);
            return localhostName;
        }
        return preliminaryResult;
    }

    private static InetAddress findAddressUsingStrategy(AddressDetectionState strategy, InetSocketAddress targetAddress, boolean logging) throws IOException {
        if (strategy == AddressDetectionState.LOCAL_HOST) {
            InetAddress localhostName = InetAddress.getLocalHost();
            if (ConnectionUtils.tryToConnect(localhostName, targetAddress, strategy.getTimeout(), logging)) {
                LOG.debug("Using InetAddress.getLocalHost() immediately for the connecting address");
                return localhostName;
            }
            return null;
        }
        byte[] targetAddressBytes = targetAddress.getAddress().getAddress();
        Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
        while (e.hasMoreElements()) {
            NetworkInterface netInterface = e.nextElement();
            Enumeration<InetAddress> ee = netInterface.getInetAddresses();
            block6: while (ee.hasMoreElements()) {
                InetAddress interfaceAddress = ee.nextElement();
                switch (strategy) {
                    case ADDRESS: {
                        if (!ConnectionUtils.hasCommonPrefix(targetAddressBytes, interfaceAddress.getAddress())) continue block6;
                        LOG.debug("Target address {} and local address {} share prefix - trying to connect.", (Object)targetAddress, (Object)interfaceAddress);
                        if (!ConnectionUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block6;
                        return ConnectionUtils.tryLocalHostBeforeReturning(interfaceAddress, targetAddress, logging);
                    }
                    case FAST_CONNECT: 
                    case SLOW_CONNECT: {
                        LOG.debug("Trying to connect to {} from local address {} with timeout {}", targetAddress, interfaceAddress, strategy.getTimeout());
                        if (!ConnectionUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block6;
                        return ConnectionUtils.tryLocalHostBeforeReturning(interfaceAddress, targetAddress, logging);
                    }
                    case HEURISTIC: {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Checking address {} using heuristics: linkLocal: {} loopback: {}", interfaceAddress, interfaceAddress.isLinkLocalAddress(), interfaceAddress.isLoopbackAddress());
                        }
                        if (!(interfaceAddress instanceof Inet4Address) || interfaceAddress.isLinkLocalAddress() || interfaceAddress.isLoopbackAddress()) continue block6;
                        return ConnectionUtils.tryLocalHostBeforeReturning(interfaceAddress, targetAddress, logging);
                    }
                }
                throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
            }
        }
        return null;
    }

    private static boolean hasCommonPrefix(byte[] address, byte[] address2) {
        return address[0] == address2[0] && address[1] == address2[1];
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean tryToConnect(InetAddress fromAddress, SocketAddress toSocket, int timeout, boolean logFailed) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Trying to connect to (" + toSocket + ") from local address " + fromAddress + " with timeout " + timeout);
        }
        try (Socket socket = new Socket();){
            InetSocketAddress bindP = new InetSocketAddress(fromAddress, 0);
            socket.bind(bindP);
            socket.connect(toSocket, timeout);
            boolean bl = true;
            return bl;
        }
        catch (Exception ex) {
            String message = "Failed to connect from address '" + fromAddress + "': " + ex.getMessage();
            if (LOG.isDebugEnabled()) {
                LOG.debug(message, ex);
                return false;
            }
            if (!logFailed) return false;
            LOG.info(message);
            return false;
        }
    }

    public static class LeaderConnectingAddressListener
    implements LeaderRetrievalListener {
        private static final FiniteDuration defaultLoggingDelay = new FiniteDuration(400L, TimeUnit.MILLISECONDS);
        private final Object retrievalLock = new Object();
        private String akkaURL;
        private LeaderRetrievalState retrievalState = LeaderRetrievalState.NOT_RETRIEVED;
        private Exception exception;

        public InetAddress findConnectingAddress(FiniteDuration timeout) throws LeaderRetrievalException {
            return this.findConnectingAddress(timeout, defaultLoggingDelay);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public InetAddress findConnectingAddress(FiniteDuration timeout, FiniteDuration startLoggingAfter) throws LeaderRetrievalException {
            long startTime = System.currentTimeMillis();
            long currentSleepTime = 50L;
            long elapsedTime = 0L;
            InetSocketAddress targetAddress = null;
            try {
                while (elapsedTime < timeout.toMillis()) {
                    long maxTimeout = timeout.toMillis() - elapsedTime;
                    Object object = this.retrievalLock;
                    synchronized (object) {
                        if (this.exception != null) {
                            throw this.exception;
                        }
                        if (this.retrievalState == LeaderRetrievalState.NOT_RETRIEVED) {
                            try {
                                this.retrievalLock.wait(maxTimeout);
                            }
                            catch (InterruptedException e) {
                                throw new Exception("Finding connecting address was interruptedwhile waiting for the leader retrieval.");
                            }
                        } else if (this.retrievalState == LeaderRetrievalState.NEWLY_RETRIEVED) {
                            targetAddress = AkkaUtils.getInetSockeAddressFromAkkaURL(this.akkaURL);
                            LOG.info("Retrieved new target address {}.", (Object)targetAddress);
                            this.retrievalState = LeaderRetrievalState.RETRIEVED;
                            currentSleepTime = 50L;
                        } else {
                            currentSleepTime = Math.min(2L * currentSleepTime, 20000L);
                        }
                    }
                    if (targetAddress != null) {
                        boolean logging;
                        AddressDetectionState strategy = AddressDetectionState.LOCAL_HOST;
                        boolean bl = logging = elapsedTime >= startLoggingAfter.toMillis();
                        if (logging) {
                            LOG.info("Trying to connect to address {}", (Object)targetAddress);
                        }
                        do {
                            InetAddress address;
                            if ((address = ConnectionUtils.findAddressUsingStrategy(strategy, targetAddress, logging)) != null) {
                                return address;
                            }
                            switch (strategy) {
                                case LOCAL_HOST: {
                                    strategy = AddressDetectionState.ADDRESS;
                                    break;
                                }
                                case ADDRESS: {
                                    strategy = AddressDetectionState.FAST_CONNECT;
                                    break;
                                }
                                case FAST_CONNECT: {
                                    strategy = AddressDetectionState.SLOW_CONNECT;
                                    break;
                                }
                                case SLOW_CONNECT: {
                                    strategy = null;
                                    break;
                                }
                                default: {
                                    throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
                                }
                            }
                        } while (strategy != null);
                    }
                    elapsedTime = System.currentTimeMillis() - startTime;
                    long timeToWait = Math.min(Math.max(timeout.toMillis() - elapsedTime, 0L), currentSleepTime);
                    if (timeToWait <= 0L) continue;
                    Object object2 = this.retrievalLock;
                    synchronized (object2) {
                        try {
                            this.retrievalLock.wait(timeToWait);
                        }
                        catch (InterruptedException e) {
                            throw new Exception("Finding connecting address was interrupted while pausing.");
                        }
                    }
                    elapsedTime = System.currentTimeMillis() - startTime;
                }
                InetAddress heuristic = null;
                if (targetAddress != null) {
                    LOG.warn("Could not connect to {}. Selecting a local address using heuristics.", (Object)targetAddress);
                    heuristic = ConnectionUtils.findAddressUsingStrategy(AddressDetectionState.HEURISTIC, targetAddress, true);
                }
                if (heuristic != null) {
                    return heuristic;
                }
                LOG.warn("Could not find any IPv4 address that is not loopback or link-local. Using localhost address.");
                return InetAddress.getLocalHost();
            }
            catch (Exception e) {
                throw new LeaderRetrievalException("Could not retrieve the connecting address to the current leader with the akka URL " + this.akkaURL + ".", e);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void notifyLeaderAddress(String leaderAddress, UUID leaderSessionID) {
            if (leaderAddress != null && !leaderAddress.equals("")) {
                Object object = this.retrievalLock;
                synchronized (object) {
                    this.akkaURL = leaderAddress;
                    this.retrievalState = LeaderRetrievalState.NEWLY_RETRIEVED;
                    this.retrievalLock.notifyAll();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleError(Exception exception) {
            Object object = this.retrievalLock;
            synchronized (object) {
                this.exception = exception;
                this.retrievalLock.notifyAll();
            }
        }

        private static enum LeaderRetrievalState {
            NOT_RETRIEVED,
            RETRIEVED,
            NEWLY_RETRIEVED;

        }
    }

    private static enum AddressDetectionState {
        LOCAL_HOST(200),
        ADDRESS(50),
        FAST_CONNECT(50),
        SLOW_CONNECT(1000),
        HEURISTIC(0);

        private final int timeout;

        private AddressDetectionState(int timeout) {
            this.timeout = timeout;
        }

        public int getTimeout() {
            return this.timeout;
        }
    }
}

