/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.filecache;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.cache.DistributedCache;
import org.apache.flink.api.java.tuple.Tuple4;
import org.apache.flink.configuration.ConfigConstants;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.util.ExecutorThreadFactory;
import org.apache.flink.runtime.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCache {
    static final Logger LOG = LoggerFactory.getLogger(FileCache.class);
    private final Object lock = new Object();
    private final Map<JobID, Map<String, Tuple4<Integer, File, Path, Future<Path>>>> entries;
    private final ScheduledExecutorService executorService;
    private final File[] storageDirectories;
    private final Thread shutdownHook;
    private int nextDirectory;

    public FileCache(Configuration config) throws IOException {
        String tempDirs = config.getString("taskmanager.tmp.dirs", ConfigConstants.DEFAULT_TASK_MANAGER_TMP_PATH);
        String[] directories = tempDirs.split(",|" + File.pathSeparator);
        String cacheDirName = "flink-dist-cache-" + UUID.randomUUID().toString();
        this.storageDirectories = new File[directories.length];
        for (int i = 0; i < directories.length; ++i) {
            this.storageDirectories[i] = new File(directories[i], cacheDirName);
            String path = this.storageDirectories[i].getAbsolutePath();
            if (!this.storageDirectories[i].mkdirs()) {
                LOG.error("User file cache cannot create directory " + path);
                for (int k = 0; k < i; ++k) {
                    if (this.storageDirectories[k].delete()) continue;
                    LOG.warn("User file cache cannot remove prior directory " + this.storageDirectories[k].getAbsolutePath());
                }
                throw new IOException("File cache cannot create temp storage directory: " + path);
            }
            LOG.info("User file cache uses directory " + path);
        }
        this.shutdownHook = FileCache.createShutdownHook(this, LOG);
        this.entries = new HashMap<JobID, Map<String, Tuple4<Integer, File, Path, Future<Path>>>>();
        this.executorService = Executors.newScheduledThreadPool(10, ExecutorThreadFactory.INSTANCE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.lock;
        synchronized (object) {
            ScheduledExecutorService es = this.executorService;
            if (es != null) {
                es.shutdown();
                try {
                    es.awaitTermination(5000L, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            this.entries.clear();
            for (File dir : this.storageDirectories) {
                try {
                    FileUtils.deleteDirectory((File)dir);
                }
                catch (IOException e) {
                    LOG.error("File cache could not properly clean up storage directory.");
                }
            }
            if (this.shutdownHook != null && this.shutdownHook != Thread.currentThread()) {
                try {
                    Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
                }
                catch (IllegalStateException illegalStateException) {
                }
                catch (Throwable t) {
                    LOG.warn("Exception while unregistering file cache's cleanup shutdown hook.");
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Future<Path> createTmpFile(String name, DistributedCache.DistributedCacheEntry entry, JobID jobID) {
        Object object = this.lock;
        synchronized (object) {
            String sourceFile;
            int posOfSep;
            Tuple4<Integer, File, Path, Future<Path>> fileEntry;
            Map<String, Tuple4<Integer, File, Path, Future<Path>>> jobEntries = this.entries.get(jobID);
            if (jobEntries == null) {
                jobEntries = new HashMap<String, Tuple4<Integer, File, Path, Future<Path>>>();
                this.entries.put(jobID, jobEntries);
            }
            if ((fileEntry = jobEntries.get(name)) != null) {
                fileEntry.f0 = (Integer)fileEntry.f0 + 1;
                return (Future)fileEntry.f3;
            }
            File tempDirToUse = new File(this.storageDirectories[this.nextDirectory++], jobID.toString());
            if (this.nextDirectory >= this.storageDirectories.length) {
                this.nextDirectory = 0;
            }
            if ((posOfSep = (sourceFile = entry.filePath).lastIndexOf("/")) > 0) {
                sourceFile = sourceFile.substring(posOfSep + 1);
            }
            Path target = new Path(tempDirToUse.getAbsolutePath() + "/" + sourceFile);
            CopyProcess cp = new CopyProcess(entry, target);
            FutureTask<Path> copyTask = new FutureTask<Path>(cp);
            this.executorService.submit(copyTask);
            jobEntries.put(name, (Tuple4<Integer, File, Path, Future<Path>>)new Tuple4((Object)1, (Object)tempDirToUse, (Object)target, copyTask));
            return copyTask;
        }
    }

    public void deleteTmpFile(String name, JobID jobID) {
        DeleteProcess dp = new DeleteProcess(this.lock, this.entries, name, jobID);
        this.executorService.schedule(dp, 5000L, TimeUnit.MILLISECONDS);
    }

    boolean holdsStillReference(String name, JobID jobId) {
        Map<String, Tuple4<Integer, File, Path, Future<Path>>> jobEntries = this.entries.get(jobId);
        if (jobEntries != null) {
            Tuple4<Integer, File, Path, Future<Path>> entry = jobEntries.get(name);
            return entry != null && (Integer)entry.f0 > 0;
        }
        return false;
    }

    public static void copy(Path sourcePath, Path targetPath, boolean executable) throws IOException {
        block6: {
            FileSystem sFS = sourcePath.getFileSystem();
            FileSystem tFS = targetPath.getFileSystem();
            if (tFS.exists(targetPath)) break block6;
            if (sFS.getFileStatus(sourcePath).isDir()) {
                FileStatus[] contents;
                tFS.mkdirs(targetPath);
                for (FileStatus content : contents = sFS.listStatus(sourcePath)) {
                    String distPath = content.getPath().toString();
                    if (content.isDir() && distPath.endsWith("/")) {
                        distPath = distPath.substring(0, distPath.length() - 1);
                    }
                    String localPath = targetPath.toString() + distPath.substring(distPath.lastIndexOf("/"));
                    FileCache.copy(content.getPath(), new Path(localPath), executable);
                }
            } else {
                try {
                    FSDataOutputStream lfsOutput = tFS.create(targetPath, false);
                    FSDataInputStream fsInput = sFS.open(sourcePath);
                    IOUtils.copyBytes((InputStream)fsInput, (OutputStream)lfsOutput);
                    new File(targetPath.toString()).setExecutable(executable);
                }
                catch (IOException ioe) {
                    LOG.error("could not copy file to local file cache.", ioe);
                }
            }
        }
    }

    private static Thread createShutdownHook(final FileCache cache, final Logger logger) {
        Thread shutdownHook = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    cache.shutdown();
                }
                catch (Throwable t) {
                    logger.error("Error during shutdown of file cache via JVM shutdown hook: " + t.getMessage(), t);
                }
            }
        });
        try {
            Runtime.getRuntime().addShutdownHook(shutdownHook);
            return shutdownHook;
        }
        catch (IllegalStateException e) {
            return null;
        }
        catch (Throwable t) {
            logger.error("Cannot register shutdown hook that cleanly terminates the file cache service.");
            return null;
        }
    }

    private static class DeleteProcess
    implements Runnable {
        private final Object lock;
        private final Map<JobID, Map<String, Tuple4<Integer, File, Path, Future<Path>>>> entries;
        private final String name;
        private final JobID jobID;

        public DeleteProcess(Object lock, Map<JobID, Map<String, Tuple4<Integer, File, Path, Future<Path>>>> entries, String name, JobID jobID) {
            this.lock = lock;
            this.entries = entries;
            this.name = name;
            this.jobID = jobID;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                Object object = this.lock;
                synchronized (object) {
                    Tuple4<Integer, File, Path, Future<Path>> entry;
                    Map<String, Tuple4<Integer, File, Path, Future<Path>>> jobEntries = this.entries.get(this.jobID);
                    if (jobEntries != null && (entry = jobEntries.get(this.name)) != null) {
                        int count = (Integer)entry.f0;
                        if (count > 1) {
                            entry.f0 = count - 1;
                        } else {
                            String[] children;
                            File parent;
                            jobEntries.remove(this.name);
                            if (jobEntries.isEmpty()) {
                                this.entries.remove(this.jobID);
                            }
                            ((Future)entry.f3).cancel(true);
                            File file = new File(((Path)entry.f2).toString());
                            if (file.exists()) {
                                if (file.isDirectory()) {
                                    FileUtils.deleteDirectory((File)file);
                                } else if (!file.delete()) {
                                    LOG.error("Could not delete locally cached file " + file.getAbsolutePath());
                                }
                            }
                            if ((parent = (File)entry.f1).isDirectory() && ((children = parent.list()) == null || children.length == 0)) {
                                parent.delete();
                            }
                        }
                    }
                }
            }
            catch (IOException e) {
                LOG.error("Could not delete file from local file cache.", e);
            }
        }
    }

    private static class CopyProcess
    implements Callable<Path> {
        private final Path filePath;
        private final Path cachedPath;
        private boolean executable;

        public CopyProcess(DistributedCache.DistributedCacheEntry e, Path cachedPath) {
            this.filePath = new Path(e.filePath);
            this.executable = e.isExecutable;
            this.cachedPath = cachedPath;
        }

        @Override
        public Path call() throws IOException {
            FileCache.copy(this.filePath, this.cachedPath, this.executable);
            return this.cachedPath;
        }
    }
}

